/*
 * kgraft_patch_bsc1050751
 *
 * Fix for CVE-2017-7533, bsc#1050751
 *
 *  Upstream commit:
 *  49d31c2f389acfe83417083e1208422b4091cd9 ("dentry name snapshots")
 *
 *  SLE12-SP2 commit:
 *  c0c716e074137426d4f89426e8c9544dc7b535e9
 *
 *  SLE12(-SP1) commit:
 *  1c082b6e8178377776644005e231e9da2a88b52b
 *
 *  Copyright (c) 2017 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/kallsyms.h>
#include <linux/dcache.h>
#include <linux/spinlock.h>
#include <linux/slab.h>
#include <linux/fsnotify.h>

#if !IS_ENABLED(CONFIG_FSNOTIFY)
#error "KGR patch supports only CONFIG_FSNOTIFY=y."
#endif

static void (*kgr__fsnotify_update_child_dentry_flags)(struct inode *inode);

static struct {
	char *name;
	void **addr;
} kgr_funcs[] = {
	{ "__fsnotify_update_child_dentry_flags",
		(void *)&kgr__fsnotify_update_child_dentry_flags },
};

/* added to fs/notify/fsnotify.c, needed for fix */
struct name_snapshot {
	const char *name;
	char inline_name[DNAME_INLINE_LEN];
};

/*
 * Copy dentry name without holding i_mutex on the directory. We have to be
 * very careful as the name can be changing under us until we grab d_lock.
 */
static void kgr_take_dentry_name_snapshot(struct name_snapshot *name,
					struct dentry *dentry)
{
	char *namebuf = NULL;
	unsigned int len;

restart:
	/* Opportunistic check and buffer allocation without lock... */
	if (dname_external(dentry)) {
		len = ACCESS_ONCE(dentry->d_name.len);
		namebuf = kmalloc(len + 1, GFP_KERNEL);
	}
	spin_lock(&dentry->d_lock);
	/* Check reliably under the lock */
	if (dname_external(dentry)) {
		if (!namebuf || len < dentry->d_name.len) {
			spin_unlock(&dentry->d_lock);
			kfree(namebuf);
			namebuf = NULL;
			goto restart;
		}
		memcpy(namebuf, dentry->d_name.name, len + 1);
		name->name = namebuf;
		namebuf = NULL;
	} else {
		memcpy(name->inline_name, dentry->d_name.name,
			dentry->d_name.len + 1);
		name->name = name->inline_name;
	}
	spin_unlock(&dentry->d_lock);
	/* Free buffer if it wasn't used */
	kfree(namebuf);
}

static void kgr_release_dentry_name_snapshot(struct name_snapshot *name)
{
	if (unlikely(name->name != name->inline_name))
		kfree(name->name);
}

/* patched */
int kgr__fsnotify_parent(struct path *path, struct dentry *dentry, __u32 mask)
{
	struct dentry *parent;
	struct inode *p_inode;
	int ret = 0;

	if (!dentry)
		dentry = path->dentry;

	if (!(dentry->d_flags & DCACHE_FSNOTIFY_PARENT_WATCHED))
		return 0;

	parent = dget_parent(dentry);
	p_inode = parent->d_inode;

	if (unlikely(!fsnotify_inode_watches_children(p_inode)))
		kgr__fsnotify_update_child_dentry_flags(p_inode);
	else if (p_inode->i_fsnotify_mask & mask) {
		/*
		 * Fix CVE-2017-7533
		 *  +2 lines
		 */
		struct name_snapshot name;

		/* we are notifying a parent so come up with the new mask which
		 * specifies these are events which came from a child. */
		mask |= FS_EVENT_ON_CHILD;

		/*
		 * Fix CVE-2017-7533
		 *  +1 line
		 */
		kgr_take_dentry_name_snapshot(&name, dentry);
		if (path)
			ret = fsnotify(p_inode, mask, path, FSNOTIFY_EVENT_PATH,
				       /*
					* Fix CVE-2017-7533
					*  -1 line, +1 line
					*/
				       name.name, 0);
		else
			ret = fsnotify(p_inode, mask, dentry->d_inode, FSNOTIFY_EVENT_INODE,
				       /*
					* Fix CVE-2017-7533
					*  -1 line, +1 line
					*/
				       name.name, 0);
		/*
		 * Fix CVE-2017-7533
		 *  +1 line
		 */
		kgr_release_dentry_name_snapshot(&name);
	}

	dput(parent);

	return ret;
}

static int kgr_patch_bsc1050751_kallsyms(void)
{
	unsigned long addr;
	int i;

	for (i = 0; i < ARRAY_SIZE(kgr_funcs); i++) {
		/* mod_find_symname would be nice, but it is not exported */
		addr = kallsyms_lookup_name(kgr_funcs[i].name);
		if (!addr) {
			pr_err("kgraft-patch: symbol %s not resolved\n",
				kgr_funcs[i].name);
			return -ENOENT;
		}

		*(kgr_funcs[i].addr) = (void *)addr;
	}

	return 0;
}

int kgr_patch_bsc1050751_init(void)
{
	return kgr_patch_bsc1050751_kallsyms();
}
