/*
 * livepatch_bsc1251203
 *
 * Fix for CVE-2025-38476, bsc#1251203
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Ali Abdallah <ali.abdallah@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/ipv6/rpl_iptunnel.c */
#include <net/dst_cache.h>

/* klp-ccp: from net/ipv6/rpl_iptunnel.c */
#include <net/ip6_route.h>

/* klp-ccp: from include/net/ip6_route.h */
static void (*klpe_ip6_route_input)(struct sk_buff *skb);

/* klp-ccp: from net/ipv6/rpl_iptunnel.c */
#include <net/lwtunnel.h>
#include <net/ipv6.h>
#include <net/rpl.h>

/* klp-ccp: from include/net/rpl.h */
static void (*klpe_ipv6_rpl_srh_compress)(struct ipv6_rpl_sr_hdr *outhdr,
			   const struct ipv6_rpl_sr_hdr *inhdr,
			   const struct in6_addr *daddr, unsigned char n);

/* klp-ccp: from net/ipv6/rpl_iptunnel.c */
struct rpl_iptunnel_encap {
	struct ipv6_rpl_sr_hdr srh[0];
};

struct rpl_lwt {
	struct dst_cache cache;
	struct rpl_iptunnel_encap tuninfo;
};

static inline struct rpl_lwt *rpl_lwt_lwtunnel(struct lwtunnel_state *lwt)
{
	return (struct rpl_lwt *)lwt->data;
}

static inline struct rpl_iptunnel_encap *
rpl_encap_lwtunnel(struct lwtunnel_state *lwt)
{
	return &rpl_lwt_lwtunnel(lwt)->tuninfo;
}

static int klpr_rpl_do_srh_inline(struct sk_buff *skb, const struct rpl_lwt *rlwt,
			     const struct ipv6_rpl_sr_hdr *srh)
{
	struct ipv6_rpl_sr_hdr *isrh, *csrh;
	struct ipv6hdr oldhdr;
	struct ipv6hdr *hdr;
	unsigned char *buf;
	size_t hdrlen;
	int err;

	memcpy(&oldhdr, ipv6_hdr(skb), sizeof(oldhdr));

	buf = kcalloc(struct_size(srh, segments.addr, srh->segments_left), 2, GFP_ATOMIC);
	if (!buf)
		return -ENOMEM;

	isrh = (struct ipv6_rpl_sr_hdr *)buf;
	csrh = (struct ipv6_rpl_sr_hdr *)(buf + ((srh->hdrlen + 1) << 3));

	memcpy(isrh, srh, sizeof(*isrh));
	memcpy(isrh->rpl_segaddr, &srh->rpl_segaddr[1],
	       (srh->segments_left - 1) * 16);
	isrh->rpl_segaddr[srh->segments_left - 1] = oldhdr.daddr;

	(*klpe_ipv6_rpl_srh_compress)(csrh, isrh, &srh->rpl_segaddr[0],
			      isrh->segments_left - 1);

	hdrlen = ((csrh->hdrlen + 1) << 3);

	err = skb_cow_head(skb, hdrlen + skb->mac_len);
	if (unlikely(err)) {
		kfree(buf);
		return err;
	}

	skb_pull(skb, sizeof(struct ipv6hdr));
	skb_postpull_rcsum(skb, skb_network_header(skb),
			   sizeof(struct ipv6hdr));

	skb_push(skb, sizeof(struct ipv6hdr) + hdrlen);
	skb_reset_network_header(skb);
	skb_mac_header_rebuild(skb);

	hdr = ipv6_hdr(skb);
	memmove(hdr, &oldhdr, sizeof(*hdr));
	isrh = (void *)hdr + sizeof(*hdr);
	memcpy(isrh, csrh, hdrlen);

	isrh->nexthdr = hdr->nexthdr;
	hdr->nexthdr = NEXTHDR_ROUTING;
	hdr->daddr = srh->rpl_segaddr[0];

	ipv6_hdr(skb)->payload_len = htons(skb->len - sizeof(struct ipv6hdr));
	skb_set_transport_header(skb, sizeof(struct ipv6hdr));

	skb_postpush_rcsum(skb, hdr, sizeof(struct ipv6hdr) + hdrlen);

	kfree(buf);

	return 0;
}

static int klpr_rpl_do_srh(struct sk_buff *skb, const struct rpl_lwt *rlwt)
{
	struct dst_entry *dst = skb_dst(skb);
	struct rpl_iptunnel_encap *tinfo;

	if (skb->protocol != htons(ETH_P_IPV6))
		return -EINVAL;

	tinfo = rpl_encap_lwtunnel(dst->lwtstate);

	return klpr_rpl_do_srh_inline(skb, rlwt, tinfo->srh);
}

int klpp_rpl_output(struct net *net, struct sock *sk, struct sk_buff *skb)
{
	struct dst_entry *orig_dst = skb_dst(skb);
	struct dst_entry *dst = NULL;
	struct rpl_lwt *rlwt;
	int err;

	rlwt = rpl_lwt_lwtunnel(orig_dst->lwtstate);

	err = klpr_rpl_do_srh(skb, rlwt);
	if (unlikely(err))
		goto drop;

	preempt_disable();
	dst = dst_cache_get(&rlwt->cache);
	preempt_enable();

	if (unlikely(!dst)) {
		struct ipv6hdr *hdr = ipv6_hdr(skb);
		struct flowi6 fl6;

		memset(&fl6, 0, sizeof(fl6));
		fl6.daddr = hdr->daddr;
		fl6.saddr = hdr->saddr;
		fl6.flowlabel = ip6_flowinfo(hdr);
		fl6.flowi6_mark = skb->mark;
		fl6.flowi6_proto = hdr->nexthdr;

		dst = ip6_route_output(net, NULL, &fl6);
		if (dst->error) {
			err = dst->error;
			dst_release(dst);
			goto drop;
		}

		preempt_disable();
		dst_cache_set_ip6(&rlwt->cache, dst, &fl6.saddr);
		preempt_enable();
	}

	skb_dst_drop(skb);
	skb_dst_set(skb, dst);

	err = skb_cow_head(skb, LL_RESERVED_SPACE(dst->dev));
	if (unlikely(err))
		goto drop;

	return dst_output(net, sk, skb);

drop:
	kfree_skb(skb);
	return err;
}

int klpp_rpl_input(struct sk_buff *skb)
{
	struct dst_entry *orig_dst = skb_dst(skb);
	struct dst_entry *dst = NULL;
	struct rpl_lwt *rlwt;
	int err;

	rlwt = rpl_lwt_lwtunnel(orig_dst->lwtstate);

	err = klpr_rpl_do_srh(skb, rlwt);
	if (unlikely(err)) {
		kfree_skb(skb);
		return err;
	}

	preempt_disable();
	dst = dst_cache_get(&rlwt->cache);
	preempt_enable();

	skb_dst_drop(skb);

	if (!dst) {
		(*klpe_ip6_route_input)(skb);
		dst = skb_dst(skb);
		if (!dst->error) {
			preempt_disable();
			dst_cache_set_ip6(&rlwt->cache, dst,
					  &ipv6_hdr(skb)->saddr);
			preempt_enable();
		}
	} else {
		skb_dst_set(skb, dst);
	}

	err = skb_cow_head(skb, LL_RESERVED_SPACE(dst->dev));
	if (unlikely(err))
		return err;

	return dst_input(skb);
}


#include "livepatch_bsc1251203.h"

#include <linux/kernel.h>
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "ip6_route_input", (void *)&klpe_ip6_route_input },
	{ "ipv6_rpl_srh_compress", (void *)&klpe_ipv6_rpl_srh_compress },
};

int livepatch_bsc1251203_init(void)
{
	return klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

