/*
 * livepatch_bsc1246189
 *
 * Fix for CVE-2025-38257, bsc#1246189
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_PKEY)

#if !IS_MODULE(CONFIG_PKEY)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from drivers/s390/crypto/pkey_api.c */
#define KMSG_COMPONENT "pkey"
#define pr_fmt(fmt) KMSG_COMPONENT ": " fmt

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/kallsyms.h>
#include <linux/debugfs.h>
#include <linux/random.h>
#include <linux/cpufeature.h>
#include <asm/zcrypt.h>
#include <asm/cpacf.h>
#include <asm/pkey.h>

/* klp-ccp: from arch/s390/include/asm/pkey.h */
static int (*klpe_pkey_keyblob2pkey)(const u8 *key, u32 keylen,
		      struct pkey_protkey *protkey);

/* klp-ccp: from drivers/s390/crypto/pkey_api.c */
#include <crypto/aes.h>
/* klp-ccp: from drivers/s390/crypto/zcrypt_api.h */
#include <linux/atomic.h>
#include <asm/debug.h>
#include <asm/zcrypt.h>
/* klp-ccp: from drivers/s390/crypto/ap_bus.h */
#include <linux/device.h>
#include <linux/types.h>
#include <linux/hashtable.h>
#include <asm/isc.h>
#include <asm/ap.h>

/* klp-ccp: from drivers/s390/crypto/zcrypt_api.h */
#define ZCRYPT_CEX3C		7

#define ZCRYPT_CEX6	       12
#define ZCRYPT_CEX7	       13

static int (*klpe_zcrypt_wait_api_operational)(void);

/* klp-ccp: from drivers/s390/crypto/zcrypt_ccamisc.h */
#include <asm/zcrypt.h>
#include <asm/pkey.h>

#define TOKTYPE_NON_CCA		 0x00 /* Non-CCA key token */
#define TOKTYPE_CCA_INTERNAL	 0x01 /* CCA internal sym key token */
#define TOKTYPE_CCA_INTERNAL_PKA 0x1f /* CCA internal asym key token */

#define TOKVER_CCA_AES		0x04 /* CCA AES key token */
#define TOKVER_CCA_VLSC		0x05 /* var length sym cipher key token */

#define MAXCCAVLSCTOKENSIZE 725

struct keytoken_header {
	u8  type;     /* one of the TOKTYPE values */
	u8  res0[1];
	u16 len;      /* vlsc token: total length in bytes */
	u8  version;  /* one of the TOKVER values */
	u8  res1[3];
} __packed;

struct secaeskeytoken {
	u8  type;     /* 0x01 for internal key token */
	u8  res0[3];
	u8  version;  /* should be 0x04 */
	u8  res1[1];
	u8  flag;     /* key flags */
	u8  res2[1];
	u64 mkvp;     /* master key verification pattern */
	u8  key[32];  /* key value (encrypted) */
	u8  cv[8];    /* control vector */
	u16 bitsize;  /* key bit size */
	u16 keysize;  /* key byte size */
	u8  tvv[4];   /* token validation value */
} __packed;

struct cipherkeytoken {
	u8  type;     /* 0x01 for internal key token */
	u8  res0[1];
	u16 len;      /* total key token length in bytes */
	u8  version;  /* should be 0x05 */
	u8  res1[3];
	u8  kms;      /* key material state, 0x03 means wrapped with MK */
	u8  kvpt;     /* key verification pattern type, should be 0x01 */
	u64 mkvp0;    /* master key verification pattern, lo part */
	u64 mkvp1;    /* master key verification pattern, hi part (unused) */
	u8  eskwm;    /* encrypted section key wrapping method */
	u8  hashalg;  /* hash algorithmus used for wrapping key */
	u8  plfver;   /* pay load format version */
	u8  res2[1];
	u8  adsver;   /* associated data section version */
	u8  res3[1];
	u16 adslen;   /* associated data section length */
	u8  kllen;    /* optional key label length */
	u8  ieaslen;  /* optional extended associated data length */
	u8  uadlen;   /* optional user definable associated data length */
	u8  res4[1];
	u16 wpllen;   /* wrapped payload length in bits: */
		      /*   plfver  0x00 0x01		 */
		      /*   AES-128  512  640		 */
		      /*   AES-192  576  640		 */
		      /*   AES-256  640  640		 */
	u8  res5[1];
	u8  algtype;  /* 0x02 for AES cipher */
	u16 keytype;  /* 0x0001 for 'cipher' */
	u8  kufc;     /* key usage field count */
	u16 kuf1;     /* key usage field 1 */
	u16 kuf2;     /* key usage field 2 */
	u8  kmfc;     /* key management field count */
	u16 kmf1;     /* key management field 1 */
	u16 kmf2;     /* key management field 2 */
	u16 kmf3;     /* key management field 3 */
	u8  vdata[]; /* variable part data follows */
} __packed;

struct eccprivkeytoken {
	u8  type;     /* 0x1f for internal asym key token */
	u8  version;  /* should be 0x00 */
	u16 len;      /* total key token length in bytes */
	u8  res1[4];
	u8  secid;    /* 0x20 for ECC priv key section marker */
	u8  secver;   /* section version */
	u16 seclen;   /* section length */
	u8  wtype;    /* wrapping method, 0x00 clear, 0x01 AES */
	u8  htype;    /* hash method, 0x02 for SHA-256 */
	u8  res2[2];
	u8  kutc;     /* key usage and translation control */
	u8  ctype;    /* curve type */
	u8  kfs;      /* key format and security */
	u8  ksrc;     /* key source */
	u16 pbitlen;  /* length of prime p in bits */
	u16 ibmadlen; /* IBM associated data length in bytes */
	u64 mkvp;     /* master key verification pattern */
	u8  opk[48];  /* encrypted object protection key data */
	u16 adatalen; /* associated data length in bytes */
	u16 fseclen;  /* formated section length in bytes */
	u8  more_data[]; /* more data follows */
} __packed;

static int (*klpe_cca_check_secaeskeytoken)(debug_info_t *dbg, int dbflvl,
			     const u8 *token, int keybitsize);

static int (*klpe_cca_check_secaescipherkey)(debug_info_t *dbg, int dbflvl,
			      const u8 *token, int keybitsize,
			      int checkcpacfexport);

static int (*klpe_cca_check_sececckeytoken)(debug_info_t *dbg, int dbflvl,
			     const u8 *token, size_t keysize,
			     int checkcpacfexport);

static int (*klpe_cca_genseckey)(u16 cardnr, u16 domain, u32 keybitsize, u8 *seckey);

static int (*klpe_cca_clr2seckey)(u16 cardnr, u16 domain, u32 keybitsize,
		   const u8 *clrkey, u8 *seckey);

static int (*klpe_cca_sec2protkey)(u16 cardnr, u16 domain,
		    const u8 *seckey, u8 *protkey, u32 *protkeylen,
		    u32 *protkeytype);

static int (*klpe_cca_gencipherkey)(u16 cardnr, u16 domain, u32 keybitsize, u32 keygenflags,
		     u8 *keybuf, size_t *keybufsize);

static int (*klpe_cca_cipher2protkey)(u16 cardnr, u16 domain, const u8 *ckey,
		       u8 *protkey, u32 *protkeylen, u32 *protkeytype);

static int (*klpe_cca_clr2cipherkey)(u16 cardnr, u16 domain, u32 keybitsize, u32 keygenflags,
		      const u8 *clrkey, u8 *keybuf, size_t *keybufsize);

static int (*klpe_cca_ecc2protkey)(u16 cardnr, u16 domain, const u8 *key,
		    u8 *protkey, u32 *protkeylen, u32 *protkeytype);

static int (*klpe_cca_findcard)(const u8 *key, u16 *pcardnr, u16 *pdomain, int verify);

static int (*klpe_cca_findcard2)(u32 **apqns, u32 *nr_apqns, u16 cardnr, u16 domain,
		  int minhwtype, int mktype, u64 cur_mkvp, u64 old_mkvp,
		  int verify);

#define AES_MK_SET  0
#define APKA_MK_SET 1

/* klp-ccp: from drivers/s390/crypto/zcrypt_ep11misc.h */
#include <asm/zcrypt.h>
#include <asm/pkey.h>

#define EP11_API_V 4  /* highest known and supported EP11 API version */
#define EP11_STRUCT_MAGIC 0x1234
#define EP11_BLOB_PKEY_EXTRACTABLE 0x00200000

#define TOKVER_EP11_AES  0x03  /* EP11 AES key blob (old style) */
#define TOKVER_EP11_AES_WITH_HEADER 0x06 /* EP11 AES key blob with header */
#define TOKVER_EP11_ECC_WITH_HEADER 0x07 /* EP11 ECC key blob with header */

struct ep11keyblob {
	union {
		u8 session[32];
		/* only used for PKEY_TYPE_EP11: */
		struct ep11kblob_header head;
	};
	u8  wkvp[16];  /* wrapping key verification pattern */
	u64 attr;      /* boolean key attributes */
	u64 mode;      /* mode bits */
	u16 version;   /* 0x1234, EP11_STRUCT_MAGIC */
	u8  iv[14];
	u8  encrypted_key_data[144];
	u8  mac[32];
} __packed;

static inline bool is_ep11_keyblob(const u8 *key)
{
	struct ep11keyblob *kb = (struct ep11keyblob *) key;

	return (kb->version == EP11_STRUCT_MAGIC);
}

static int (*klpe_ep11_check_aes_key_with_hdr)(debug_info_t *dbg, int dbflvl,
				const u8 *key, size_t keylen, int checkcpacfexp);

static int (*klpe_ep11_check_ecc_key_with_hdr)(debug_info_t *dbg, int dbflvl,
				const u8 *key, size_t keylen, int checkcpacfexp);

static int (*klpe_ep11_check_aes_key)(debug_info_t *dbg, int dbflvl,
		       const u8 *key, size_t keylen, int checkcpacfexp);

static int (*klpe_ep11_genaeskey)(u16 card, u16 domain, u32 keybitsize, u32 keygenflags,
		   u8 *keybuf, size_t *keybufsize);

static int (*klpe_ep11_clr2keyblob)(u16 cardnr, u16 domain, u32 keybitsize, u32 keygenflags,
		     const u8 *clrkey, u8 *keybuf, size_t *keybufsize);

static int (*klpe_ep11_findcard2)(u32 **apqns, u32 *nr_apqns, u16 cardnr, u16 domain,
		   int minhwtype, int minapi, const u8 *wkvp);

static int (*klpe_ep11_kblob2protkey)(u16 card, u16 dom, const u8 *key, size_t keylen,
		       u8 *protkey, u32 *protkeylen, u32 *protkeytype);

/* klp-ccp: from drivers/s390/crypto/pkey_api.c */
#define KEYBLOBBUFSIZE 8192	/* key buffer size used for internal processing */
#define PROTKEYBLOBBUFSIZE 256	/* protected key buffer size used internal */

static debug_info_t *(*klpe_debug_info);

static int (*klpe_pkey_clr2protkey)(u32 keytype,
			    const struct pkey_clrkey *clrkey,
			    struct pkey_protkey *protkey);

static int (*klpe_pkey_skey2pkey)(const u8 *key, struct pkey_protkey *pkey);

#define KLPR_DEBUG_DBG(...)  debug_sprintf_event((*klpe_debug_info), 6, ##__VA_ARGS__)
#define KLPR_DEBUG_ERR(...)  debug_sprintf_event((*klpe_debug_info), 3, ##__VA_ARGS__)

static int klpr_pkey_verifykey(const struct pkey_seckey *seckey,
			  u16 *pcardnr, u16 *pdomain,
			  u16 *pkeysize, u32 *pattributes)
{
	struct secaeskeytoken *t = (struct secaeskeytoken *) seckey;
	u16 cardnr, domain;
	int rc;

	/* check the secure key for valid AES secure key */
	rc = (*klpe_cca_check_secaeskeytoken)((*klpe_debug_info), 3, (u8 *) seckey, 0);
	if (rc)
		goto out;
	if (pattributes)
		*pattributes = PKEY_VERIFY_ATTR_AES;
	if (pkeysize)
		*pkeysize = t->bitsize;

	/* try to find a card which can handle this key */
	rc = (*klpe_cca_findcard)(seckey->seckey, &cardnr, &domain, 1);
	if (rc < 0)
		goto out;

	if (rc > 0) {
		/* key mkvp matches to old master key mkvp */
		KLPR_DEBUG_DBG("%s secure key has old mkvp\n", __func__);
		if (pattributes)
			*pattributes |= PKEY_VERIFY_ATTR_OLD_MKVP;
		rc = 0;
	}

	if (pcardnr)
		*pcardnr = cardnr;
	if (pdomain)
		*pdomain = domain;

out:
	KLPR_DEBUG_DBG("%s rc=%d\n", __func__, rc);
	return rc;
}

static int (*klpe_pkey_genprotkey)(u32 keytype, struct pkey_protkey *protkey);

static int (*klpe_pkey_verifyprotkey)(const struct pkey_protkey *protkey);

static int (*klpe_pkey_nonccatok2pkey)(const u8 *key, u32 keylen,
			       struct pkey_protkey *protkey);

static int klpr_pkey_genseckey2(const struct pkey_apqn *apqns, size_t nr_apqns,
			   enum pkey_key_type ktype, enum pkey_key_size ksize,
			   u32 kflags, u8 *keybuf, size_t *keybufsize)
{
	int i, card, dom, rc;

	/* check for at least one apqn given */
	if (!apqns || !nr_apqns)
		return -EINVAL;

	/* check key type and size */
	switch (ktype) {
	case PKEY_TYPE_CCA_DATA:
	case PKEY_TYPE_CCA_CIPHER:
		if (*keybufsize < SECKEYBLOBSIZE)
			return -EINVAL;
		break;
	case PKEY_TYPE_EP11:
		if (*keybufsize < MINEP11AESKEYBLOBSIZE)
			return -EINVAL;
		break;
	default:
		return -EINVAL;
	}
	switch (ksize) {
	case PKEY_SIZE_AES_128:
	case PKEY_SIZE_AES_192:
	case PKEY_SIZE_AES_256:
		break;
	default:
		return -EINVAL;
	}

	/* simple try all apqns from the list */
	for (i = 0, rc = -ENODEV; i < nr_apqns; i++) {
		card = apqns[i].card;
		dom = apqns[i].domain;
		if (ktype == PKEY_TYPE_EP11) {
			rc = (*klpe_ep11_genaeskey)(card, dom, ksize, kflags,
					    keybuf, keybufsize);
		} else if (ktype == PKEY_TYPE_CCA_DATA) {
			rc = (*klpe_cca_genseckey)(card, dom, ksize, keybuf);
			*keybufsize = (rc ? 0 : SECKEYBLOBSIZE);
		} else /* TOKVER_CCA_VLSC */
			rc = (*klpe_cca_gencipherkey)(card, dom, ksize, kflags,
					      keybuf, keybufsize);
		if (rc == 0)
			break;
	}

	return rc;
}

static int klpr_pkey_clr2seckey2(const struct pkey_apqn *apqns, size_t nr_apqns,
			    enum pkey_key_type ktype, enum pkey_key_size ksize,
			    u32 kflags, const u8 *clrkey,
			    u8 *keybuf, size_t *keybufsize)
{
	int i, card, dom, rc;

	/* check for at least one apqn given */
	if (!apqns || !nr_apqns)
		return -EINVAL;

	/* check key type and size */
	switch (ktype) {
	case PKEY_TYPE_CCA_DATA:
	case PKEY_TYPE_CCA_CIPHER:
		if (*keybufsize < SECKEYBLOBSIZE)
			return -EINVAL;
		break;
	case PKEY_TYPE_EP11:
		if (*keybufsize < MINEP11AESKEYBLOBSIZE)
			return -EINVAL;
		break;
	default:
		return -EINVAL;
	}
	switch (ksize) {
	case PKEY_SIZE_AES_128:
	case PKEY_SIZE_AES_192:
	case PKEY_SIZE_AES_256:
		break;
	default:
		return -EINVAL;
	}

	(*klpe_zcrypt_wait_api_operational)();

	/* simple try all apqns from the list */
	for (i = 0, rc = -ENODEV; i < nr_apqns; i++) {
		card = apqns[i].card;
		dom = apqns[i].domain;
		if (ktype == PKEY_TYPE_EP11) {
			rc = (*klpe_ep11_clr2keyblob)(card, dom, ksize, kflags,
					      clrkey, keybuf, keybufsize);
		} else if (ktype == PKEY_TYPE_CCA_DATA) {
			rc = (*klpe_cca_clr2seckey)(card, dom, ksize,
					    clrkey, keybuf);
			*keybufsize = (rc ? 0 : SECKEYBLOBSIZE);
		} else /* TOKVER_CCA_VLSC */
			rc = (*klpe_cca_clr2cipherkey)(card, dom, ksize, kflags,
					       clrkey, keybuf, keybufsize);
		if (rc == 0)
			break;
	}

	return rc;
}

static int klpr_pkey_verifykey2(const u8 *key, size_t keylen,
			   u16 *cardnr, u16 *domain,
			   enum pkey_key_type *ktype,
			   enum pkey_key_size *ksize, u32 *flags)
{
	int rc;
	u32 _nr_apqns, *_apqns = NULL;
	struct keytoken_header *hdr = (struct keytoken_header *)key;

	if (keylen < sizeof(struct keytoken_header))
		return -EINVAL;

	if (hdr->type == TOKTYPE_CCA_INTERNAL
	    && hdr->version == TOKVER_CCA_AES) {
		struct secaeskeytoken *t = (struct secaeskeytoken *)key;

		rc = (*klpe_cca_check_secaeskeytoken)((*klpe_debug_info), 3, key, 0);
		if (rc)
			goto out;
		if (ktype)
			*ktype = PKEY_TYPE_CCA_DATA;
		if (ksize)
			*ksize = (enum pkey_key_size) t->bitsize;

		rc = (*klpe_cca_findcard2)(&_apqns, &_nr_apqns, *cardnr, *domain,
				   ZCRYPT_CEX3C, AES_MK_SET, t->mkvp, 0, 1);
		if (rc == 0 && flags)
			*flags = PKEY_FLAGS_MATCH_CUR_MKVP;
		if (rc == -ENODEV) {
			rc = (*klpe_cca_findcard2)(&_apqns, &_nr_apqns,
					   *cardnr, *domain,
					   ZCRYPT_CEX3C, AES_MK_SET,
					   0, t->mkvp, 1);
			if (rc == 0 && flags)
				*flags = PKEY_FLAGS_MATCH_ALT_MKVP;
		}
		if (rc)
			goto out;

		*cardnr = ((struct pkey_apqn *)_apqns)->card;
		*domain = ((struct pkey_apqn *)_apqns)->domain;

	} else if (hdr->type == TOKTYPE_CCA_INTERNAL
		   && hdr->version == TOKVER_CCA_VLSC) {
		struct cipherkeytoken *t = (struct cipherkeytoken *)key;

		rc = (*klpe_cca_check_secaescipherkey)((*klpe_debug_info), 3, key, 0, 1);
		if (rc)
			goto out;
		if (ktype)
			*ktype = PKEY_TYPE_CCA_CIPHER;
		if (ksize) {
			*ksize = PKEY_SIZE_UNKNOWN;
			if (!t->plfver && t->wpllen == 512)
				*ksize = PKEY_SIZE_AES_128;
			else if (!t->plfver && t->wpllen == 576)
				*ksize = PKEY_SIZE_AES_192;
			else if (!t->plfver && t->wpllen == 640)
				*ksize = PKEY_SIZE_AES_256;
		}

		rc = (*klpe_cca_findcard2)(&_apqns, &_nr_apqns, *cardnr, *domain,
				   ZCRYPT_CEX6, AES_MK_SET, t->mkvp0, 0, 1);
		if (rc == 0 && flags)
			*flags = PKEY_FLAGS_MATCH_CUR_MKVP;
		if (rc == -ENODEV) {
			rc = (*klpe_cca_findcard2)(&_apqns, &_nr_apqns,
					   *cardnr, *domain,
					   ZCRYPT_CEX6, AES_MK_SET,
					   0, t->mkvp0, 1);
			if (rc == 0 && flags)
				*flags = PKEY_FLAGS_MATCH_ALT_MKVP;
		}
		if (rc)
			goto out;

		*cardnr = ((struct pkey_apqn *)_apqns)->card;
		*domain = ((struct pkey_apqn *)_apqns)->domain;

	} else if (hdr->type == TOKTYPE_NON_CCA
		   && hdr->version == TOKVER_EP11_AES) {
		struct ep11keyblob *kb = (struct ep11keyblob *)key;

		rc = (*klpe_ep11_check_aes_key)((*klpe_debug_info), 3, key, keylen, 1);
		if (rc)
			goto out;
		if (ktype)
			*ktype = PKEY_TYPE_EP11;
		if (ksize)
			*ksize = kb->head.bitlen;

		rc = (*klpe_ep11_findcard2)(&_apqns, &_nr_apqns, *cardnr, *domain,
				    ZCRYPT_CEX7, EP11_API_V, kb->wkvp);
		if (rc)
			goto out;

		if (flags)
			*flags = PKEY_FLAGS_MATCH_CUR_MKVP;

		*cardnr = ((struct pkey_apqn *)_apqns)->card;
		*domain = ((struct pkey_apqn *)_apqns)->domain;

	} else
		rc = -EINVAL;

out:
	kfree(_apqns);
	return rc;
}

static int klpr_pkey_keyblob2pkey2(const struct pkey_apqn *apqns, size_t nr_apqns,
			      const u8 *key, size_t keylen,
			      struct pkey_protkey *pkey)
{
	int i, card, dom, rc;
	struct keytoken_header *hdr = (struct keytoken_header *)key;

	/* check for at least one apqn given */
	if (!apqns || !nr_apqns)
		return -EINVAL;

	if (keylen < sizeof(struct keytoken_header))
		return -EINVAL;

	if (hdr->type == TOKTYPE_CCA_INTERNAL) {
		if (hdr->version == TOKVER_CCA_AES) {
			if (keylen != sizeof(struct secaeskeytoken))
				return -EINVAL;
			if ((*klpe_cca_check_secaeskeytoken)((*klpe_debug_info), 3, key, 0))
				return -EINVAL;
		} else if (hdr->version == TOKVER_CCA_VLSC) {
			if (keylen < hdr->len || keylen > MAXCCAVLSCTOKENSIZE)
				return -EINVAL;
			if ((*klpe_cca_check_secaescipherkey)((*klpe_debug_info), 3, key, 0, 1))
				return -EINVAL;
		} else {
			KLPR_DEBUG_ERR("%s unknown CCA internal token version %d\n",
				  __func__, hdr->version);
			return -EINVAL;
		}
	} else if (hdr->type == TOKTYPE_NON_CCA) {
		if (hdr->version == TOKVER_EP11_AES) {
			if (keylen < sizeof(struct ep11keyblob))
				return -EINVAL;
			if ((*klpe_ep11_check_aes_key)((*klpe_debug_info), 3, key, keylen, 1))
				return -EINVAL;
		} else {
			return (*klpe_pkey_nonccatok2pkey)(key, keylen, pkey);
		}
	} else {
		KLPR_DEBUG_ERR("%s unknown/unsupported blob type %d\n",
			  __func__, hdr->type);
		return -EINVAL;
	}

	(*klpe_zcrypt_wait_api_operational)();

	/* simple try all apqns from the list */
	for (i = 0, rc = -ENODEV; i < nr_apqns; i++) {
		card = apqns[i].card;
		dom = apqns[i].domain;
		if (hdr->type == TOKTYPE_CCA_INTERNAL
		    && hdr->version == TOKVER_CCA_AES)
			rc = (*klpe_cca_sec2protkey)(card, dom, key, pkey->protkey,
					     &pkey->len, &pkey->type);
		else if (hdr->type == TOKTYPE_CCA_INTERNAL
			 && hdr->version == TOKVER_CCA_VLSC)
			rc = (*klpe_cca_cipher2protkey)(card, dom, key, pkey->protkey,
						&pkey->len, &pkey->type);
		else { /* EP11 AES secure key blob */
			struct ep11keyblob *kb = (struct ep11keyblob *) key;

			pkey->len = sizeof(pkey->protkey);
			rc = (*klpe_ep11_kblob2protkey)(card, dom, key, kb->head.len,
						pkey->protkey, &pkey->len,
						&pkey->type);
		}
		if (rc == 0)
			break;
	}

	return rc;
}

static int klpr_pkey_apqns4key(const u8 *key, size_t keylen, u32 flags,
			  struct pkey_apqn *apqns, size_t *nr_apqns)
{
	int rc;
	u32 _nr_apqns, *_apqns = NULL;
	struct keytoken_header *hdr = (struct keytoken_header *)key;

	if (keylen < sizeof(struct keytoken_header) || flags == 0)
		return -EINVAL;

	(*klpe_zcrypt_wait_api_operational)();

	if (hdr->type == TOKTYPE_NON_CCA
	    && (hdr->version == TOKVER_EP11_AES_WITH_HEADER
		|| hdr->version == TOKVER_EP11_ECC_WITH_HEADER)
	    && is_ep11_keyblob(key + sizeof(struct ep11kblob_header))) {
		int minhwtype = 0, api = 0;
		struct ep11keyblob *kb = (struct ep11keyblob *)
			(key + sizeof(struct ep11kblob_header));

		if (flags != PKEY_FLAGS_MATCH_CUR_MKVP)
			return -EINVAL;
		if (kb->attr & EP11_BLOB_PKEY_EXTRACTABLE) {
			minhwtype = ZCRYPT_CEX7;
			api = EP11_API_V;
		}
		rc = (*klpe_ep11_findcard2)(&_apqns, &_nr_apqns, 0xFFFF, 0xFFFF,
				    minhwtype, api, kb->wkvp);
		if (rc)
			goto out;
	} else if (hdr->type == TOKTYPE_NON_CCA
		   && hdr->version == TOKVER_EP11_AES
		   && is_ep11_keyblob(key)) {
		int minhwtype = 0, api = 0;
		struct ep11keyblob *kb = (struct ep11keyblob *) key;

		if (flags != PKEY_FLAGS_MATCH_CUR_MKVP)
			return -EINVAL;
		if (kb->attr & EP11_BLOB_PKEY_EXTRACTABLE) {
			minhwtype = ZCRYPT_CEX7;
			api = EP11_API_V;
		}
		rc = (*klpe_ep11_findcard2)(&_apqns, &_nr_apqns, 0xFFFF, 0xFFFF,
				    minhwtype, api, kb->wkvp);
		if (rc)
			goto out;
	} else if (hdr->type == TOKTYPE_CCA_INTERNAL) {
		int minhwtype = ZCRYPT_CEX3C;
		u64 cur_mkvp = 0, old_mkvp = 0;

		if (hdr->version == TOKVER_CCA_AES) {
			struct secaeskeytoken *t = (struct secaeskeytoken *)key;

			if (flags & PKEY_FLAGS_MATCH_CUR_MKVP)
				cur_mkvp = t->mkvp;
			if (flags & PKEY_FLAGS_MATCH_ALT_MKVP)
				old_mkvp = t->mkvp;
		} else if (hdr->version == TOKVER_CCA_VLSC) {
			struct cipherkeytoken *t = (struct cipherkeytoken *)key;

			minhwtype = ZCRYPT_CEX6;
			if (flags & PKEY_FLAGS_MATCH_CUR_MKVP)
				cur_mkvp = t->mkvp0;
			if (flags & PKEY_FLAGS_MATCH_ALT_MKVP)
				old_mkvp = t->mkvp0;
		} else {
			/* unknown cca internal token type */
			return -EINVAL;
		}
		rc = (*klpe_cca_findcard2)(&_apqns, &_nr_apqns, 0xFFFF, 0xFFFF,
				   minhwtype, AES_MK_SET,
				   cur_mkvp, old_mkvp, 1);
		if (rc)
			goto out;
	} else if (hdr->type == TOKTYPE_CCA_INTERNAL_PKA) {
		u64 cur_mkvp = 0, old_mkvp = 0;
		struct eccprivkeytoken *t = (struct eccprivkeytoken *)key;

		if (t->secid == 0x20) {
			if (flags & PKEY_FLAGS_MATCH_CUR_MKVP)
				cur_mkvp = t->mkvp;
			if (flags & PKEY_FLAGS_MATCH_ALT_MKVP)
				old_mkvp = t->mkvp;
		} else {
			/* unknown cca internal 2 token type */
			return -EINVAL;
		}
		rc = (*klpe_cca_findcard2)(&_apqns, &_nr_apqns, 0xFFFF, 0xFFFF,
				   ZCRYPT_CEX7, APKA_MK_SET,
				   cur_mkvp, old_mkvp, 1);
		if (rc)
			goto out;
	} else
		return -EINVAL;

	if (apqns) {
		if (*nr_apqns < _nr_apqns)
			rc = -ENOSPC;
		else
			memcpy(apqns, _apqns, _nr_apqns * sizeof(u32));
	}
	*nr_apqns = _nr_apqns;

out:
	kfree(_apqns);
	return rc;
}

static int klpr_pkey_apqns4keytype(enum pkey_key_type ktype,
			      u8 cur_mkvp[32], u8 alt_mkvp[32], u32 flags,
			      struct pkey_apqn *apqns, size_t *nr_apqns)
{
	int rc;
	u32 _nr_apqns, *_apqns = NULL;

	(*klpe_zcrypt_wait_api_operational)();

	if (ktype == PKEY_TYPE_CCA_DATA || ktype == PKEY_TYPE_CCA_CIPHER) {
		u64 cur_mkvp = 0, old_mkvp = 0;
		int minhwtype = ZCRYPT_CEX3C;

		if (flags & PKEY_FLAGS_MATCH_CUR_MKVP)
			cur_mkvp = *((u64 *) cur_mkvp);
		if (flags & PKEY_FLAGS_MATCH_ALT_MKVP)
			old_mkvp = *((u64 *) alt_mkvp);
		if (ktype == PKEY_TYPE_CCA_CIPHER)
			minhwtype = ZCRYPT_CEX6;
		rc = (*klpe_cca_findcard2)(&_apqns, &_nr_apqns, 0xFFFF, 0xFFFF,
				   minhwtype, AES_MK_SET,
				   cur_mkvp, old_mkvp, 1);
		if (rc)
			goto out;
	} else if (ktype == PKEY_TYPE_CCA_ECC) {
		u64 cur_mkvp = 0, old_mkvp = 0;

		if (flags & PKEY_FLAGS_MATCH_CUR_MKVP)
			cur_mkvp = *((u64 *) cur_mkvp);
		if (flags & PKEY_FLAGS_MATCH_ALT_MKVP)
			old_mkvp = *((u64 *) alt_mkvp);
		rc = (*klpe_cca_findcard2)(&_apqns, &_nr_apqns, 0xFFFF, 0xFFFF,
				   ZCRYPT_CEX7, APKA_MK_SET,
				   cur_mkvp, old_mkvp, 1);
		if (rc)
			goto out;

	} else if (ktype == PKEY_TYPE_EP11 ||
		   ktype == PKEY_TYPE_EP11_AES ||
		   ktype == PKEY_TYPE_EP11_ECC) {
		u8 *wkvp = NULL;

		if (flags & PKEY_FLAGS_MATCH_CUR_MKVP)
			wkvp = cur_mkvp;
		rc = (*klpe_ep11_findcard2)(&_apqns, &_nr_apqns, 0xFFFF, 0xFFFF,
				    ZCRYPT_CEX7, EP11_API_V, wkvp);
		if (rc)
			goto out;

	} else
		return -EINVAL;

	if (apqns) {
		if (*nr_apqns < _nr_apqns)
			rc = -ENOSPC;
		else
			memcpy(apqns, _apqns, _nr_apqns * sizeof(u32));
	}
	*nr_apqns = _nr_apqns;

out:
	kfree(_apqns);
	return rc;
}

static int klpr_pkey_keyblob2pkey3(const struct pkey_apqn *apqns, size_t nr_apqns,
			      const u8 *key, size_t keylen, u32 *protkeytype,
			      u8 *protkey, u32 *protkeylen)
{
	int i, card, dom, rc;
	struct keytoken_header *hdr = (struct keytoken_header *)key;

	/* check for at least one apqn given */
	if (!apqns || !nr_apqns)
		return -EINVAL;

	if (keylen < sizeof(struct keytoken_header))
		return -EINVAL;

	if (hdr->type == TOKTYPE_NON_CCA
	    && hdr->version == TOKVER_EP11_AES_WITH_HEADER
	    && is_ep11_keyblob(key + sizeof(struct ep11kblob_header))) {
		/* EP11 AES key blob with header */
		if ((*klpe_ep11_check_aes_key_with_hdr)((*klpe_debug_info), 3, key, keylen, 1))
			return -EINVAL;
	} else if (hdr->type == TOKTYPE_NON_CCA
		   && hdr->version == TOKVER_EP11_ECC_WITH_HEADER
		   && is_ep11_keyblob(key + sizeof(struct ep11kblob_header))) {
		/* EP11 ECC key blob with header */
		if ((*klpe_ep11_check_ecc_key_with_hdr)((*klpe_debug_info), 3, key, keylen, 1))
			return -EINVAL;
	} else if (hdr->type == TOKTYPE_NON_CCA
		   && hdr->version == TOKVER_EP11_AES
		   && is_ep11_keyblob(key)) {
		/* EP11 AES key blob with header in session field */
		if ((*klpe_ep11_check_aes_key)((*klpe_debug_info), 3, key, keylen, 1))
			return -EINVAL;
	} else	if (hdr->type == TOKTYPE_CCA_INTERNAL) {
		if (hdr->version == TOKVER_CCA_AES) {
			/* CCA AES data key */
			if (keylen != sizeof(struct secaeskeytoken))
				return -EINVAL;
			if ((*klpe_cca_check_secaeskeytoken)((*klpe_debug_info), 3, key, 0))
				return -EINVAL;
		} else if (hdr->version == TOKVER_CCA_VLSC) {
			/* CCA AES cipher key */
			if (keylen < hdr->len || keylen > MAXCCAVLSCTOKENSIZE)
				return -EINVAL;
			if ((*klpe_cca_check_secaescipherkey)((*klpe_debug_info), 3, key, 0, 1))
				return -EINVAL;
		} else {
			KLPR_DEBUG_ERR("%s unknown CCA internal token version %d\n",
				  __func__, hdr->version);
			return -EINVAL;
		}
	} else if (hdr->type == TOKTYPE_CCA_INTERNAL_PKA) {
		/* CCA ECC (private) key */
		if (keylen < sizeof(struct eccprivkeytoken))
			return -EINVAL;
		if ((*klpe_cca_check_sececckeytoken)((*klpe_debug_info), 3, key, keylen, 1))
			return -EINVAL;
	} else if (hdr->type == TOKTYPE_NON_CCA) {
		struct pkey_protkey pkey;

		rc = (*klpe_pkey_nonccatok2pkey)(key, keylen, &pkey);
		if (rc)
			return rc;
		memcpy(protkey, pkey.protkey, pkey.len);
		*protkeylen = pkey.len;
		*protkeytype = pkey.type;
		return 0;
	} else {
		KLPR_DEBUG_ERR("%s unknown/unsupported blob type %d\n",
			  __func__, hdr->type);
		return -EINVAL;
	}

	/* simple try all apqns from the list */
	for (rc = -ENODEV, i = 0; rc && i < nr_apqns; i++) {
		card = apqns[i].card;
		dom = apqns[i].domain;
		if (hdr->type == TOKTYPE_NON_CCA
		    && (hdr->version == TOKVER_EP11_AES_WITH_HEADER
			|| hdr->version == TOKVER_EP11_ECC_WITH_HEADER)
		    && is_ep11_keyblob(key + sizeof(struct ep11kblob_header)))
			rc = (*klpe_ep11_kblob2protkey)(card, dom, key, hdr->len,
						protkey, protkeylen, protkeytype);
		else if (hdr->type == TOKTYPE_NON_CCA
			 && hdr->version == TOKVER_EP11_AES
			 && is_ep11_keyblob(key))
			rc = (*klpe_ep11_kblob2protkey)(card, dom, key, hdr->len,
						protkey, protkeylen, protkeytype);
		else if (hdr->type == TOKTYPE_CCA_INTERNAL &&
			 hdr->version == TOKVER_CCA_AES)
			rc = (*klpe_cca_sec2protkey)(card, dom, key, protkey,
					     protkeylen, protkeytype);
		else if (hdr->type == TOKTYPE_CCA_INTERNAL &&
			 hdr->version == TOKVER_CCA_VLSC)
			rc = (*klpe_cca_cipher2protkey)(card, dom, key, protkey,
						protkeylen, protkeytype);
		else if (hdr->type == TOKTYPE_CCA_INTERNAL_PKA)
			rc = (*klpe_cca_ecc2protkey)(card, dom, key, protkey,
					     protkeylen, protkeytype);
		else
			return -EINVAL;
	}

	return rc;
}

static void *_copy_key_from_user(void __user *ukey, size_t keylen)
{
	if (!ukey || keylen < MINKEYBLOBSIZE || keylen > KEYBLOBBUFSIZE)
		return ERR_PTR(-EINVAL);

	return memdup_user(ukey, keylen);
}

static void *klpp__copy_apqns_from_user(void __user *uapqns, size_t nr_apqns)
{
	if (!uapqns || nr_apqns == 0)
		return NULL;

	return memdup_array_user(uapqns, nr_apqns, sizeof(struct pkey_apqn));
}

long klpp_pkey_unlocked_ioctl(struct file *filp, unsigned int cmd,
				unsigned long arg)
{
	int rc;

	switch (cmd) {
	case PKEY_GENSECK: {
		struct pkey_genseck __user *ugs = (void __user *) arg;
		struct pkey_genseck kgs;

		if (copy_from_user(&kgs, ugs, sizeof(kgs)))
			return -EFAULT;
		rc = (*klpe_cca_genseckey)(kgs.cardnr, kgs.domain,
				   kgs.keytype, kgs.seckey.seckey);
		KLPR_DEBUG_DBG("%s cca_genseckey()=%d\n", __func__, rc);
		if (!rc && copy_to_user(ugs, &kgs, sizeof(kgs)))
			rc = -EFAULT;
		memzero_explicit(&kgs, sizeof(kgs));
		break;
	}
	case PKEY_CLR2SECK: {
		struct pkey_clr2seck __user *ucs = (void __user *) arg;
		struct pkey_clr2seck kcs;

		if (copy_from_user(&kcs, ucs, sizeof(kcs)))
			return -EFAULT;
		rc = (*klpe_cca_clr2seckey)(kcs.cardnr, kcs.domain, kcs.keytype,
				    kcs.clrkey.clrkey, kcs.seckey.seckey);
		KLPR_DEBUG_DBG("%s cca_clr2seckey()=%d\n", __func__, rc);
		if (!rc && copy_to_user(ucs, &kcs, sizeof(kcs)))
			rc = -EFAULT;
		memzero_explicit(&kcs, sizeof(kcs));
		break;
	}
	case PKEY_SEC2PROTK: {
		struct pkey_sec2protk __user *usp = (void __user *) arg;
		struct pkey_sec2protk ksp;

		if (copy_from_user(&ksp, usp, sizeof(ksp)))
			return -EFAULT;
		rc = (*klpe_cca_sec2protkey)(ksp.cardnr, ksp.domain,
				     ksp.seckey.seckey, ksp.protkey.protkey,
				     &ksp.protkey.len, &ksp.protkey.type);
		KLPR_DEBUG_DBG("%s cca_sec2protkey()=%d\n", __func__, rc);
		if (!rc && copy_to_user(usp, &ksp, sizeof(ksp)))
			rc = -EFAULT;
		memzero_explicit(&ksp, sizeof(ksp));
		break;
	}
	case PKEY_CLR2PROTK: {
		struct pkey_clr2protk __user *ucp = (void __user *) arg;
		struct pkey_clr2protk kcp;

		if (copy_from_user(&kcp, ucp, sizeof(kcp)))
			return -EFAULT;
		rc = (*klpe_pkey_clr2protkey)(kcp.keytype,
				      &kcp.clrkey, &kcp.protkey);
		KLPR_DEBUG_DBG("%s pkey_clr2protkey()=%d\n", __func__, rc);
		if (!rc && copy_to_user(ucp, &kcp, sizeof(kcp)))
			rc = -EFAULT;
		memzero_explicit(&kcp, sizeof(kcp));
		break;
	}
	case PKEY_FINDCARD: {
		struct pkey_findcard __user *ufc = (void __user *) arg;
		struct pkey_findcard kfc;

		if (copy_from_user(&kfc, ufc, sizeof(kfc)))
			return -EFAULT;
		rc = (*klpe_cca_findcard)(kfc.seckey.seckey,
				  &kfc.cardnr, &kfc.domain, 1);
		KLPR_DEBUG_DBG("%s cca_findcard()=%d\n", __func__, rc);
		if (rc < 0)
			break;
		if (copy_to_user(ufc, &kfc, sizeof(kfc)))
			return -EFAULT;
		break;
	}
	case PKEY_SKEY2PKEY: {
		struct pkey_skey2pkey __user *usp = (void __user *) arg;
		struct pkey_skey2pkey ksp;

		if (copy_from_user(&ksp, usp, sizeof(ksp)))
			return -EFAULT;
		rc = (*klpe_pkey_skey2pkey)(ksp.seckey.seckey, &ksp.protkey);
		KLPR_DEBUG_DBG("%s pkey_skey2pkey()=%d\n", __func__, rc);
		if (!rc && copy_to_user(usp, &ksp, sizeof(ksp)))
			rc = -EFAULT;
		memzero_explicit(&ksp, sizeof(ksp));
		break;
	}
	case PKEY_VERIFYKEY: {
		struct pkey_verifykey __user *uvk = (void __user *) arg;
		struct pkey_verifykey kvk;

		if (copy_from_user(&kvk, uvk, sizeof(kvk)))
			return -EFAULT;
		rc = klpr_pkey_verifykey(&kvk.seckey, &kvk.cardnr, &kvk.domain,
				    &kvk.keysize, &kvk.attributes);
		KLPR_DEBUG_DBG("%s pkey_verifykey()=%d\n", __func__, rc);
		if (!rc && copy_to_user(uvk, &kvk, sizeof(kvk)))
			rc = -EFAULT;
		memzero_explicit(&kvk, sizeof(kvk));
		break;
	}
	case PKEY_GENPROTK: {
		struct pkey_genprotk __user *ugp = (void __user *) arg;
		struct pkey_genprotk kgp;

		if (copy_from_user(&kgp, ugp, sizeof(kgp)))
			return -EFAULT;
		rc = (*klpe_pkey_genprotkey)(kgp.keytype, &kgp.protkey);
		KLPR_DEBUG_DBG("%s pkey_genprotkey()=%d\n", __func__, rc);
		if (!rc && copy_to_user(ugp, &kgp, sizeof(kgp)))
			rc = -EFAULT;
		memzero_explicit(&kgp, sizeof(kgp));
		break;
	}
	case PKEY_VERIFYPROTK: {
		struct pkey_verifyprotk __user *uvp = (void __user *) arg;
		struct pkey_verifyprotk kvp;

		if (copy_from_user(&kvp, uvp, sizeof(kvp)))
			return -EFAULT;
		rc = (*klpe_pkey_verifyprotkey)(&kvp.protkey);
		KLPR_DEBUG_DBG("%s pkey_verifyprotkey()=%d\n", __func__, rc);
		memzero_explicit(&kvp, sizeof(kvp));
		break;
	}
	case PKEY_KBLOB2PROTK: {
		struct pkey_kblob2pkey __user *utp = (void __user *) arg;
		struct pkey_kblob2pkey ktp;
		u8 *kkey;

		if (copy_from_user(&ktp, utp, sizeof(ktp)))
			return -EFAULT;
		kkey = _copy_key_from_user(ktp.key, ktp.keylen);
		if (IS_ERR(kkey))
			return PTR_ERR(kkey);
		rc = (*klpe_pkey_keyblob2pkey)(kkey, ktp.keylen, &ktp.protkey);
		KLPR_DEBUG_DBG("%s pkey_keyblob2pkey()=%d\n", __func__, rc);
		kfree_sensitive(kkey);
		if (!rc && copy_to_user(utp, &ktp, sizeof(ktp)))
			rc = -EFAULT;
		memzero_explicit(&ktp, sizeof(ktp));
		break;
	}
	case PKEY_GENSECK2: {
		struct pkey_genseck2 __user *ugs = (void __user *) arg;
		struct pkey_genseck2 kgs;
		struct pkey_apqn *apqns;
		size_t klen = KEYBLOBBUFSIZE;
		u8 *kkey;

		if (copy_from_user(&kgs, ugs, sizeof(kgs)))
			return -EFAULT;
		apqns = klpp__copy_apqns_from_user(kgs.apqns, kgs.apqn_entries);
		if (IS_ERR(apqns))
			return PTR_ERR(apqns);
		kkey = kmalloc(klen, GFP_KERNEL);
		if (!kkey) {
			kfree(apqns);
			return -ENOMEM;
		}
		rc = klpr_pkey_genseckey2(apqns, kgs.apqn_entries,
				     kgs.type, kgs.size, kgs.keygenflags,
				     kkey, &klen);
		KLPR_DEBUG_DBG("%s pkey_genseckey2()=%d\n", __func__, rc);
		kfree(apqns);
		if (rc) {
			kfree_sensitive(kkey);
			break;
		}
		if (kgs.key) {
			if (kgs.keylen < klen) {
				kfree_sensitive(kkey);
				return -EINVAL;
			}
			if (copy_to_user(kgs.key, kkey, klen)) {
				kfree_sensitive(kkey);
				return -EFAULT;
			}
		}
		kgs.keylen = klen;
		if (copy_to_user(ugs, &kgs, sizeof(kgs)))
			rc = -EFAULT;
		kfree_sensitive(kkey);
		break;
	}
	case PKEY_CLR2SECK2: {
		struct pkey_clr2seck2 __user *ucs = (void __user *) arg;
		struct pkey_clr2seck2 kcs;
		struct pkey_apqn *apqns;
		size_t klen = KEYBLOBBUFSIZE;
		u8 *kkey;

		if (copy_from_user(&kcs, ucs, sizeof(kcs)))
			return -EFAULT;
		apqns = klpp__copy_apqns_from_user(kcs.apqns, kcs.apqn_entries);
		if (IS_ERR(apqns)) {
			memzero_explicit(&kcs, sizeof(kcs));
			return PTR_ERR(apqns);
		}
		kkey = kmalloc(klen, GFP_KERNEL);
		if (!kkey) {
			kfree(apqns);
			memzero_explicit(&kcs, sizeof(kcs));
			return -ENOMEM;
		}
		rc = klpr_pkey_clr2seckey2(apqns, kcs.apqn_entries,
				      kcs.type, kcs.size, kcs.keygenflags,
				      kcs.clrkey.clrkey, kkey, &klen);
		KLPR_DEBUG_DBG("%s pkey_clr2seckey2()=%d\n", __func__, rc);
		kfree(apqns);
		if (rc) {
			kfree_sensitive(kkey);
			memzero_explicit(&kcs, sizeof(kcs));
			break;
		}
		if (kcs.key) {
			if (kcs.keylen < klen) {
				kfree_sensitive(kkey);
				memzero_explicit(&kcs, sizeof(kcs));
				return -EINVAL;
			}
			if (copy_to_user(kcs.key, kkey, klen)) {
				kfree_sensitive(kkey);
				memzero_explicit(&kcs, sizeof(kcs));
				return -EFAULT;
			}
		}
		kcs.keylen = klen;
		if (copy_to_user(ucs, &kcs, sizeof(kcs)))
			rc = -EFAULT;
		memzero_explicit(&kcs, sizeof(kcs));
		kfree_sensitive(kkey);
		break;
	}
	case PKEY_VERIFYKEY2: {
		struct pkey_verifykey2 __user *uvk = (void __user *) arg;
		struct pkey_verifykey2 kvk;
		u8 *kkey;

		if (copy_from_user(&kvk, uvk, sizeof(kvk)))
			return -EFAULT;
		kkey = _copy_key_from_user(kvk.key, kvk.keylen);
		if (IS_ERR(kkey))
			return PTR_ERR(kkey);
		rc = klpr_pkey_verifykey2(kkey, kvk.keylen,
				     &kvk.cardnr, &kvk.domain,
				     &kvk.type, &kvk.size, &kvk.flags);
		KLPR_DEBUG_DBG("%s pkey_verifykey2()=%d\n", __func__, rc);
		kfree_sensitive(kkey);
		if (rc)
			break;
		if (copy_to_user(uvk, &kvk, sizeof(kvk)))
			return -EFAULT;
		break;
	}
	case PKEY_KBLOB2PROTK2: {
		struct pkey_kblob2pkey2 __user *utp = (void __user *) arg;
		struct pkey_kblob2pkey2 ktp;
		struct pkey_apqn *apqns = NULL;
		u8 *kkey;

		if (copy_from_user(&ktp, utp, sizeof(ktp)))
			return -EFAULT;
		apqns = klpp__copy_apqns_from_user(ktp.apqns, ktp.apqn_entries);
		if (IS_ERR(apqns))
			return PTR_ERR(apqns);
		kkey = _copy_key_from_user(ktp.key, ktp.keylen);
		if (IS_ERR(kkey)) {
			kfree(apqns);
			return PTR_ERR(kkey);
		}
		rc = klpr_pkey_keyblob2pkey2(apqns, ktp.apqn_entries,
					kkey, ktp.keylen, &ktp.protkey);
		KLPR_DEBUG_DBG("%s pkey_keyblob2pkey2()=%d\n", __func__, rc);
		kfree(apqns);
		kfree_sensitive(kkey);
		if (!rc && copy_to_user(utp, &ktp, sizeof(ktp)))
			rc = -EFAULT;
		memzero_explicit(&ktp, sizeof(ktp));
		break;
	}
	case PKEY_APQNS4K: {
		struct pkey_apqns4key __user *uak = (void __user *) arg;
		struct pkey_apqns4key kak;
		struct pkey_apqn *apqns = NULL;
		size_t nr_apqns, len;
		u8 *kkey;

		if (copy_from_user(&kak, uak, sizeof(kak)))
			return -EFAULT;
		nr_apqns = kak.apqn_entries;
		if (nr_apqns) {
			apqns = kmalloc_array(nr_apqns,
					      sizeof(struct pkey_apqn),
					      GFP_KERNEL);
			if (!apqns)
				return -ENOMEM;
		}
		kkey = _copy_key_from_user(kak.key, kak.keylen);
		if (IS_ERR(kkey)) {
			kfree(apqns);
			return PTR_ERR(kkey);
		}
		rc = klpr_pkey_apqns4key(kkey, kak.keylen, kak.flags,
				    apqns, &nr_apqns);
		KLPR_DEBUG_DBG("%s pkey_apqns4key()=%d\n", __func__, rc);
		kfree_sensitive(kkey);
		if (rc && rc != -ENOSPC) {
			kfree(apqns);
			break;
		}
		if (!rc && kak.apqns) {
			if (nr_apqns > kak.apqn_entries) {
				kfree(apqns);
				return -EINVAL;
			}
			len = nr_apqns * sizeof(struct pkey_apqn);
			if (len) {
				if (copy_to_user(kak.apqns, apqns, len)) {
					kfree(apqns);
					return -EFAULT;
				}
			}
		}
		kak.apqn_entries = nr_apqns;
		if (copy_to_user(uak, &kak, sizeof(kak)))
			rc = -EFAULT;
		kfree(apqns);
		break;
	}
	case PKEY_APQNS4KT: {
		struct pkey_apqns4keytype __user *uat = (void __user *) arg;
		struct pkey_apqns4keytype kat;
		struct pkey_apqn *apqns = NULL;
		size_t nr_apqns, len;

		if (copy_from_user(&kat, uat, sizeof(kat)))
			return -EFAULT;
		nr_apqns = kat.apqn_entries;
		if (nr_apqns) {
			apqns = kmalloc_array(nr_apqns,
					      sizeof(struct pkey_apqn),
					      GFP_KERNEL);
			if (!apqns)
				return -ENOMEM;
		}
		rc = klpr_pkey_apqns4keytype(kat.type, kat.cur_mkvp, kat.alt_mkvp,
					kat.flags, apqns, &nr_apqns);
		KLPR_DEBUG_DBG("%s pkey_apqns4keytype()=%d\n", __func__, rc);
		if (rc && rc != -ENOSPC) {
			kfree(apqns);
			break;
		}
		if (!rc && kat.apqns) {
			if (nr_apqns > kat.apqn_entries) {
				kfree(apqns);
				return -EINVAL;
			}
			len = nr_apqns * sizeof(struct pkey_apqn);
			if (len) {
				if (copy_to_user(kat.apqns, apqns, len)) {
					kfree(apqns);
					return -EFAULT;
				}
			}
		}
		kat.apqn_entries = nr_apqns;
		if (copy_to_user(uat, &kat, sizeof(kat)))
			rc = -EFAULT;
		kfree(apqns);
		break;
	}
	case PKEY_KBLOB2PROTK3: {
		struct pkey_kblob2pkey3 __user *utp = (void __user *) arg;
		struct pkey_kblob2pkey3 ktp;
		struct pkey_apqn *apqns = NULL;
		u32 protkeylen = PROTKEYBLOBBUFSIZE;
		u8 *kkey, *protkey;

		if (copy_from_user(&ktp, utp, sizeof(ktp)))
			return -EFAULT;
		apqns = klpp__copy_apqns_from_user(ktp.apqns, ktp.apqn_entries);
		if (IS_ERR(apqns))
			return PTR_ERR(apqns);
		kkey = _copy_key_from_user(ktp.key, ktp.keylen);
		if (IS_ERR(kkey)) {
			kfree(apqns);
			return PTR_ERR(kkey);
		}
		protkey = kmalloc(protkeylen, GFP_KERNEL);
		if (!protkey) {
			kfree(apqns);
			kfree_sensitive(kkey);
			return -ENOMEM;
		}
		rc = klpr_pkey_keyblob2pkey3(apqns, ktp.apqn_entries, kkey,
					ktp.keylen, &ktp.pkeytype,
					protkey, &protkeylen);
		KLPR_DEBUG_DBG("%s pkey_keyblob2pkey3()=%d\n", __func__, rc);
		kfree(apqns);
		kfree_sensitive(kkey);
		if (rc) {
			kfree_sensitive(protkey);
			break;
		}
		if (ktp.pkey && ktp.pkeylen) {
			if (protkeylen > ktp.pkeylen) {
				kfree_sensitive(protkey);
				return -EINVAL;
			}
			if (copy_to_user(ktp.pkey, protkey, protkeylen)) {
				kfree_sensitive(protkey);
				return -EFAULT;
			}
		}
		kfree_sensitive(protkey);
		ktp.pkeylen = protkeylen;
		if (copy_to_user(utp, &ktp, sizeof(ktp)))
			return -EFAULT;
		break;
	}
	default:
		/* unknown/unsupported ioctl cmd */
		return -ENOTTY;
	}

	return rc;
}


#include "livepatch_bsc1246189.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "pkey"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "cca_check_secaescipherkey", (void *)&klpe_cca_check_secaescipherkey,
	  "zcrypt" },
	{ "cca_check_secaeskeytoken", (void *)&klpe_cca_check_secaeskeytoken,
	  "zcrypt" },
	{ "cca_check_sececckeytoken", (void *)&klpe_cca_check_sececckeytoken,
	  "zcrypt" },
	{ "cca_cipher2protkey", (void *)&klpe_cca_cipher2protkey, "zcrypt" },
	{ "cca_clr2cipherkey", (void *)&klpe_cca_clr2cipherkey, "zcrypt" },
	{ "cca_clr2seckey", (void *)&klpe_cca_clr2seckey, "zcrypt" },
	{ "cca_ecc2protkey", (void *)&klpe_cca_ecc2protkey, "zcrypt" },
	{ "cca_findcard", (void *)&klpe_cca_findcard, "zcrypt" },
	{ "cca_findcard2", (void *)&klpe_cca_findcard2, "zcrypt" },
	{ "cca_gencipherkey", (void *)&klpe_cca_gencipherkey, "zcrypt" },
	{ "cca_genseckey", (void *)&klpe_cca_genseckey, "zcrypt" },
	{ "cca_sec2protkey", (void *)&klpe_cca_sec2protkey, "zcrypt" },
	{ "ep11_check_aes_key", (void *)&klpe_ep11_check_aes_key, "zcrypt" },
	{ "ep11_check_aes_key_with_hdr",
	  (void *)&klpe_ep11_check_aes_key_with_hdr, "zcrypt" },
	{ "ep11_check_ecc_key_with_hdr",
	  (void *)&klpe_ep11_check_ecc_key_with_hdr, "zcrypt" },
	{ "ep11_clr2keyblob", (void *)&klpe_ep11_clr2keyblob, "zcrypt" },
	{ "ep11_findcard2", (void *)&klpe_ep11_findcard2, "zcrypt" },
	{ "ep11_genaeskey", (void *)&klpe_ep11_genaeskey, "zcrypt" },
	{ "ep11_kblob2protkey", (void *)&klpe_ep11_kblob2protkey, "zcrypt" },
	{ "zcrypt_wait_api_operational",
	  (void *)&klpe_zcrypt_wait_api_operational, "zcrypt" },
	{ "debug_info", (void *)&klpe_debug_info, "pkey" },
	{ "pkey_clr2protkey", (void *)&klpe_pkey_clr2protkey, "pkey" },
	{ "pkey_genprotkey", (void *)&klpe_pkey_genprotkey, "pkey" },
	{ "pkey_keyblob2pkey", (void *)&klpe_pkey_keyblob2pkey, "pkey" },
	{ "pkey_nonccatok2pkey", (void *)&klpe_pkey_nonccatok2pkey, "pkey" },
	{ "pkey_skey2pkey", (void *)&klpe_pkey_skey2pkey, "pkey" },
	{ "pkey_verifyprotkey", (void *)&klpe_pkey_verifyprotkey, "pkey" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1246189_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void livepatch_bsc1246189_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}

#endif /* IS_ENABLED(CONFIG_PKEY) */
