/*
 * Copyright IBM Corp. and others 2023
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 */
package org.openj9.test.lworld;

import org.objectweb.asm.*;

import static org.objectweb.asm.Opcodes.*;

public class ValhallaUtils {
	/**
	 * Currently value type is built on JDK26, so use java file major version (44 + 26) for now.
	 * If moved this needs to be incremented to the next class file version.
	 * VALUE_TYPES_MAJOR_VERSION in oti/j9consts.h needs to be updated as well.
	 * Minor version is in 16 most significant bits for asm.
	 */
	static final int VALUE_TYPE_CLASS_FILE_VERSION = (65535 << 16) | (44 + 26);

	/* workaround till the new ASM is released */
	static final int ACC_IDENTITY = 0x20;
	static final int ACC_STRICT_INIT = ACC_STRICT;

	/* ImplicitCreation flags */
	static final int ACC_DEFAULT = 0x1;
	static final int ACC_NON_ATOMIC = 0x2;

	static final class LoadableDescriptorsAttribute extends Attribute {
		private final String[] classes;

		public LoadableDescriptorsAttribute(String[] classes) {
			super("LoadableDescriptors");
			this.classes = classes;
		}

		public Label[] getLabels() {
			return null;
		}

		public boolean isCodeAttribute() {
			return false;
		}

		public boolean isUnknown() {
			return true;
		}

		public ByteVector write(ClassWriter cw, byte[] code, int len, int maxStack, int maxLocals) {
			ByteVector b = new ByteVector();

			b.putShort(classes.length);

			int cpIndex;
			for (String clazz : classes) {
				cpIndex = cw.newUTF8(clazz.replace('.', '/'));
				b.putShort(cpIndex);
			}
			return b;
		}
	}

	static final class ImplicitCreationAttribute extends Attribute {
		private final int flags;

		public ImplicitCreationAttribute() {
			super("ImplicitCreation");
			/* this is the default flag generated by the compiler for the implicit modifier. */
			this.flags = ValhallaUtils.ACC_DEFAULT;
		}

		public ImplicitCreationAttribute(int flags) {
			super("ImplicitCreation");
			this.flags = flags;
		}

		public Label[] getLabels() {
			return null;
		}

		public boolean isCodeAttribute() {
			return false;
		}

		public boolean isUnknown() {
			return true;
		}

		public ByteVector write(ClassWriter cw, byte[] code, int len, int maxStack, int maxLocals) {
			ByteVector b = new ByteVector();
			b.putShort(flags);
			return b;
		}
	}

	static final class NullRestrictedAttribute extends Attribute {
		public NullRestrictedAttribute() {
			super("NullRestricted");
		}

		public Label[] getLabels() {
			return null;
		}

		public boolean isCodeAttribute() {
			return false;
		}

		public boolean isUnknown() {
			return true;
		}

		public ByteVector write(ClassWriter cw, byte[] code, int len, int maxStack, int maxLocals) {
			return new ByteVector();
		}
	}
}
