/*
 * Copyright IBM Corp. and others 2001
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 */
package org.openj9.test.floatsanity.custom;

import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;
import org.testng.log4testng.Logger;

@Test(groups = { "level.sanity" })
public class CheckStrictMath {

	static long expectedResults1[][];
	static long expectedResults2[][];

	public static Logger logger = Logger.getLogger(CheckStrictMath.class);

	@BeforeClass
	public void groupName() {
		try {
			Class.forName("java.lang.StrictMath");
		} catch (ClassNotFoundException e) {
			Assert.fail("java.lang.StrictMath not found, CheckStrictMath tests skipped");
		}
		logger.debug("Check StrictMath functions for exactness");
		initializeResults();
	}

	public void strictSinTest() {
		for (int i = 0; i < expectedResults1.length; i++) {
			double arg = Double.longBitsToDouble(expectedResults1[i][0]);
			double expected = Double.longBitsToDouble(expectedResults1[i][1]);
			String operation = "testing StrictMath: double StrictMath.sin( " + arg + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.sin(arg);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	public void strictCosTest() {
		for (int i = 0; i < expectedResults1.length; i++) {
			double arg = Double.longBitsToDouble(expectedResults1[i][0]);
			double expected = Double.longBitsToDouble(expectedResults1[i][2]);
			String operation = "testing StrictMath: double StrictMath.cos( " + arg + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.cos(arg);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	public void strictTanTest() {
		for (int i = 0; i < expectedResults1.length; i++) {
			double arg = Double.longBitsToDouble(expectedResults1[i][0]);
			double expected = Double.longBitsToDouble(expectedResults1[i][3]);
			String operation = "testing StrictMath: double StrictMath.tan( " + arg + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.tan(arg);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	public void strictAsinTest() {
		for (int i = 0; i < expectedResults1.length; i++) {
			double arg = Double.longBitsToDouble(expectedResults1[i][0]);
			double expected = Double.longBitsToDouble(expectedResults1[i][4]);
			String operation = "testing StrictMath: double StrictMath.asin( " + arg + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.asin(arg);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, Math.ulp(expected), operation);
			}
		}
	}

	public void strictAcosTest() {
		for (int i = 0; i < expectedResults1.length; i++) {
			double arg = Double.longBitsToDouble(expectedResults1[i][0]);
			double expected = Double.longBitsToDouble(expectedResults1[i][5]);
			String operation = "testing StrictMath: double StrictMath.acos( " + arg + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.acos(arg);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	public void strictAtanTest() {
		for (int i = 0; i < expectedResults1.length; i++) {
			double arg = Double.longBitsToDouble(expectedResults1[i][0]);
			double expected = Double.longBitsToDouble(expectedResults1[i][6]);
			String operation = "testing StrictMath: double StrictMath.atan( " + arg + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.atan(arg);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	public void strictExpTest() {
		for (int i = 0; i < expectedResults1.length; i++) {
			double arg = Double.longBitsToDouble(expectedResults1[i][0]);
			double expected = Double.longBitsToDouble(expectedResults1[i][7]);
			String operation = "testing StrictMath: double StrictMath.exp( " + arg + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.exp(arg);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	public void strictLogTest() {
		for (int i = 0; i < expectedResults1.length; i++) {
			double arg = Double.longBitsToDouble(expectedResults1[i][0]);
			double expected = Double.longBitsToDouble(expectedResults1[i][8]);
			String operation = "testing StrictMath: double StrictMath.log( " + arg + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.log(arg);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	public void strictAtan2Test() {
		for (int i = 0; i < expectedResults2.length; i++) {
			double arg1 = Double.longBitsToDouble(expectedResults2[i][0]);
			double arg2 = Double.longBitsToDouble(expectedResults2[i][1]);
			double expected = Double.longBitsToDouble(expectedResults2[i][2]);
			String operation = "testing StrictMath: double StrictMath.atan2( " + arg1 + " , " + arg2 + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.atan2(arg1, arg2);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	public void strictPowTest() {
		for (int i = 0; i < expectedResults2.length; i++) {
			double arg1 = Double.longBitsToDouble(expectedResults2[i][0]);
			double arg2 = Double.longBitsToDouble(expectedResults2[i][1]);
			double expected = Double.longBitsToDouble(expectedResults2[i][3]);
			String operation = "testing StrictMath: double StrictMath.pow( " + arg1 + " , " + arg2 + " ) == " + expected;
			logger.debug(operation);
			double result = StrictMath.pow(arg1, arg2);
			if (Double.isNaN(expected)) {
				Assert.assertTrue(Double.isNaN(expected), operation);
			} else {
				Assert.assertEquals(result, expected, 0, operation);
			}
		}
	}

	/**
	 * Initializes the expectedResults arrays with input values
	 * and (hardcoded) expected output values.
	 */
	static public void initializeResults() {
		expectedResults1 = new long[127][9];	//First column contains input values
		expectedResults2 = new long[33][4];		//First two columns contain input values
		
		//expectedResults1 -- for functions with 1 arg
		expectedResults1[0][0] = 0x3fd58139f95ac4e8l;
		expectedResults1[0][1] = 0x3fd51a373acfdca7l;
		expectedResults1[0][2] = 0x3fee35e0f4b55e22l;
		expectedResults1[0][3] = 0x3fd65a382e2e6682l;
		expectedResults1[0][4] = 0x3fd5ee7793246ee4l;
		expectedResults1[0][5] = 0x3ff3a65d6f7b115fl;
		expectedResults1[0][6] = 0x3fd4bf087919d602l;
		expectedResults1[0][7] = 0x3ff663c373081452l;
		expectedResults1[0][8] = 0xbff17320af371d10l;
		expectedResults1[1][0] = 0x3fd4d753ca24770cl;
		expectedResults1[1][1] = 0x3fd47987bfff310al;
		expectedResults1[1][2] = 0x3fee5178f065f97el;
		expectedResults1[1][3] = 0x3fd59c46eeff3ccbl;
		expectedResults1[1][4] = 0x3fd53a6dd5fba55bl;
		expectedResults1[1][5] = 0x3ff3d35fdec543c2l;
		expectedResults1[1][6] = 0x3fd425e52d374fd5l;
		expectedResults1[1][7] = 0x3ff628a21af76d6al;
		expectedResults1[1][8] = 0xbff1f3871a479ce6l;
		expectedResults1[2][0] = 0x3fe1f1b5e2ee80f8l;
		expectedResults1[2][1] = 0x3fe104b894a3cb77l;
		expectedResults1[2][2] = 0x3feb196c6ff072f5l;
		expectedResults1[2][3] = 0x3fe4189933b68763l;
		expectedResults1[2][4] = 0x3fe30caeca684475l;
		expectedResults1[2][5] = 0x3fef3747de2015bcl;
		expectedResults1[2][6] = 0x3fe05aa1a5894b1el;
		expectedResults1[2][7] = 0x3ffc082ce53d1b82l;
		expectedResults1[2][8] = 0xbfe282d37c0aae48l;
		expectedResults1[3][0] = 0x3fe49c30b7f0c2cdl;
		expectedResults1[3][1] = 0x3fe336e710633129l;
		expectedResults1[3][2] = 0x3fe996d2488f93f4l;
		expectedResults1[3][3] = 0x3fe8073c7985e209l;
		expectedResults1[3][4] = 0x3fe664c7107a43b8l;
		expectedResults1[3][5] = 0x3febdf2f980e1679l;
		expectedResults1[3][6] = 0x3fe24f66ad79ca87l;
		expectedResults1[3][7] = 0x3ffe77a2dbceb852l;
		expectedResults1[3][8] = 0xbfdc2833d4d98c74l;
		expectedResults1[4][0] = 0x3fe7d46bb0e4050cl;
		expectedResults1[4][1] = 0x3fe5b004f3fe2209l;
		expectedResults1[4][2] = 0x3fe7879d9963bbe4l;
		expectedResults1[4][3] = 0x3fed7ea36427b8e4l;
		expectedResults1[4][4] = 0x3feae1bc5e451773l;
		expectedResults1[4][5] = 0x3fe7623a4a4342bdl;
		expectedResults1[4][6] = 0x3fe47b995947302al;
		expectedResults1[4][7] = 0x4000d89ce7cb96d1l;
		expectedResults1[4][8] = 0xbfd2de023c2740cdl;
		expectedResults1[5][0] = 0x3fc0e1f109547674l;
		expectedResults1[5][1] = 0x3fc0d56be415defcl;
		expectedResults1[5][2] = 0x3fefb8d90b2a9ffel;
		expectedResults1[5][3] = 0x3fc0fb2dde0d7d33l;
		expectedResults1[5][4] = 0x3fc0ee9258db82del;
		expectedResults1[5][5] = 0x3ff704290928bcbcl;
		expectedResults1[5][6] = 0x3fc0c923504efb31l;
		expectedResults1[5][7] = 0x3ff2417d64992e83l;
		expectedResults1[5][8] = 0xc00034bb35339376l;
		expectedResults1[6][0] = 0x3fd1a461a53076c8l;
		expectedResults1[6][1] = 0x3fd16b667e833319l;
		expectedResults1[6][2] = 0x3feecab8c00adf4fl;
		expectedResults1[6][3] = 0x3fd21a5d62f483f6l;
		expectedResults1[6][4] = 0x3fd1dfa0e9ad847el;
		expectedResults1[6][5] = 0x3ff4aa1319d8cbf9l;
		expectedResults1[6][6] = 0x3fd136ef08337d8al;
		expectedResults1[6][7] = 0x3ff5140f305f354fl;
		expectedResults1[6][8] = 0xbff49e14e4e43098l;
		expectedResults1[7][0] = 0x3fe6d154187c4bbfl;
		expectedResults1[7][1] = 0x3fe4eec348fa586cl;
		expectedResults1[7][2] = 0x3fe8342b90f4e4fcl;
		expectedResults1[7][3] = 0x3febacdb9df13a74l;
		expectedResults1[7][4] = 0x3fe9672920d59fafl;
		expectedResults1[7][5] = 0x3fe8dccd87b2ba82l;
		expectedResults1[7][6] = 0x3fe3d2645a4493d5l;
		expectedResults1[7][7] = 0x40005259acf27b96l;
		expectedResults1[7][8] = 0xbfd5a510d5e5fe4dl;
		expectedResults1[8][0] = 0x3fe7ba16f9d650c2l;
		expectedResults1[8][1] = 0x3fe59ca123fe0dbdl;
		expectedResults1[8][2] = 0x3fe7996e0e513a09l;
		expectedResults1[8][3] = 0x3fed4e14e7af2c4cl;
		expectedResults1[8][4] = 0x3feaba63ae2fcc55l;
		expectedResults1[8][5] = 0x3fe78992fa588ddbl;
		expectedResults1[8][6] = 0x3fe46aa28b75a0f0l;
		expectedResults1[8][7] = 0x4000cac601bc684bl;
		expectedResults1[8][8] = 0xbfd324e1111ac992l;
		expectedResults1[9][0] = 0x3fe8ff6c11a7fd62l;
		expectedResults1[9][1] = 0x3fe6882143e85738l;
		expectedResults1[9][2] = 0x3fe6b90124bd062fl;
		expectedResults1[9][3] = 0x3fefbb2bb5b7df9el;
		expectedResults1[9][4] = 0x3fecb08f8c77b0d4l;
		expectedResults1[9][5] = 0x3fe593671c10a95cl;
		expectedResults1[9][6] = 0x3fe538999de9b836l;
		expectedResults1[9][7] = 0x400178ed7ff1a047l;
		expectedResults1[9][8] = 0xbfcf9c11dd9dba58l;
		expectedResults1[10][0] = 0x3fda4581212ad2ccl;
		expectedResults1[10][1] = 0x3fd98a35e1ef9f53l;
		expectedResults1[10][2] = 0x3fed5772128b9e61l;
		expectedResults1[10][3] = 0x3fdbda9667e67156l;
		expectedResults1[10][4] = 0x3fdb1254851ee9a6l;
		expectedResults1[10][5] = 0x3ff25d6632fc72afl;
		expectedResults1[10][6] = 0x3fd8eddf2887c7d5l;
		expectedResults1[10][7] = 0x3ff81ef6dc1330a1l;
		expectedResults1[10][8] = 0xbfec7e249e708036l;
		expectedResults1[11][0] = 0x3fd48eb0e8c22c3cl;
		expectedResults1[11][1] = 0x3fd434a8fce241a7l;
		expectedResults1[11][2] = 0x3fee5d03d163d5dal;
		expectedResults1[11][3] = 0x3fd54b7ac1d12023l;
		expectedResults1[11][4] = 0x3fd4edaac18d32fcl;
		expectedResults1[11][5] = 0x3ff3e690a3e0e059l;
		expectedResults1[11][6] = 0x3fd3e42337b1ca1el;
		expectedResults1[11][7] = 0x3ff60f8a3fc11af9l;
		expectedResults1[11][8] = 0xbff22bacb320e642l;
		expectedResults1[12][0] = 0x3fde27bd7edbf5f0l;
		expectedResults1[12][1] = 0x3fdd0d416cdd4a9cl;
		expectedResults1[12][2] = 0x3fec835c6c486212l;
		expectedResults1[12][3] = 0x3fe04d60fb2f6f86l;
		expectedResults1[12][4] = 0x3fdf665e4c2c2f99l;
		expectedResults1[12][5] = 0x3ff14863c1392132l;
		expectedResults1[12][6] = 0x3fdc2e41ced1d1f8l;
		expectedResults1[12][7] = 0x3ff9a148e8eed700l;
		expectedResults1[12][8] = 0xbfe814ae55f71f0al;
		expectedResults1[13][0] = 0x3fc04b98dca6b9acl;
		expectedResults1[13][1] = 0x3fc0405687b227eel;
		expectedResults1[13][2] = 0x3fefbdb49504a2bel;
		expectedResults1[13][3] = 0x3fc06247d9005debl;
		expectedResults1[13][4] = 0x3fc056f2c62d5f45l;
		expectedResults1[13][5] = 0x3ff7171cfb7e8130l;
		expectedResults1[13][6] = 0x3fc03546fc631ab9l;
		expectedResults1[13][7] = 0x3ff22c18a51e0062l;
		expectedResults1[13][8] = 0xc0007d3e072b7cfdl;
		expectedResults1[14][0] = 0x3fe1d234547f8161l;
		expectedResults1[14][1] = 0x3fe0ea0201296c9al;
		expectedResults1[14][2] = 0x3feb2a20c45f1206l;
		expectedResults1[14][3] = 0x3fe3ecc5bbf60098l;
		expectedResults1[14][4] = 0x3fe2e6b0ccda08f0l;
		expectedResults1[14][5] = 0x3fef5d45dbae5140l;
		expectedResults1[14][6] = 0x3fe0429f7f115743l;
		expectedResults1[14][7] = 0x3ffbeca10dbdbe1bl;
		expectedResults1[14][8] = 0xbfe2bb3467d6e42dl;
		expectedResults1[15][0] = 0x3fe3af128232e8e3l;
		expectedResults1[15][1] = 0x3fe2774131a0a026l;
		expectedResults1[15][2] = 0x3fea226f7dbe9899l;
		expectedResults1[15][3] = 0x3fe69c36b236f94el;
		expectedResults1[15][4] = 0x3fe533863ac2b90dl;
		expectedResults1[15][5] = 0x3fed10706dc5a123l;
		expectedResults1[15][6] = 0x3fe1a597834bd750l;
		expectedResults1[15][7] = 0x3ffd991c77314f0cl;
		expectedResults1[15][8] = 0xbfdf19939afc9947l;
		expectedResults1[16][0] = 0x3fe8cf7cf31f38d8l;
		expectedResults1[16][1] = 0x3fe665fe843645c5l;
		expectedResults1[16][2] = 0x3fe6daa7e90d2494l;
		expectedResults1[16][3] = 0x3fef5ca6ed397ec1l;
		expectedResults1[16][4] = 0x3fec643997d63f1el;
		expectedResults1[16][5] = 0x3fe5dfbd10b21b13l;
		expectedResults1[16][6] = 0x3fe51abf4926751el;
		expectedResults1[16][7] = 0x40015ed4e3b81787l;
		expectedResults1[16][8] = 0xbfd04938167d77fdl;
		expectedResults1[17][0] = 0x3fe871053cd8f2bfl;
		expectedResults1[17][1] = 0x3fe6222580241f5el;
		expectedResults1[17][2] = 0x3fe71c6359a77858l;
		expectedResults1[17][3] = 0x3feea58221e88dfbl;
		expectedResults1[17][4] = 0x3febd04745223fd0l;
		expectedResults1[17][5] = 0x3fe673af63661a60l;
		expectedResults1[17][6] = 0x3fe4df6a9e1a792bl;
		expectedResults1[17][7] = 0x40012bd8bd7f3de1l;
		expectedResults1[17][8] = 0xbfd13ebbd3fe14a3l;
		expectedResults1[18][0] = 0x3f932ffb96cf3360l;
		expectedResults1[18][1] = 0x3f932fb201cf7008l;
		expectedResults1[18][2] = 0x3feffe8fda6b327fl;
		expectedResults1[18][3] = 0x3f93308ec6c2beael;
		expectedResults1[18][4] = 0x3f9330452f1db15dl;
		expectedResults1[18][5] = 0x3ff8d53a3f87b653l;
		expectedResults1[18][6] = 0x3f932f6874159f27l;
		expectedResults1[18][7] = 0x3ff04d792a3e37dcl;
		expectedResults1[18][8] = 0xc00fd156c5f1010bl;
		expectedResults1[19][0] = 0x3fb378cd50af46e8l;
		expectedResults1[19][1] = 0x3fb373ff3256949fl;
		expectedResults1[19][2] = 0x3fefe85074e6a04fl;
		expectedResults1[19][3] = 0x3fb3826ff8f3ec07l;
		expectedResults1[19][4] = 0x3fb37d9f011063cdl;
		expectedResults1[19][5] = 0x3ff7ea21643326dbl;
		expectedResults1[19][6] = 0x3fb36f38dfaea990l;
		expectedResults1[19][7] = 0x3ff143b4749e4b2el;
		expectedResults1[19][8] = 0xc0049c11d9fb7b5fl;
		expectedResults1[20][0] = 0x3ff921f0d7e968a9L;
		expectedResults1[20][1] = 0x3feffffffff920c8L;
		expectedResults1[20][2] = 0x3ee4f8b588dd0ce3L;
		expectedResults1[20][3] = 0x40f86a0000022921L;
		expectedResults1[20][4] = 0x7ff8000000000000L;
		expectedResults1[20][5] = 0x7ff8000000000000L;
		expectedResults1[20][6] = 0x3ff00fe681c0d759L;
		expectedResults1[20][7] = 0x40133de12c17fc98L;
		expectedResults1[20][8] = 0x3fdce6a071fd6622L;
		expectedResults1[21][0] = 0xc01921fb54442d18L;
		expectedResults1[21][1] = 0x3cb1a62633145c07L;
		expectedResults1[21][2] = 0x3ff0000000000000L;
		expectedResults1[21][3] = 0x3cb1a62633145c07L;
		expectedResults1[21][4] = 0x7ff8000000000000L;
		expectedResults1[21][5] = 0x7ff8000000000000L;
		expectedResults1[21][6] = 0xbff69b8154baf42eL;
		expectedResults1[21][7] = 0x3f5e989f5d6dff5eL;
		expectedResults1[21][8] = 0x7ff8000000000000L;
		expectedResults1[22][0] = 0x3ff921fb54442d1bL;
		expectedResults1[22][1] = 0x3ff0000000000000L;
		expectedResults1[22][2] = 0xbcc5cb3b399d747fL;
		expectedResults1[22][3] = 0xc3177e2ae31272dbL;
		expectedResults1[22][4] = 0x7ff8000000000000L;
		expectedResults1[22][5] = 0x7ff8000000000000L;
		expectedResults1[22][6] = 0x3ff00fe987ed0300L;
		expectedResults1[22][7] = 0x40133dedc8559363L;
		expectedResults1[22][8] = 0x3fdce6bb25aa131dL;
		expectedResults1[23][0] = 0xffefffffffffffffL;
		expectedResults1[23][1] = 0xbf7452fc98b34e97L;
		expectedResults1[23][2] = 0xbfefffe62ecfab75L;
		expectedResults1[23][3] = 0x3f74530cfe729484L;
		expectedResults1[23][4] = 0x7ff8000000000000L;
		expectedResults1[23][5] = 0x7ff8000000000000L;
		expectedResults1[23][6] = 0xbff921fb54442d18L;
		expectedResults1[23][7] = 0x0L;
		expectedResults1[23][8] = 0x7ff8000000000000L;
		expectedResults1[24][0] = 0xffecccccccccccccL;
		expectedResults1[24][1] = 0x3fe5db7d99630b7eL;
		expectedResults1[24][2] = 0x3fe75f41752c793cL;
		expectedResults1[24][3] = 0x3feded16fd2cea61L;
		expectedResults1[24][4] = 0x7ff8000000000000L;
		expectedResults1[24][5] = 0x7ff8000000000000L;
		expectedResults1[24][6] = 0xbff921fb54442d18L;
		expectedResults1[24][7] = 0x0L;
		expectedResults1[24][8] = 0x7ff8000000000000L;
		expectedResults1[25][0] = 0xffe9999999999999L;
		expectedResults1[25][1] = 0xbfefead33cce3294L;
		expectedResults1[25][2] = 0xbfb26501cc4f4b13L;
		expectedResults1[25][3] = 0x402bc32ac72af18aL;
		expectedResults1[25][4] = 0x7ff8000000000000L;
		expectedResults1[25][5] = 0x7ff8000000000000L;
		expectedResults1[25][6] = 0xbff921fb54442d18L;
		expectedResults1[25][7] = 0x0L;
		expectedResults1[25][8] = 0x7ff8000000000000L;
		expectedResults1[26][0] = 0xffe6666666666666L;
		expectedResults1[26][1] = 0x3fe8fb5401d9641eL;
		expectedResults1[26][2] = 0xbfe3ff700993c32eL;
		expectedResults1[26][3] = 0xbff3fcd3143ea3f4L;
		expectedResults1[26][4] = 0x7ff8000000000000L;
		expectedResults1[26][5] = 0x7ff8000000000000L;
		expectedResults1[26][6] = 0xbff921fb54442d18L;
		expectedResults1[26][7] = 0x0L;
		expectedResults1[26][8] = 0x7ff8000000000000L;
		expectedResults1[27][0] = 0xffe3333333333333L;
		expectedResults1[27][1] = 0xbfc2f9b111618394L;
		expectedResults1[27][2] = 0x3fefa57bdf3b0f0eL;
		expectedResults1[27][3] = 0xbfc32ff732203a72L;
		expectedResults1[27][4] = 0x7ff8000000000000L;
		expectedResults1[27][5] = 0x7ff8000000000000L;
		expectedResults1[27][6] = 0xbff921fb54442d18L;
		expectedResults1[27][7] = 0x0L;
		expectedResults1[27][8] = 0x7ff8000000000000L;
		expectedResults1[28][0] = 0xffe0000000000000L;
		expectedResults1[28][1] = 0xbfe205248cbdb760L;
		expectedResults1[28][2] = 0xbfea719f26c232beL;
		expectedResults1[28][3] = 0x3fe5ce6b4c0d02a3L;
		expectedResults1[28][4] = 0x7ff8000000000000L;
		expectedResults1[28][5] = 0x7ff8000000000000L;
		expectedResults1[28][6] = 0xbff921fb54442d18L;
		expectedResults1[28][7] = 0x0L;
		expectedResults1[28][8] = 0x7ff8000000000000L;
		expectedResults1[29][0] = 0xffd999999999999aL;
		expectedResults1[29][1] = 0x3fef3048f2c8e08bL;
		expectedResults1[29][2] = 0x3fcca42f239b6d74L;
		expectedResults1[29][3] = 0x40116c48f7ff1c34L;
		expectedResults1[29][4] = 0x7ff8000000000000L;
		expectedResults1[29][5] = 0x7ff8000000000000L;
		expectedResults1[29][6] = 0xbff921fb54442d18L;
		expectedResults1[29][7] = 0x0L;
		expectedResults1[29][8] = 0x7ff8000000000000L;
		expectedResults1[30][0] = 0xffd3333333333334L;
		expectedResults1[30][1] = 0xbfebbfecdc58ea25L;
		expectedResults1[30][2] = 0x3fdfdf31928d6af3L;
		expectedResults1[30][3] = 0xbffbdc7d141d05f6L;
		expectedResults1[30][4] = 0x7ff8000000000000L;
		expectedResults1[30][5] = 0x7ff8000000000000L;
		expectedResults1[30][6] = 0xbff921fb54442d18L;
		expectedResults1[30][7] = 0x0L;
		expectedResults1[30][8] = 0x7ff8000000000000L;
		expectedResults1[31][0] = 0xffc999999999999cL;
		expectedResults1[31][1] = 0x3fd311ae84602e3dL;
		expectedResults1[31][2] = 0xbfee8bebb3b7f6b2L;
		expectedResults1[31][3] = 0xbfd3f9f4ef99370cL;
		expectedResults1[31][4] = 0x7ff8000000000000L;
		expectedResults1[31][5] = 0x7ff8000000000000L;
		expectedResults1[31][6] = 0xbff921fb54442d18L;
		expectedResults1[31][7] = 0x0L;
		expectedResults1[31][8] = 0x7ff8000000000000L;
		expectedResults1[32][0] = 0xffb999999999999fL;
		expectedResults1[32][1] = 0x3fe8b77bd74ebbe8L;
		expectedResults1[32][2] = 0xbfe4530ede10f077L;
		expectedResults1[32][3] = 0xbff3752e4ecbfa28L;
		expectedResults1[32][4] = 0x7ff8000000000000L;
		expectedResults1[32][5] = 0x7ff8000000000000L;
		expectedResults1[32][6] = 0xbff921fb54442d18L;
		expectedResults1[32][7] = 0x0L;
		expectedResults1[32][8] = 0x7ff8000000000000L;
		expectedResults1[33][0] = 0xfc98000000000000L;
		expectedResults1[33][1] = 0x3fefd028112e4d1aL;
		expectedResults1[33][2] = 0xbfbba08e22038be8L;
		expectedResults1[33][3] = 0xc0226c9b73e8b50eL;
		expectedResults1[33][4] = 0x7ff8000000000000L;
		expectedResults1[33][5] = 0x7ff8000000000000L;
		expectedResults1[33][6] = 0xbff921fb54442d18L;
		expectedResults1[33][7] = 0x0L;
		expectedResults1[33][8] = 0x7ff8000000000000L;
		expectedResults1[34][0] = 0x7fb9999999999993L;
		expectedResults1[34][1] = 0x3fec808dbadb4253L;
		expectedResults1[34][2] = 0x3fdd18444457902eL;
		expectedResults1[34][3] = 0x3fff59234f7457d7L;
		expectedResults1[34][4] = 0x7ff8000000000000L;
		expectedResults1[34][5] = 0x7ff8000000000000L;
		expectedResults1[34][6] = 0x3ff921fb54442d18L;
		expectedResults1[34][7] = 0x7ff0000000000000L;
		expectedResults1[34][8] = 0x40861bd74d3e849aL;
		expectedResults1[35][0] = 0x7fc9999999999996L;
		expectedResults1[35][1] = 0x3fdfbbeb47f987f9L;
		expectedResults1[35][2] = 0x3febca0658cd0708L;
		expectedResults1[35][3] = 0x3fe24575faaf88ddL;
		expectedResults1[35][4] = 0x7ff8000000000000L;
		expectedResults1[35][5] = 0x7ff8000000000000L;
		expectedResults1[35][6] = 0x3ff921fb54442d18L;
		expectedResults1[35][7] = 0x7ff0000000000000L;
		expectedResults1[35][8] = 0x40862162ddfe4329L;
		expectedResults1[36][0] = 0x7fd3333333333331L;
		expectedResults1[36][1] = 0xbfee85d748ca8a44L;
		expectedResults1[36][2] = 0xbfd33878098d10e2L;
		expectedResults1[36][3] = 0x4009688fbe930b1cL;
		expectedResults1[36][4] = 0x7ff8000000000000L;
		expectedResults1[36][5] = 0x7ff8000000000000L;
		expectedResults1[36][6] = 0x3ff921fb54442d18L;
		expectedResults1[36][7] = 0x7ff0000000000000L;
		expectedResults1[36][8] = 0x408624a1427bdb1cL;
		expectedResults1[37][0] = 0x7fd9999999999997L;
		expectedResults1[37][1] = 0x3fececfaa88c329cL;
		expectedResults1[37][2] = 0xbfdb5f2e7479d432L;
		expectedResults1[37][3] = 0xc000e887a7fc8cf6L;
		expectedResults1[37][4] = 0x7ff8000000000000L;
		expectedResults1[37][5] = 0x7ff8000000000000L;
		expectedResults1[37][6] = 0x3ff921fb54442d18L;
		expectedResults1[37][7] = 0x7ff0000000000000L;
		expectedResults1[37][8] = 0x408626ee6ebe01b7L;
		expectedResults1[38][0] = 0x7fdffffffffffffdL;
		expectedResults1[38][1] = 0xbfd7da2c35379e4bL;
		expectedResults1[38][2] = 0x3fedb1cf920df86bL;
		expectedResults1[38][3] = 0xbfd9b43e5977046eL;
		expectedResults1[38][4] = 0x7ff8000000000000L;
		expectedResults1[38][5] = 0x7ff8000000000000L;
		expectedResults1[38][6] = 0x3ff921fb54442d18L;
		expectedResults1[38][7] = 0x7ff0000000000000L;
		expectedResults1[38][8] = 0x408628b76e3a7b61L;
		expectedResults1[39][0] = 0x7fe3333333333332L;
		expectedResults1[39][1] = 0xbfef2717570b159dL;
		expectedResults1[39][2] = 0xbfcd42908438f5f2L;
		expectedResults1[39][3] = 0x401108f3112492c0L;
		expectedResults1[39][4] = 0x7ff8000000000000L;
		expectedResults1[39][5] = 0x7ff8000000000000L;
		expectedResults1[39][6] = 0x3ff921fb54442d18L;
		expectedResults1[39][7] = 0x7ff0000000000000L;
		expectedResults1[39][8] = 0x40862a2cd33b99aaL;
		expectedResults1[40][0] = 0x7fe6666666666665L;
		expectedResults1[40][1] = 0x3febd4149f80cedeL;
		expectedResults1[40][2] = 0xbfdf98983042cf30L;
		expectedResults1[40][3] = 0xbffc2f27d67b108dL;
		expectedResults1[40][4] = 0x7ff8000000000000L;
		expectedResults1[40][5] = 0x7ff8000000000000L;
		expectedResults1[40][6] = 0x3ff921fb54442d18L;
		expectedResults1[40][7] = 0x7ff0000000000000L;
		expectedResults1[40][8] = 0x40862b6886959dc7L;
		expectedResults1[41][0] = 0x7fe9999999999998L;
		expectedResults1[41][1] = 0xbfd35f39cdd75be3L;
		expectedResults1[41][2] = 0x3fee7fb68dcfa7caL;
		expectedResults1[41][3] = 0xbfd453503701f38dL;
		expectedResults1[41][4] = 0x7ff8000000000000L;
		expectedResults1[41][5] = 0x7ff8000000000000L;
		expectedResults1[41][6] = 0x3ff921fb54442d18L;
		expectedResults1[41][7] = 0x7ff0000000000000L;
		expectedResults1[41][8] = 0x40862c79ff7dc046L;
		expectedResults1[42][0] = 0x7feccccccccccccbL;
		expectedResults1[42][1] = 0xbfdb3a6aa73d718cL;
		expectedResults1[42][2] = 0xbfecf5a6113d9dceL;
		expectedResults1[42][3] = 0x3fde163bc277c5faL;
		expectedResults1[42][4] = 0x7ff8000000000000L;
		expectedResults1[42][5] = 0x7ff8000000000000L;
		expectedResults1[42][6] = 0x3ff921fb54442d18L;
		expectedResults1[42][7] = 0x7ff0000000000000L;
		expectedResults1[42][8] = 0x40862d6b37b9319dL;
		expectedResults1[43][0] = 0x7feffffffffffffeL;
		expectedResults1[43][1] = 0x3fedaa3677c6ee8aL;
		expectedResults1[43][2] = 0x3fd7ffdfb4c53090L;
		expectedResults1[43][3] = 0x4003c6e9970f78b8L;
		expectedResults1[43][4] = 0x7ff8000000000000L;
		expectedResults1[43][5] = 0x7ff8000000000000L;
		expectedResults1[43][6] = 0x3ff921fb54442d18L;
		expectedResults1[43][7] = 0x7ff0000000000000L;
		expectedResults1[43][8] = 0x40862e42fefa39efL;
		expectedResults1[44][0] = 0xc32fc00000000000L;
		expectedResults1[44][1] = 0xbfee9ef13cac7ae5L;
		expectedResults1[44][2] = 0x3fd296162def6e19L;
		expectedResults1[44][3] = 0xc00a5c27c4d7c6b2L;
		expectedResults1[44][4] = 0x7ff8000000000000L;
		expectedResults1[44][5] = 0x7ff8000000000000L;
		expectedResults1[44][6] = 0xbff921fb54442d17L;
		expectedResults1[44][7] = 0x0L;
		expectedResults1[44][8] = 0x7ff8000000000000L;
		expectedResults1[45][0] = 0x7fefffffffffffffL;
		expectedResults1[45][1] = 0x3f7452fc98b34e97L;
		expectedResults1[45][2] = 0xbfefffe62ecfab75L;
		expectedResults1[45][3] = 0xbf74530cfe729484L;
		expectedResults1[45][4] = 0x7ff8000000000000L;
		expectedResults1[45][5] = 0x7ff8000000000000L;
		expectedResults1[45][6] = 0x3ff921fb54442d18L;
		expectedResults1[45][7] = 0x7ff0000000000000L;
		expectedResults1[45][8] = 0x40862e42fefa39efL;
		expectedResults1[46][0] = 0x2L;
		expectedResults1[46][1] = 0x2L;
		expectedResults1[46][2] = 0x3ff0000000000000L;
		expectedResults1[46][3] = 0x2L;
		expectedResults1[46][4] = 0x2L;
		expectedResults1[46][5] = 0x3ff921fb54442d18L;
		expectedResults1[46][6] = 0x2L;
		expectedResults1[46][7] = 0x3ff0000000000000L;
		expectedResults1[46][8] = 0xc0873df9b3adb335L;
		expectedResults1[47][0] = 0x9L;
		expectedResults1[47][1] = 0x9L;
		expectedResults1[47][2] = 0x3ff0000000000000L;
		expectedResults1[47][3] = 0x9L;
		expectedResults1[47][4] = 0x9L;
		expectedResults1[47][5] = 0x3ff921fb54442d18L;
		expectedResults1[47][6] = 0x9L;
		expectedResults1[47][7] = 0x3ff0000000000000L;
		expectedResults1[47][8] = 0xc08731f159f2c4c0L;
		expectedResults1[48][0] = 0x4000000000000L;
		expectedResults1[48][1] = 0x4000000000000L;
		expectedResults1[48][2] = 0x3ff0000000000000L;
		expectedResults1[48][3] = 0x4000000000000L;
		expectedResults1[48][4] = 0x4000000000000L;
		expectedResults1[48][5] = 0x3ff921fb54442d18L;
		expectedResults1[48][6] = 0x4000000000000L;
		expectedResults1[48][7] = 0x3ff0000000000000L;
		expectedResults1[48][8] = 0xc0862e42fefa39efL;
		expectedResults1[49][0] = 0x8000000000000L;
		expectedResults1[49][1] = 0x8000000000000L;
		expectedResults1[49][2] = 0x3ff0000000000000L;
		expectedResults1[49][3] = 0x8000000000000L;
		expectedResults1[49][4] = 0x8000000000000L;
		expectedResults1[49][5] = 0x3ff921fb54442d18L;
		expectedResults1[49][6] = 0x8000000000000L;
		expectedResults1[49][7] = 0x3ff0000000000000L;
		expectedResults1[49][8] = 0xc08628b76e3a7b61L;
		expectedResults1[50][0] = 0xffffffffffffeL;
		expectedResults1[50][1] = 0xffffffffffffeL;
		expectedResults1[50][2] = 0x3ff0000000000000L;
		expectedResults1[50][3] = 0xffffffffffffeL;
		expectedResults1[50][4] = 0xffffffffffffeL;
		expectedResults1[50][5] = 0x3ff921fb54442d18L;
		expectedResults1[50][6] = 0xffffffffffffeL;
		expectedResults1[50][7] = 0x3ff0000000000000L;
		expectedResults1[50][8] = 0xc086232bdd7abcd2L;
		expectedResults1[51][0] = 0xfffffffffffffL;
		expectedResults1[51][1] = 0xfffffffffffffL;
		expectedResults1[51][2] = 0x3ff0000000000000L;
		expectedResults1[51][3] = 0xfffffffffffffL;
		expectedResults1[51][4] = 0xfffffffffffffL;
		expectedResults1[51][5] = 0x3ff921fb54442d18L;
		expectedResults1[51][6] = 0xfffffffffffffL;
		expectedResults1[51][7] = 0x3ff0000000000000L;
		expectedResults1[51][8] = 0xc086232bdd7abcd2L;
		expectedResults1[52][0] = 0x8000000000000002L;
		expectedResults1[52][1] = 0x8000000000000002L;
		expectedResults1[52][2] = 0x3ff0000000000000L;
		expectedResults1[52][3] = 0x8000000000000002L;
		expectedResults1[52][4] = 0x8000000000000002L;
		expectedResults1[52][5] = 0x3ff921fb54442d18L;
		expectedResults1[52][6] = 0x8000000000000002L;
		expectedResults1[52][7] = 0x3ff0000000000000L;
		expectedResults1[52][8] = 0x7ff8000000000000L;
		expectedResults1[53][0] = 0x8000000000000009L;
		expectedResults1[53][1] = 0x8000000000000009L;
		expectedResults1[53][2] = 0x3ff0000000000000L;
		expectedResults1[53][3] = 0x8000000000000009L;
		expectedResults1[53][4] = 0x8000000000000009L;
		expectedResults1[53][5] = 0x3ff921fb54442d18L;
		expectedResults1[53][6] = 0x8000000000000009L;
		expectedResults1[53][7] = 0x3ff0000000000000L;
		expectedResults1[53][8] = 0x7ff8000000000000L;
		expectedResults1[54][0] = 0x8004000000000000L;
		expectedResults1[54][1] = 0x8004000000000000L;
		expectedResults1[54][2] = 0x3ff0000000000000L;
		expectedResults1[54][3] = 0x8004000000000000L;
		expectedResults1[54][4] = 0x8004000000000000L;
		expectedResults1[54][5] = 0x3ff921fb54442d18L;
		expectedResults1[54][6] = 0x8004000000000000L;
		expectedResults1[54][7] = 0x3ff0000000000000L;
		expectedResults1[54][8] = 0x7ff8000000000000L;
		expectedResults1[55][0] = 0x8008000000000000L;
		expectedResults1[55][1] = 0x8008000000000000L;
		expectedResults1[55][2] = 0x3ff0000000000000L;
		expectedResults1[55][3] = 0x8008000000000000L;
		expectedResults1[55][4] = 0x8008000000000000L;
		expectedResults1[55][5] = 0x3ff921fb54442d18L;
		expectedResults1[55][6] = 0x8008000000000000L;
		expectedResults1[55][7] = 0x3ff0000000000000L;
		expectedResults1[55][8] = 0x7ff8000000000000L;
		expectedResults1[56][0] = 0x800ffffffffffffeL;
		expectedResults1[56][1] = 0x800ffffffffffffeL;
		expectedResults1[56][2] = 0x3ff0000000000000L;
		expectedResults1[56][3] = 0x800ffffffffffffeL;
		expectedResults1[56][4] = 0x800ffffffffffffeL;
		expectedResults1[56][5] = 0x3ff921fb54442d18L;
		expectedResults1[56][6] = 0x800ffffffffffffeL;
		expectedResults1[56][7] = 0x3ff0000000000000L;
		expectedResults1[56][8] = 0x7ff8000000000000L;
		expectedResults1[57][0] = 0x800fffffffffffffL;
		expectedResults1[57][1] = 0x800fffffffffffffL;
		expectedResults1[57][2] = 0x3ff0000000000000L;
		expectedResults1[57][3] = 0x800fffffffffffffL;
		expectedResults1[57][4] = 0x800fffffffffffffL;
		expectedResults1[57][5] = 0x3ff921fb54442d18L;
		expectedResults1[57][6] = 0x800fffffffffffffL;
		expectedResults1[57][7] = 0x3ff0000000000000L;
		expectedResults1[57][8] = 0x7ff8000000000000L;
		expectedResults1[58][0] = 0x408fb4e147ae149bL;
		expectedResults1[58][1] = 0x3fbfc56e12d2c0a2L;
		expectedResults1[58][2] = 0xbfefc0aac510014cL;
		expectedResults1[58][3] = 0xbfc0026666e05633L;
		expectedResults1[58][4] = 0x7ff8000000000000L;
		expectedResults1[58][5] = 0x7ff8000000000000L;
		expectedResults1[58][6] = 0x3ff91df1da43d3f6L;
		expectedResults1[58][7] = 0x7ff0000000000000L;
		expectedResults1[58][8] = 0x401bb064d0ce72b3L;
		expectedResults1[59][0] = 0x4090961eb851eb95L;
		expectedResults1[59][1] = 0xbfd4a30694203fa3L;
		expectedResults1[59][2] = 0x3fee4a6f8d99e3abL;
		expectedResults1[59][3] = 0xbfd5cd225a4c53e0L;
		expectedResults1[59][4] = 0x7ff8000000000000L;
		expectedResults1[59][5] = 0x7ff8000000000000L;
		expectedResults1[59][6] = 0x3ff91e1f88585ba6L;
		expectedResults1[59][7] = 0x7ff0000000000000L;
		expectedResults1[59][8] = 0x401bdeaf8ca268b6L;
		expectedResults1[60][0] = 0xc086300000000000L;
		expectedResults1[60][1] = 0xbf0f9bd0303f6fafL;
		expectedResults1[60][2] = 0x3fefffffff063930L;
		expectedResults1[60][3] = 0xbf0f9bd03136287dL;
		expectedResults1[60][4] = 0x7ff8000000000000L;
		expectedResults1[60][5] = 0x7ff8000000000000L;
		expectedResults1[60][6] = 0xbff91c367668c617L;
		expectedResults1[60][7] = 0x33802fd28b3c3L;
		expectedResults1[60][8] = 0x7ff8000000000000L;
		expectedResults1[61][0] = 0xc08627eb851eb852L;
		expectedResults1[61][1] = 0x3feb18fbd908b222L;
		expectedResults1[61][2] = 0x3fe1056bd7a48fb1L;
		expectedResults1[61][3] = 0x3ff978cd909624bbL;
		expectedResults1[61][4] = 0x7ff8000000000000L;
		expectedResults1[61][5] = 0x7ff8000000000000L;
		expectedResults1[61][6] = 0xbff91c345bd0938eL;
		expectedResults1[61][7] = 0x8d668444ff378L;
		expectedResults1[61][8] = 0x7ff8000000000000L;
		expectedResults1[62][0] = 0x4051ef5c28f5c132L;
		expectedResults1[62][1] = 0x3fdf9cefd9928309L;
		expectedResults1[62][2] = 0xbfebd2d8f8269e40L;
		expectedResults1[62][3] = 0xbfe22dd9cd09c84eL;
		expectedResults1[62][4] = 0x7ff8000000000000L;
		expectedResults1[62][5] = 0x7ff8000000000000L;
		expectedResults1[62][6] = 0x3ff8e8e3f05ea8bbL;
		expectedResults1[62][7] = 0x46669c5f8bfbd36eL;
		expectedResults1[62][8] = 0x4011179a04c7a392L;
		expectedResults1[63][0] = 0x3ff0a2b23def12b7L;
		expectedResults1[63][1] = 0x3feb97a83d246884L;
		expectedResults1[63][2] = 0x3fe034edcdd8ceadL;
		expectedResults1[63][3] = 0x3ffb3d8b5a2991b7L;
		expectedResults1[63][4] = 0x7ff8000000000000L;
		expectedResults1[63][5] = 0x7ff8000000000000L;
		expectedResults1[63][6] = 0x3fe9c17d541a4926L;
		expectedResults1[63][7] = 0x4006a09e64a8def6L;
		expectedResults1[63][8] = 0x3fa3f18a5f14e7cbL;
		expectedResults1[64][0] = 0x3ff0a2b23e33cae7L;
		expectedResults1[64][1] = 0x3feb97a83d6a0408L;
		expectedResults1[64][2] = 0x3fe034edcd624c7fL;
		expectedResults1[64][3] = 0x3ffb3d8b5b357a7fL;
		expectedResults1[64][4] = 0x7ff8000000000000L;
		expectedResults1[64][5] = 0x7ff8000000000000L;
		expectedResults1[64][6] = 0x3fe9c17d545c546eL;
		expectedResults1[64][7] = 0x4006a09e650a0e12L;
		expectedResults1[64][8] = 0x3fa3f18a6757eb3eL;
		expectedResults1[65][0] = 0x3ff0a2b23f8b63d7L;
		expectedResults1[65][1] = 0x3feb97a83ec60d9bL;
		expectedResults1[65][2] = 0x3fe034edcb11c19aL;
		expectedResults1[65][3] = 0x3ffb3d8b60710665L;
		expectedResults1[65][4] = 0x7ff8000000000000L;
		expectedResults1[65][5] = 0x7ff8000000000000L;
		expectedResults1[65][6] = 0x3fe9c17d55a68cd4L;
		expectedResults1[65][7] = 0x4006a09e66eff99eL;
		expectedResults1[65][8] = 0x3fa3f18a90a6fc7eL;
		expectedResults1[66][0] = 0xbfefffffffffffffL;
		expectedResults1[66][1] = 0xbfeaed548f090cedL;
		expectedResults1[66][2] = 0x3fe14a280fb5068cL;
		expectedResults1[66][3] = 0xbff8eb245cbee3a4L;
		expectedResults1[66][4] = 0xbff921fb50442d18L;
		expectedResults1[66][5] = 0x400921fb52442d18L;
		expectedResults1[66][6] = 0xbfe921fb54442d18L;
		expectedResults1[66][7] = 0x3fd78b56362cef38L;
		expectedResults1[66][8] = 0x7ff8000000000000L;
		expectedResults1[67][0] = 0x3feffffffffffff0L;
		expectedResults1[67][1] = 0x3feaed548f090ce5L;
		expectedResults1[67][2] = 0x3fe14a280fb50699L;
		expectedResults1[67][3] = 0x3ff8eb245cbee38aL;
		expectedResults1[67][4] = 0x3ff921fb44442d18L;
		expectedResults1[67][5] = 0x3e70000000000001L;
		expectedResults1[67][6] = 0x3fe921fb54442d10L;
		expectedResults1[67][7] = 0x4005bf0a8b14575eL;
		expectedResults1[67][8] = 0xbce0000000000004L;
		expectedResults1[68][0] = 0x3fefffffffffffffL;
		expectedResults1[68][1] = 0x3feaed548f090cedL;
		expectedResults1[68][2] = 0x3fe14a280fb5068cL;
		expectedResults1[68][3] = 0x3ff8eb245cbee3a4L;
		expectedResults1[68][4] = 0x3ff921fb50442d18L;
		expectedResults1[68][5] = 0x3e50000000000000L;
		expectedResults1[68][6] = 0x3fe921fb54442d18L;
		expectedResults1[68][7] = 0x4005bf0a8b145769L;
		expectedResults1[68][8] = 0xbca0000000000000L;
		expectedResults1[69][0] = 0x3feffffffffffe14L;
		expectedResults1[69][1] = 0x3feaed548f090be4L;
		expectedResults1[69][2] = 0x3fe14a280fb5082aL;
		expectedResults1[69][3] = 0x3ff8eb245cbee05bL;
		expectedResults1[69][4] = 0x3ff921fafb8ac1e4L;
		expectedResults1[69][5] = 0x3e962e5acd0c3edaL;
		expectedResults1[69][6] = 0x3fe921fb54442c22L;
		expectedResults1[69][7] = 0x4005bf0a8b14561bL;
		expectedResults1[69][8] = 0xbd2ec000000000ecL;
		expectedResults1[70][0] = 0xc0152c38c4e8043aL;
		expectedResults1[70][1] = 0x3feac0b98d964236L;
		expectedResults1[70][2] = 0x3fe18ede09db088bL;
		expectedResults1[70][3] = 0x3ff860fadcc58f2cL;
		expectedResults1[70][4] = 0x7ff8000000000000L;
		expectedResults1[70][5] = 0x7ff8000000000000L;
		expectedResults1[70][6] = 0xbff6252bb146b6ddL;
		expectedResults1[70][7] = 0x3f7495db1e712c5eL;
		expectedResults1[70][8] = 0x7ff8000000000000L;
		expectedResults1[71][0] = 0xc014e48ab06d22f4L;
		expectedResults1[71][1] = 0x3febea560b650e20L;
		expectedResults1[71][2] = 0x3fdf49ae20e12edaL;
		expectedResults1[71][3] = 0x3ffc8d010d5ddbeaL;
		expectedResults1[71][4] = 0x7ff8000000000000L;
		expectedResults1[71][5] = 0x7ff8000000000000L;
		expectedResults1[71][6] = 0xbff61b297d3e0e68L;
		expectedResults1[71][7] = 0x3f7613f6aea6785fL;
		expectedResults1[71][8] = 0x7ff8000000000000L;
		expectedResults1[72][0] = 0xc014da4d3fc94beaL;
		expectedResults1[72][1] = 0x3fec1206cabc9f15L;
		expectedResults1[72][2] = 0x3fdeba5af96c7878L;
		expectedResults1[72][3] = 0x3ffd3b81eb0fc8f7L;
		expectedResults1[72][4] = 0x7ff8000000000000L;
		expectedResults1[72][5] = 0x7ff8000000000000L;
		expectedResults1[72][6] = 0xbff619b60a9dc919L;
		expectedResults1[72][7] = 0x3f764cc44bc2a102L;
		expectedResults1[72][8] = 0x7ff8000000000000L;
		expectedResults1[73][0] = 0xc010d00fcf2574f8L;
		expectedResults1[73][1] = 0x3febf0b4930167beL;
		expectedResults1[73][2] = 0xbfdf32e821ba51a5L;
		expectedResults1[73][3] = 0xbffca8609e09cda0L;
		expectedResults1[73][4] = 0x7ff8000000000000L;
		expectedResults1[73][5] = 0x7ff8000000000000L;
		expectedResults1[73][6] = 0xbff56544e5c56948L;
		expectedResults1[73][7] = 0x3f8e9cff789decc5L;
		expectedResults1[73][8] = 0x7ff8000000000000L;
		expectedResults1[74][0] = 0xc0014e34192c3200L;
		expectedResults1[74][1] = 0xbfea8c2889a3e92dL;
		expectedResults1[74][2] = 0xbfe1ddf55b6082faL;
		expectedResults1[74][3] = 0x3ff7c5fd6787b299L;
		expectedResults1[74][4] = 0x7ff8000000000000L;
		expectedResults1[74][5] = 0x7ff8000000000000L;
		expectedResults1[74][6] = 0xbff234550150dc91L;
		expectedResults1[74][7] = 0x3fbd6de93a9c0c88L;
		expectedResults1[74][8] = 0x7ff8000000000000L;
		expectedResults1[75][0] = 0x40124ea882c60fb8L;
		expectedResults1[75][1] = 0xbfefb4d44501f17cL;
		expectedResults1[75][2] = 0xbfc14ce6d059235aL;
		expectedResults1[75][3] = 0x401d52b8c7d83379L;
		expectedResults1[75][4] = 0x7ff8000000000000L;
		expectedResults1[75][5] = 0x7ff8000000000000L;
		expectedResults1[75][6] = 0x3ff5b0e20963c5ffL;
		expectedResults1[75][7] = 0x40584d12d7cab944L;
		expectedResults1[75][8] = 0x3ff8560787802903L;
		expectedResults1[76][0] = 0x401258e5f369e6c2L;
		expectedResults1[76][1] = 0xbfefbf7ed3268492L;
		expectedResults1[76][2] = 0xbfc008032dcabb25L;
		expectedResults1[76][3] = 0x401fafa0b72d5b9fL;
		expectedResults1[76][4] = 0x7ff8000000000000L;
		expectedResults1[76][5] = 0x7ff8000000000000L;
		expectedResults1[76][6] = 0x3ff5b2bed054172eL;
		expectedResults1[76][7] = 0x40588b98afecbafeL;
		expectedResults1[76][8] = 0x3ff85ef81736d8daL;
		expectedResults1[77][0] = 0x4014e8421c5fa942L;
		expectedResults1[77][1] = 0xbfebdbc103f8d2f5L;
		expectedResults1[77][2] = 0x3fdf7d803dae986eL;
		expectedResults1[77][3] = 0xbffc4f33cca5f788L;
		expectedResults1[77][4] = 0x7ff8000000000000L;
		expectedResults1[77][5] = 0x7ff8000000000000L;
		expectedResults1[77][6] = 0x3ff61baff67c4ec2L;
		expectedResults1[77][7] = 0x4067465c390d728eL;
		expectedResults1[77][8] = 0x3ffa75f922ab607bL;
		expectedResults1[78][0] = 0xbff80d0c0962a17eL;
		expectedResults1[78][1] = 0xbfefed4885ccf313L;
		expectedResults1[78][2] = 0x3fb14b94a9f94002L;
		expectedResults1[78][3] = 0xc02d89300597e0f9L;
		expectedResults1[78][4] = 0x7ff8000000000000L;
		expectedResults1[78][5] = 0x7ff8000000000000L;
		expectedResults1[78][6] = 0xbfef7b103625ad40L;
		expectedResults1[78][7] = 0x3fcc7846d421440aL;
		expectedResults1[78][8] = 0x7ff8000000000000L;
		expectedResults1[79][0] = 0xbff03601cbf1fda5L;
		expectedResults1[79][1] = 0xbfeb2716f3222754L;
		expectedResults1[79][2] = 0x3fe0eee23a526324L;
		expectedResults1[79][3] = 0xbff9a808aef7a0d2L;
		expectedResults1[79][4] = 0x7ff8000000000000L;
		expectedResults1[79][5] = 0x7ff8000000000000L;
		expectedResults1[79][6] = 0xbfe957a260b2a15bL;
		expectedResults1[79][7] = 0x3fd73c62d877a092L;
		expectedResults1[79][8] = 0x7ff8000000000000L;
		expectedResults1[80][0] = 0x3ff111ac4888e3a9L;
		expectedResults1[80][1] = 0x3fec057633227726L;
		expectedResults1[80][2] = 0x3fdee81844084bd9L;
		expectedResults1[80][3] = 0x3ffd033c848e8cdfL;
		expectedResults1[80][4] = 0x7ff8000000000000L;
		expectedResults1[80][5] = 0x7ff8000000000000L;
		expectedResults1[80][6] = 0x3fea2ab72541dbd3L;
		expectedResults1[80][7] = 0x40073fb57efe0350L;
		expectedResults1[80][8] = 0x3fb08eb0cc49256fL;
		expectedResults1[81][0] = 0x3ff911ac4888e3abL;
		expectedResults1[81][1] = 0x3fefffef606375ffL;
		expectedResults1[81][2] = 0x3f704f08e8548708L;
		expectedResults1[81][3] = 0x406f64dbe84e30aaL;
		expectedResults1[81][4] = 0x7ff8000000000000L;
		expectedResults1[81][5] = 0x7ff8000000000000L;
		expectedResults1[81][6] = 0x3ff00b3345490851L;
		expectedResults1[81][7] = 0x40132a5ac638654eL;
		expectedResults1[81][8] = 0x3fdcbd25fba031e5L;
		expectedResults1[82][0] = 0x3fefced91687291cL;
		expectedResults1[82][1] = 0x3fead2a6423fafa6L;
		expectedResults1[82][2] = 0x3fe1736fc1ec49f3L;
		expectedResults1[82][3] = 0x3ff897bc2141e404L;
		expectedResults1[82][4] = 0x3ff7611012ab59b1L;
		expectedResults1[82][5] = 0x3fbc0eb4198d366eL;
		expectedResults1[82][6] = 0x3fe90954f6074d05L;
		expectedResults1[82][7] = 0x40059dbd2b7284fdL;
		expectedResults1[82][8] = 0xbf78a667fd2d02c9L;
		expectedResults1[83][0] = 0x3fefe76c8b439398L;
		expectedResults1[83][1] = 0x3feae00555a42845L;
		expectedResults1[83][2] = 0x3fe15ed1083bd0b0L;
		expectedResults1[83][3] = 0x3ff8c13eb13ed3d6L;
		expectedResults1[83][4] = 0x3ff7e4a0a982f42dL;
		expectedResults1[83][5] = 0x3fb3d5aaac138eb0L;
		expectedResults1[83][6] = 0x3fe915ace0bb1511L;
		expectedResults1[83][7] = 0x4005ae5d766494b2L;
		expectedResults1[83][8] = 0xbf689ce97f745bbbL;
		expectedResults1[84][0] = 0x3feff3b645a1c8d6L;
		expectedResults1[84][1] = 0x3feae6aeee145f6fL;
		expectedResults1[84][2] = 0x3fe1547dd310618eL;
		expectedResults1[84][3] = 0x3ff8d6250a55e733L;
		expectedResults1[84][4] = 0x3ff8419b46585385L;
		expectedResults1[84][5] = 0x3fac0c01bd7b326aL;
		expectedResults1[84][6] = 0x3fe91bd548f1009dL;
		expectedResults1[84][7] = 0x4005b6b266e78c61L;
		expectedResults1[84][8] = 0xbf58982de7a989d5L;
		expectedResults1[85][0] = 0x3feff7ced9168540L;
		expectedResults1[85][1] = 0x3feae8e6949c8142L;
		expectedResults1[85][2] = 0x3fe1510c2fe3ff44L;
		expectedResults1[85][3] = 0x3ff8dd2207973883L;
		expectedResults1[85][4] = 0x3ff86ac9ad18e290L;
		expectedResults1[85][5] = 0x3fa6e634e56950fbL;
		expectedResults1[85][6] = 0x3fe91de23a8c3eabL;
		expectedResults1[85][7] = 0x4005b97a17d41045L;
		expectedResults1[85][8] = 0xbf5064670d9b726fL;
		expectedResults1[86][0] = 0xc014eec82110f9feL;
		expectedResults1[86][1] = 0x3febc1ee5a36f25dL;
		expectedResults1[86][2] = 0x3fdfd8343c886a7dL;
		expectedResults1[86][3] = 0x3ffbe49e982fe51cL;
		expectedResults1[86][4] = 0x7ff8000000000000L;
		expectedResults1[86][5] = 0x7ff8000000000000L;
		expectedResults1[86][6] = 0xbff61c9b9147d9d9L;
		expectedResults1[86][7] = 0x3f75dbb9c28ec00aL;
		expectedResults1[86][8] = 0x7ff8000000000000L;
		expectedResults1[87][0] = 0x400d7e98b3a09a7bL;
		expectedResults1[87][1] = 0xbfe0986fb2d58dbbL;
		expectedResults1[87][2] = 0xbfeb5c4276679031L;
		expectedResults1[87][3] = 0x3fe368dc56d54bc4L;
		expectedResults1[87][4] = 0x7ff8000000000000L;
		expectedResults1[87][5] = 0x7ff8000000000000L;
		expectedResults1[87][6] = 0x3ff4e518ff4ce0fdL;
		expectedResults1[87][7] = 0x4043f57077d62aadL;
		expectedResults1[87][8] = 0x3ff4e04f04dc69c5L;
		expectedResults1[88][0] = 0xc00295e22da71340L;
		expectedResults1[88][1] = 0xbfe75ca359ff3aacL;
		expectedResults1[88][2] = 0xbfe5de49cebf2ba3L;
		expectedResults1[88][3] = 0x3ff117bf3f685075L;
		expectedResults1[88][4] = 0x7ff8000000000000L;
		expectedResults1[88][5] = 0x7ff8000000000000L;
		expectedResults1[88][6] = 0xbff2a1123ed9dc11L;
		expectedResults1[88][7] = 0x3fb913f9baea74fbL;
		expectedResults1[88][8] = 0x7ff8000000000000L;
		expectedResults1[89][0] = 0x3ffff2f3f69d5e8eL;
		expectedResults1[89][1] = 0x3fed23c92140e547L;
		expectedResults1[89][2] = 0xbfda72a928eec7b5L;
		expectedResults1[89][3] = 0xc001a0e5275f25b2L;
		expectedResults1[89][4] = 0x7ff8000000000000L;
		expectedResults1[89][5] = 0x7ff8000000000000L;
		expectedResults1[89][6] = 0x3ff1b444b6e6c70bL;
		expectedResults1[89][7] = 0x401d76549bcb1836L;
		expectedResults1[89][8] = 0x3fe621344bf83881L;
		expectedResults1[90][0] = 0x401058e5f369e6ceL;
		expectedResults1[90][1] = 0xbfe9f0a7f99762b9L;
		expectedResults1[90][2] = 0xbfe2bcee697ad940L;
		expectedResults1[90][3] = 0x3ff626517ff7121fL;
		expectedResults1[90][4] = 0x7ff8000000000000L;
		expectedResults1[90][5] = 0x7ff8000000000000L;
		expectedResults1[90][6] = 0x3ff54b0c28591dc2L;
		expectedResults1[90][7] = 0x404dc66345a519c2L;
		expectedResults1[90][8] = 0x3ff686357f5a898aL;
		expectedResults1[91][0] = 0xc00e81674c5f64e4L;
		expectedResults1[91][1] = 0x3fe3e956585240b2L;
		expectedResults1[91][2] = 0xbfe90cf4f9711550L;
		expectedResults1[91][3] = 0xbfe96f68c42f6d5aL;
		expectedResults1[91][4] = 0x7ff8000000000000L;
		expectedResults1[91][5] = 0x7ff8000000000000L;
		expectedResults1[91][6] = 0xbff5077869f71168L;
		expectedResults1[91][7] = 0x3f969b8c37115eb6L;
		expectedResults1[91][8] = 0x7ff8000000000000L;
		expectedResults1[92][0] = 0x3fc6978d4fdf3b6aL;
		expectedResults1[92][1] = 0x3fc67991fedafec7L;
		expectedResults1[92][2] = 0x3fef80bb35eb18c0L;
		expectedResults1[92][3] = 0x3fc6d45de6fab71aL;
		expectedResults1[92][4] = 0x3fc6b602663773a0L;
		expectedResults1[92][5] = 0x3ff64b3b077d3ea4L;
		expectedResults1[92][6] = 0x3fc65c97e08cdea8L;
		expectedResults1[92][7] = 0x3ff316ab445ef40aL;
		expectedResults1[92][8] = 0xbffbc03e49c43144L;
		expectedResults1[93][0] = 0x3fd126e978d4fdf8L;
		expectedResults1[93][1] = 0x3fd0f289cdafc67cL;
		expectedResults1[93][2] = 0x3feedb90c5c2370dL;
		expectedResults1[93][3] = 0x3fd193259b3f7f71L;
		expectedResults1[93][4] = 0x3fd15d3fd6ef496cL;
		expectedResults1[93][5] = 0x3ff4caab5e885abeL;
		expectedResults1[93][6] = 0x3fd0c2190b4dad6cL;
		expectedResults1[93][7] = 0x3ff4eae4d5ae21efL;
		expectedResults1[93][8] = 0xbff5117ba2a57fd8L;
		expectedResults1[94][0] = 0x3fd2978d4fdf3b69L;
		expectedResults1[94][1] = 0x3fd254e3fe9f7cbeL;
		expectedResults1[94][2] = 0x3feea8c2f75873d8L;
		expectedResults1[94][3] = 0x3fd3221e306ccca5L;
		expectedResults1[94][4] = 0x3fd2dd2c4be44395L;
		expectedResults1[94][5] = 0x3ff46ab0414b1c33L;
		expectedResults1[94][6] = 0x3fd2180fa38984a5L;
		expectedResults1[94][7] = 0x3ff564bea3454ac0L;
		expectedResults1[94][8] = 0xbff3c74701ff2f17L;
		expectedResults1[95][0] = 0x3fd6a7ef9db22d14L;
		expectedResults1[95][1] = 0x3fd62f8e28141e03L;
		expectedResults1[95][2] = 0x3fee040b2d038764L;
		expectedResults1[95][3] = 0x3fd7a70116ae58b5L;
		expectedResults1[95][4] = 0x3fd728764e48e613L;
		expectedResults1[95][5] = 0x3ff357ddc0b1f394L;
		expectedResults1[95][6] = 0x3fd5c66427cc8a39L;
		expectedResults1[95][7] = 0x3ff6cbcc182bccd1L;
		expectedResults1[95][8] = 0xbff09d8684f93898L;
		expectedResults1[96][0] = 0x3fd7c6a7ef9db233L;
		expectedResults1[96][1] = 0x3fd73b9d0a6bf5e1L;
		expectedResults1[96][2] = 0x3fedd12c6a1c4143L;
		expectedResults1[96][3] = 0x3fd8ef0a2883e8ceL;
		expectedResults1[96][4] = 0x3fd85c254a4baffcL;
		expectedResults1[96][5] = 0x3ff30af201b14119L;
		expectedResults1[96][6] = 0x3fd6c3c4473a99d1L;
		expectedResults1[96][7] = 0x3ff732d291d3829fL;
		expectedResults1[96][8] = 0xbfefafc55cfeb74aL;
		expectedResults1[97][0] = 0x3fd7df3b645a1cb2L;
		expectedResults1[97][1] = 0x3fd752819a53e4e9L;
		expectedResults1[97][2] = 0x3fedccb446587de6L;
		expectedResults1[97][3] = 0x3fd90b5cc686f62bL;
		expectedResults1[97][4] = 0x3fd8769fe8dfde2cL;
		expectedResults1[97][5] = 0x3ff304535a0c358eL;
		expectedResults1[97][6] = 0x3fd6d95a07aa94a4L;
		expectedResults1[97][7] = 0x3ff73bbcc9e9d92eL;
		expectedResults1[97][8] = 0xbfef8ec2c5ddc336L;
		expectedResults1[98][0] = 0x3fd820c49ba5e35aL;
		expectedResults1[98][1] = 0x3fd78f7ceb77da0fL;
		expectedResults1[98][2] = 0x3fedc0b3c29a6649L;
		expectedResults1[98][3] = 0x3fd9570d85d305fbL;
		expectedResults1[98][4] = 0x3fd8bd51d868d08aL;
		expectedResults1[98][5] = 0x3ff2f2a6de29f8f6L;
		expectedResults1[98][6] = 0x3fd712ced0372601L;
		expectedResults1[98][7] = 0x3ff7539378172670L;
		expectedResults1[98][8] = 0xbfef3760f081d33bL;
		expectedResults1[99][0] = 0x3fd9810624dd2f21L;
		expectedResults1[99][1] = 0x3fd8d59642c1bad8L;
		expectedResults1[99][2] = 0x3fed7e1bbc8255edL;
		expectedResults1[99][3] = 0x3fdaf217e75268b5L;
		expectedResults1[99][4] = 0x3fda3b7ca8b8996eL;
		expectedResults1[99][5] = 0x3ff2931c2a1606bdL;
		expectedResults1[99][6] = 0x3fd84504f26249faL;
		expectedResults1[99][7] = 0x3ff7d55ae54f49e6L;
		expectedResults1[99][8] = 0xbfed71080b22c5cbL;
		expectedResults1[100][0] = 0x3fdbdf3b645a1cb3L;
		expectedResults1[100][1] = 0x3fdaffd0f494e65dL;
		expectedResults1[100][2] = 0x3fed035a3453c121L;
		expectedResults1[100][3] = 0x3fddc763a9aac05cL;
		expectedResults1[100][4] = 0x3fdcd686b643233cL;
		expectedResults1[100][5] = 0x3ff1ec59a6b36449L;
		expectedResults1[100][6] = 0x3fda4968aaa7e55eL;
		expectedResults1[100][7] = 0x3ff8bb5564836c6aL;
		expectedResults1[100][8] = 0xbfea99af95737c3aL;
		expectedResults1[101][0] = 0x3fdc20c49ba5e35bL;
		expectedResults1[101][1] = 0x3fdb3b2dfa2274fcL;
		expectedResults1[101][2] = 0x3fecf57827196de0L;
		expectedResults1[101][3] = 0x3fde17434c6e73aaL;
		expectedResults1[101][4] = 0x3fdd1f684ace3146L;
		expectedResults1[101][5] = 0x3ff1da214190a0c7L;
		expectedResults1[101][6] = 0x3fda806a855ee4eaL;
		expectedResults1[101][7] = 0x3ff8d4b5a97df31fL;
		expectedResults1[101][8] = 0xbfea4ec97d3490b0L;
		expectedResults1[102][0] = 0x3fdc28f5c28f5c30L;
		expectedResults1[102][1] = 0x3fdb42979a223067L;
		expectedResults1[102][2] = 0x3fecf3b9c29100ceL;
		expectedResults1[102][3] = 0x3fde2144a61010c4L;
		expectedResults1[102][4] = 0x3fdd2887569581d2L;
		expectedResults1[102][5] = 0x3ff1d7d97e9ecca4L;
		expectedResults1[102][6] = 0x3fda8747d793c3e1L;
		expectedResults1[102][7] = 0x3ff8d7e3863c4f1fL;
		expectedResults1[102][8] = 0xbfea457901b95797L;
		expectedResults1[103][0] = 0x3fdcbc6a7ef9db2aL;
		expectedResults1[103][1] = 0x3fdbc7b804e683d3L;
		expectedResults1[103][2] = 0x3fecd405afa3d6deL;
		expectedResults1[103][3] = 0x3fded62d471a885aL;
		expectedResults1[103][4] = 0x3fddcd24554c5d69L;
		expectedResults1[103][5] = 0x3ff1aeb23ef115beL;
		expectedResults1[103][6] = 0x3fdb02689f44a796L;
		expectedResults1[103][7] = 0x3ff91162d87dab88L;
		expectedResults1[103][8] = 0xbfe99f9989272434L;
		expectedResults1[104][0] = 0x3fdf645a1cac0839L;
		expectedResults1[104][1] = 0x3fde25f62f3f372dL;
		expectedResults1[104][2] = 0x3fec3a24669b8b61L;
		expectedResults1[104][3] = 0x3fe116c746df318dL;
		expectedResults1[104][4] = 0x3fe067bd1f33f4bdL;
		expectedResults1[104][5] = 0x3ff0ee1cc4aa32baL;
		expectedResults1[104][6] = 0x3fdd2f69567bf2eeL;
		expectedResults1[104][7] = 0x3ffa214fa16518f2L;
		expectedResults1[104][8] = 0xbfe6cb6848bb828dL;
		expectedResults1[105][0] = 0x3fe0083126e978d8L;
		expectedResults1[105][1] = 0x3fdebd484bd8dcdfL;
		expectedResults1[105][2] = 0x3fec1139ad332b23L;
		expectedResults1[105][3] = 0x3fe185f3e353789eL;
		expectedResults1[105][4] = 0x3fe0cac89c39e9c6L;
		expectedResults1[105][5] = 0x3ff0bc9706273835L;
		expectedResults1[105][6] = 0x3fddb9811f32abbaL;
		expectedResults1[105][7] = 0x3ffa67eb3ab98d23L;
		expectedResults1[105][8] = 0xbfe61de4e1773bf2L;
		expectedResults1[106][0] = 0x3fe01cac083126eaL;
		expectedResults1[106][1] = 0x3fdee12f14647212L;
		expectedResults1[106][2] = 0x3fec075dc2b923faL;
		expectedResults1[106][3] = 0x3fe1a09c4b645b89L;
		expectedResults1[106][4] = 0x3fe0e277b15be34eL;
		expectedResults1[106][5] = 0x3ff0b0bf7b963b72L;
		expectedResults1[106][6] = 0x3fddda369f7b42ccL;
		expectedResults1[106][7] = 0x3ffa78d6fd6adc01L;
		expectedResults1[106][8] = 0xbfe5f51e12de7794L;
		expectedResults1[107][0] = 0x3fe0b4395810623cL;
		expectedResults1[107][1] = 0x3fdfe94df8bb4036L;
		expectedResults1[107][2] = 0x3febbd05060058edL;
		expectedResults1[107][3] = 0x3fe26834e3d9ed65L;
		expectedResults1[107][4] = 0x3fe192ff7b30de75L;
		expectedResults1[107][5] = 0x3ff0587b96abbddeL;
		expectedResults1[107][6] = 0x3fdeca3766bc184aL;
		expectedResults1[107][7] = 0x3ffaf760d280fdbaL;
		expectedResults1[107][8] = 0xbfe4cd84b399be90L;
		expectedResults1[108][0] = 0x3fe0bc6a7ef9db10L;
		expectedResults1[108][1] = 0x3fdff780a5ba8405L;
		expectedResults1[108][2] = 0x3febb8ee719660d2L;
		expectedResults1[108][3] = 0x3fe2731d8d5e19c0L;
		expectedResults1[108][4] = 0x3fe19c9b14fe0511L;
		expectedResults1[108][5] = 0x3ff053adc9c52a90L;
		expectedResults1[108][6] = 0x3fded7163421b71cL;
		expectedResults1[108][7] = 0x3ffafe48f8956d06L;
		expectedResults1[108][8] = 0xbfe4bdd703262d21L;
		expectedResults1[109][0] = 0x3fe1ba5e353f7cbcL;
		expectedResults1[109][1] = 0x3fe0d5c145f4290cL;
		expectedResults1[109][2] = 0x3feb36b298d1049dL;
		expectedResults1[109][3] = 0x3fe3cbc12155e7e4L;
		expectedResults1[109][4] = 0x3fe2ca06830c33f3L;
		expectedResults1[109][5] = 0x3fef79f0257c263eL;
		expectedResults1[109][6] = 0x3fe0306827ced1edL;
		expectedResults1[109][7] = 0x3ffbd7dbdd9260d7L;
		expectedResults1[109][8] = 0xbfe2e61e3ddf2b15L;
		expectedResults1[110][0] = 0x3fe1c6a7ef9db1faL;
		expectedResults1[110][1] = 0x3fe0e0333e314e45L;
		expectedResults1[110][2] = 0x3feb3039a52e8c9fL;
		expectedResults1[110][3] = 0x3fe3dcc2c05d3c63L;
		expectedResults1[110][4] = 0x3fe2d8cb5bc7bc1dL;
		expectedResults1[110][5] = 0x3fef6b2b4cc09e14L;
		expectedResults1[110][6] = 0x3fe039cd9c4cbc50L;
		expectedResults1[110][7] = 0x3ffbe28f0464cf18L;
		expectedResults1[110][8] = 0xbfe2cff7b49f8898L;
		expectedResults1[111][0] = 0x3fe1d2f1a9fbe738L;
		expectedResults1[111][1] = 0x3fe0eaa2b96462d8L;
		expectedResults1[111][2] = 0x3feb29bcaf386d5bL;
		expectedResults1[111][3] = 0x3fe3edcc7caf0434L;
		expectedResults1[111][4] = 0x3fe2e794c3a8f340L;
		expectedResults1[111][5] = 0x3fef5c61e4df66f1L;
		expectedResults1[111][6] = 0x3fe0433001104be7L;
		expectedResults1[111][7] = 0x3ffbed4647d6db41L;
		expectedResults1[111][8] = 0xbfe2b9e075f5f072L;
		expectedResults1[112][0] = 0x3fe1d70a3d70a3a2L;
		expectedResults1[112][1] = 0x3fe0ee1ca9f630d8L;
		expectedResults1[112][2] = 0x3feb279223fe1c84L;
		expectedResults1[112][3] = 0x3fe3f37c359e192bL;
		expectedResults1[112][4] = 0x3fe2ec839598c4dcL;
		expectedResults1[112][5] = 0x3fef577312ef9555L;
		expectedResults1[112][6] = 0x3fe046501f22e21eL;
		expectedResults1[112][7] = 0x3ffbf0d99dd804b8L;
		expectedResults1[112][8] = 0xbfe2b286c3564b3aL;
		expectedResults1[113][0] = 0x3fe21cac083126acL;
		expectedResults1[113][1] = 0x3fe1290b04b17396L;
		expectedResults1[113][2] = 0x3feb027af053b99eL;
		expectedResults1[113][3] = 0x3fe454b3b198aa3aL;
		expectedResults1[113][4] = 0x3fe340ae93c3d16cL;
		expectedResults1[113][5] = 0x3fef034814c488c5L;
		expectedResults1[113][6] = 0x3fe07b3e0d776c60L;
		expectedResults1[113][7] = 0x3ffc2de88e4240aaL;
		expectedResults1[113][8] = 0xbfe236918cefed90L;
		expectedResults1[114][0] = 0x3fe29ba5e353f782L;
		expectedResults1[114][1] = 0x3fe193af7c31a5b6L;
		expectedResults1[114][2] = 0x3feabd8f6981c186L;
		expectedResults1[114][3] = 0x3fe508b8445e28b7L;
		expectedResults1[114][4] = 0x3fe3dbb8410e25a8L;
		expectedResults1[114][5] = 0x3fee683e677a3489L;
		expectedResults1[114][6] = 0x3fe0dac57419630eL;
		expectedResults1[114][7] = 0x3ffc9e985649c322L;
		expectedResults1[114][8] = 0xbfe1593f048c1a6eL;
		expectedResults1[115][0] = 0x3fe33f7ced916812L;
		expectedResults1[115][1] = 0x3fe21bb00a63dfd1L;
		expectedResults1[115][2] = 0x3fea623413c9aaf2L;
		expectedResults1[115][3] = 0x3fe5f681b0150cf2L;
		expectedResults1[115][4] = 0x3fe4a6ee917d09e3L;
		expectedResults1[115][5] = 0x3fed9d08170b504eL;
		expectedResults1[115][6] = 0x3fe15425572c198fL;
		expectedResults1[115][7] = 0x3ffd329a238ad31eL;
		expectedResults1[115][8] = 0xbfe0443aa427a49eL;
		expectedResults1[116][0] = 0x3fe38d4fdf3b63f0L;
		expectedResults1[116][1] = 0x3fe25ba477163c4cL;
		expectedResults1[116][2] = 0x3fea35dc3b033b8dL;
		expectedResults1[116][3] = 0x3fe669bedbde6382L;
		expectedResults1[116][4] = 0x3fe508caa54d62e1L;
		expectedResults1[116][5] = 0x3fed3b2c033af750L;
		expectedResults1[116][6] = 0x3fe18d0dcf91e414L;
		expectedResults1[116][7] = 0x3ffd79f2fde94ba5L;
		expectedResults1[116][8] = 0xbfdf87b6c2ec1325L;
		expectedResults1[117][0] = 0x3fe3916872b0205aL;
		expectedResults1[117][1] = 0x3fe25eff2d479c8eL;
		expectedResults1[117][2] = 0x3fea3382762b1770L;
		expectedResults1[117][3] = 0x3fe66fda626d6550L;
		expectedResults1[117][4] = 0x3fe50df78bd53391L;
		expectedResults1[117][5] = 0x3fed35ff1cb3269fL;
		expectedResults1[117][6] = 0x3fe190092c8e2487L;
		expectedResults1[117][7] = 0x3ffd7db91d66a703L;
		expectedResults1[117][8] = 0xbfdf7a4fd673e6f0L;
		expectedResults1[118][0] = 0x3fe399999999992eL;
		expectedResults1[118][1] = 0x3fe265b3b272e7e9L;
		expectedResults1[118][2] = 0x3fea2ecda2d4aca2L;
		expectedResults1[118][3] = 0x3fe67c14ba45daf8L;
		expectedResults1[118][4] = 0x3fe518534a515ec4L;
		expectedResults1[118][5] = 0x3fed2ba35e36fb6dL;
		expectedResults1[118][6] = 0x3fe195fee1a91f61L;
		expectedResults1[118][7] = 0x3ffd8546cf86b4ddL;
		expectedResults1[118][8] = 0xbfdf5f8a65f532ecL;
		expectedResults1[119][0] = 0x3fe40c49ba5e34c6L;
		expectedResults1[119][1] = 0x3fe2c3139aaa3371L;
		expectedResults1[119][2] = 0x3fe9ec367560f8fcL;
		expectedResults1[119][3] = 0x3fe7291aedc114d6L;
		expectedResults1[119][4] = 0x3fe5aa6e4efef1feL;
		expectedResults1[119][5] = 0x3fec998859896833L;
		expectedResults1[119][6] = 0x3fe1e8e03e38c3a1L;
		expectedResults1[119][7] = 0x3ffdefd293290a3fL;
		expectedResults1[119][8] = 0xbfdded43d5368088L;
		expectedResults1[120][0] = 0x3fe4e147ae147a4eL;
		expectedResults1[120][1] = 0x3fe36df9aeb7d609L;
		expectedResults1[120][2] = 0x3fe96d1a98c057daL;
		expectedResults1[120][3] = 0x3fe873f8be0ffbc0L;
		expectedResults1[120][4] = 0x3fe6bf84fc98a816L;
		expectedResults1[120][5] = 0x3feb8471abefb21bL;
		expectedResults1[120][6] = 0x3fe2800beadc1dc9L;
		expectedResults1[120][7] = 0x3ffeb9b1e5d32440L;
		expectedResults1[120][8] = 0xbfdb530d8366318eL;
		expectedResults1[121][0] = 0x3fe5ef9db22d0da2L;
		expectedResults1[121][1] = 0x3fe4420757c105bdL;
		expectedResults1[121][2] = 0x3fe8c572d84032e4L;
		expectedResults1[121][3] = 0x3fea2b6843332eb9L;
		expectedResults1[121][4] = 0x3fe82b56662633c8L;
		expectedResults1[121][5] = 0x3fea18a042622669L;
		expectedResults1[121][6] = 0x3fe33acd54bbb4deL;
		expectedResults1[121][7] = 0x3fffc197dd32496aL;
		expectedResults1[121][8] = 0xbfd82ab5a0b0b5cdL;
		expectedResults1[122][0] = 0x3fe651eb851eb792L;
		expectedResults1[122][1] = 0x3fe48dc037b81effL;
		expectedResults1[122][2] = 0x3fe886c2dcdfd9eaL;
		expectedResults1[122][3] = 0x3fead11722d683a6L;
		expectedResults1[122][4] = 0x3fe8b36d72baa8cbL;
		expectedResults1[122][5] = 0x3fe9908935cdb166L;
		expectedResults1[122][6] = 0x3fe37d4e54fdfe2bL;
		expectedResults1[122][7] = 0x400011de35cb184eL;
		expectedResults1[122][8] = 0xbfd70e619d4cc817L;
		expectedResults1[123][0] = 0x3fe676c8b439574cL;
		expectedResults1[123][1] = 0x3fe4a9f3b00c5f47L;
		expectedResults1[123][2] = 0x3fe86f05125e09b1L;
		expectedResults1[123][3] = 0x3feb1014d25e6f94L;
		expectedResults1[123][4] = 0x3fe8e707b3cd8fe3L;
		expectedResults1[123][5] = 0x3fe95ceef4baca4eL;
		expectedResults1[123][6] = 0x3fe3960d80ae2eb8L;
		expectedResults1[123][7] = 0x4000246c10b3340aL;
		expectedResults1[123][8] = 0xbfd6a5048163014eL;
		expectedResults1[124][0] = 0x3fe810624dd2f0b4L;
		expectedResults1[124][1] = 0x3fe5dbf62742e8e5L;
		expectedResults1[124][2] = 0x3fe75ed0b5215bd1L;
		expectedResults1[124][3] = 0x3fedee4c6fd16dcaL;
		expectedResults1[124][4] = 0x3feb3c2331c74c7cL;
		expectedResults1[124][5] = 0x3fe707d376c10db4L;
		expectedResults1[124][6] = 0x3fe4a2096a783ffbL;
		expectedResults1[124][7] = 0x4000f84bc2ed6bbfL;
		expectedResults1[124][8] = 0xbfd23dc024a6a766L;
		expectedResults1[125][0] = 0x3fe84189374bc5acL;
		expectedResults1[125][1] = 0x3fe5ffc1ea968148L;
		expectedResults1[125][2] = 0x3fe73d21cb2058faL;
		expectedResults1[125][3] = 0x3fee4af90e381c87L;
		expectedResults1[125][4] = 0x3feb8718d8ada81aL;
		expectedResults1[125][5] = 0x3fe6bcddcfdab217L;
		expectedResults1[125][6] = 0x3fe4c157e38ab1ccL;
		expectedResults1[125][7] = 0x40011270a89b1b28L;
		expectedResults1[125][8] = 0xbfd1bb8bc022418eL;
		expectedResults1[126][0] = 0x3feac083126e9644L;
		expectedResults1[126][1] = 0x3fe7be323f09b05bL;
		expectedResults1[126][2] = 0x3fe574364df7c168L;
		expectedResults1[126][3] = 0x3ff1b504660a72b5L;
		expectedResults1[126][4] = 0x3fefadb130c88653L;
		expectedResults1[126][5] = 0x3fe2964577bfd3deL;
		expectedResults1[126][6] = 0x3fe6482be037cc0dL;
		expectedResults1[126][7] = 0x400274fb56a7525eL;
		expectedResults1[126][8] = 0xbfc6ed9f61ef8334L;
		
		//expectedResults2 -- for functions with two args
		expectedResults2[0][0] = 0x3fe9714bcf50de85l;
		expectedResults2[0][1] = 0x3fe2bbe0e180da96l;
		expectedResults2[0][2] = 0x3fedf4972bdb7236l;
		expectedResults2[0][3] = 0x3febfadea2c65e30l;
		expectedResults2[1][0] = 0x3fdc04b6788aa740l;
		expectedResults2[1][1] = 0x3fd56c62e02f455cl;
		expectedResults2[1][2] = 0x3fed604c2b19e492l;
		expectedResults2[1][3] = 0x3fe84512511a26c8l;
		expectedResults2[2][0] = 0x3fe41b588169c12cl;
		expectedResults2[2][1] = 0x3fc10dad63ed43dcl;
		expectedResults2[2][2] = 0x3ff5ca2907ffc9b2l;
		expectedResults2[2][3] = 0x3fee1438f0e2bdadl;
		expectedResults2[3][0] = 0x3fded48b71791c4cl;
		expectedResults2[3][1] = 0x3fde5af622c984f8l;
		expectedResults2[3][2] = 0x3fe9619182be7135l;
		expectedResults2[3][3] = 0x3fe6a17ca3c61062l;
		expectedResults2[4][0] = 0x3fdd866881d49d4el;
		expectedResults2[4][1] = 0x3fb89bd799752d58l;
		expectedResults2[4][2] = 0x3ff5d887ced39d05l;
		expectedResults2[4][3] = 0x3fedb4df68d1ad63l;
		expectedResults2[5][0] = 0x3fedf59561d869d3l;
		expectedResults2[5][1] = 0x3fc6358d3a72c068l;
		expectedResults2[5][2] = 0x3ff633650643e0a7l;
		expectedResults2[5][3] = 0x3fefa2deb7904e02l;
		expectedResults2[6][0] = 0x3fe4f24f5fd8620dl;
		expectedResults2[6][1] = 0x3fdc37bd366d4f48l;
		expectedResults2[6][2] = 0x3fef4c0ac6a90a20l;
		expectedResults2[6][3] = 0x3fea8be74bd05bc3l;
		expectedResults2[7][0] = 0x3fd32a55213bbd8cl;
		expectedResults2[7][1] = 0x3fe0a10006105786l;
		expectedResults2[7][2] = 0x3fe0baa8b245b7fal;
		expectedResults2[7][3] = 0x3fe119e8bc598ec3l;
		expectedResults2[8][0] = 0x3fea71c91cd9120bl;
		expectedResults2[8][1] = 0x3fecf1a4c4332aa2l;
		expectedResults2[8][2] = 0x3fe7b08fd5d7e129l;
		expectedResults2[8][3] = 0x3feaee338413b212l;
		expectedResults2[9][0] = 0x3fa533dd98f095c0l;
		expectedResults2[9][1] = 0x3fe642379d2990f6l;
		expectedResults2[9][2] = 0x3fae72219718a10el;
		expectedResults2[9][3] = 0x3fbbf25e57b041cbl;
		expectedResults2[10][0] = 0x3fee3167608b9281l;
		expectedResults2[10][1] = 0x3fe703528e82ac11l;
		expectedResults2[10][2] = 0x3fed6cd51c61a7a8l;
		expectedResults2[10][3] = 0x3feeb09e1ef1b465l;
		expectedResults2[11][0] = 0x3fe12d092174fde4l;
		expectedResults2[11][1] = 0x3fe88a89ae7c35a8l;
		expectedResults2[11][2] = 0x3fe38a6b3469d84fl;
		expectedResults2[11][3] = 0x3fe3db522353431cl;
		expectedResults2[12][0] = 0x3fed0b031272716cl;
		expectedResults2[12][1] = 0x3fdf9ea57963412cl;
		expectedResults2[12][2] = 0x3ff1281d49201513l;
		expectedResults2[12][3] = 0x3fee80d4b2e4b05al;
		expectedResults2[13][0] = 0x3fe0997ac994ed1el;
		expectedResults2[13][1] = 0x3fe762922cc2e835l;
		expectedResults2[13][2] = 0x3fe3c0e464fa3cb4l;
		expectedResults2[13][3] = 0x3fe3cecf414c24e3l;
		expectedResults2[14][0] = 0x3febdcedcc3eee8cl;
		expectedResults2[14][1] = 0x3fe27f8c0671a88fl;
		expectedResults2[14][2] = 0x3fef82c2ba8421b9l;
		expectedResults2[14][3] = 0x3fed89f97969a96cl;
		expectedResults2[15][0] = 0x3fe32f5aa37d5dd7l;
		expectedResults2[15][1] = 0x3fd4cf74d19fb7c6l;
		expectedResults2[15][2] = 0x3ff12e6f653fd8e1l;
		expectedResults2[15][3] = 0x3feb18890115d7a6l;
		expectedResults2[16][0] = 0x3fc919ef6a103c4cl;
		expectedResults2[16][1] = 0x3f8c7d6bcd670040l;
		expectedResults2[16][2] = 0x3ff7ffe884c1e637l;
		expectedResults2[16][3] = 0x3fef486f5a71df5bl;
		expectedResults2[17][0] = 0x3fd4d15f93162414l;
		expectedResults2[17][1] = 0x3fe35d339446083fl;
		expectedResults2[17][2] = 0x3fdf9101d0297574l;
		expectedResults2[17][3] = 0x3fe037dcbe5f94b5l;
		expectedResults2[18][0] = 0x3fe4c206230350cal;
		expectedResults2[18][1] = 0x3fe2c66a93ca7041l;
		expectedResults2[18][2] = 0x3feabc856e9d4bd9l;
		expectedResults2[18][3] = 0x3fe8d2dcb93bab52l;
		expectedResults2[19][0] = 0x3fe3b01f6e5f5fdfl;
		expectedResults2[19][1] = 0x3fedaa7de0a1b871l;
		expectedResults2[19][2] = 0x3fe2bfd0447e5d75l;
		expectedResults2[19][3] = 0x3fe465e29207c272l;
		expectedResults2[20][0] = 0xc008000000000000L;
		expectedResults2[20][1] = 0x7ff0000000000000L;
		expectedResults2[20][2] = 0x8000000000000000L;
		expectedResults2[20][3] = 0x7ff0000000000000L;
		expectedResults2[21][0] = 0x8000000000000000L;
		expectedResults2[21][1] = 0x4008000000000000L;
		expectedResults2[21][2] = 0x8000000000000000L;
		expectedResults2[21][3] = 0x8000000000000000L;
		expectedResults2[22][0] = 0xfff0000000000000L;
		expectedResults2[22][1] = 0xc008000000000000L;
		expectedResults2[22][2] = 0xbff921fb54442d18L;
		expectedResults2[22][3] = 0x8000000000000000L;
		expectedResults2[23][0] = 0x4000000000000000L;
		expectedResults2[23][1] = 0xc08ff80000000000L;
		expectedResults2[23][2] = 0x40091dfa5459b27dL;
		expectedResults2[23][3] = 0x8000000000000L;
		expectedResults2[24][0] = 0xc077d428f5c28f30L;
		expectedResults2[24][1] = 0xc034000000000000L;
		expectedResults2[24][2] = 0xbff9f8a6c7afa124L;
		expectedResults2[24][3] = 0x3536be4bf32c01fbL;
		expectedResults2[25][0] = 0xc0736e147ae14780L;
		expectedResults2[25][1] = 0xc034000000000000L;
		expectedResults2[25][2] = 0xbffa2921101f40acL;
		expectedResults2[25][3] = 0x35950cb1cadce763L;
		expectedResults2[26][0] = 0x4063ca3d70a3d766L;
		expectedResults2[26][1] = 0x4033800000000000L;
		expectedResults2[26][2] = 0x3ff72c0340bb930eL;
		expectedResults2[26][3] = 0x48d6536c03c325beL;
		expectedResults2[27][0] = 0x4087468f5c28f5dfL;
		expectedResults2[27][1] = 0xbfd0000000000000L;
		expectedResults2[27][2] = 0x3ff9235b490b58f8L;
		expectedResults2[27][3] = 0x3fc88072cf42cf6fL;
		expectedResults2[28][0] = 0x408c6851eb851ed7L;
		expectedResults2[28][1] = 0x4033800000000000L;
		expectedResults2[28][2] = 0x3ff8ca21904207c9L;
		expectedResults2[28][3] = 0x4be91add58b1583fL;
		expectedResults2[29][0] = 0x408f570a3d70a3f7L;
		expectedResults2[29][1] = 0x4033800000000000L;
		expectedResults2[29][2] = 0x3ff8d259628a523aL;
		expectedResults2[29][3] = 0x4c155044ab7671cbL;
		expectedResults2[30][0] = 0x4090961eb851eb95L;
		expectedResults2[30][1] = 0xbfd0000000000000L;
		expectedResults2[30][2] = 0x3ff922f247439e70L;
		expectedResults2[30][3] = 0x3fc66cba04bdebd0L;
		expectedResults2[31][0] = 0x4095e6ccccccccdfL;
		expectedResults2[31][1] = 0x4023400000000000L;
		expectedResults2[31][2] = 0x3ff905db39b6fe2bL;
		expectedResults2[31][3] = 0x46386a82970bbc5cL;
		expectedResults2[32][0] = 0x4097eaeb851eb865L;
		expectedResults2[32][1] = 0xc034000000000000L;
		expectedResults2[32][2] = 0x3ff9577ee381a8aeL;
		expectedResults2[32][3] = 0x32b51c65572b061bL;

	}

}
