/*******************************************************************************
 * Copyright IBM Corp. and others 2000
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution
 * and is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License, v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception [1] and GNU General Public
 * License, version 2 with the OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 *******************************************************************************/

#include "env/MemorySegment.hpp"
#include "env/SegmentProvider.hpp"
#include "env/Region.hpp"
#include "infra/ReferenceWrapper.hpp"
#include "env/TRMemory.hpp"

namespace TR {

Region::Region(TR::SegmentProvider &segmentProvider, TR::RawAllocator rawAllocator)
    : _bytesAllocated(0)
    , _segmentProvider(segmentProvider)
    , _rawAllocator(rawAllocator)
    , _initialSegment(_initialSegmentArea.data, INITIAL_SEGMENT_SIZE)
    , _currentSegment(TR::ref(_initialSegment))
    , _lastDestroyer(NULL)
{}

Region::Region(const Region &prototype)
    : _bytesAllocated(0)
    , _segmentProvider(prototype._segmentProvider)
    , _rawAllocator(prototype._rawAllocator)
    , _initialSegment(_initialSegmentArea.data, INITIAL_SEGMENT_SIZE)
    , _currentSegment(TR::ref(_initialSegment))
    , _lastDestroyer(NULL)
{}

Region::~Region() throw()
{
    /*
     * Destroy all object instances that depend on the region
     * to manage their lifetimes.
     */
    Destroyer *lastDestroyer = _lastDestroyer;
    while (lastDestroyer != NULL) {
        lastDestroyer->destroy();
        lastDestroyer = lastDestroyer->prev();
    }

    for (TR::reference_wrapper<TR::MemorySegment> latestSegment(_currentSegment);
         latestSegment.get() != _initialSegment; latestSegment = _currentSegment) {
        _currentSegment = TR::ref(latestSegment.get().unlink());
        _segmentProvider.release(latestSegment);
    }
    TR_ASSERT(_currentSegment.get() == _initialSegment, "self-referencial link was broken");
}

void *Region::allocate(size_t const size, void *hint)
{
    size_t const roundedSize = (size + 15) & (~15);
    if (_currentSegment.get().remaining() >= roundedSize) {
        _bytesAllocated += roundedSize;
        return _currentSegment.get().allocate(roundedSize);
    }
    TR::MemorySegment &newSegment = _segmentProvider.request(roundedSize);
    TR_ASSERT(newSegment.remaining() >= roundedSize, "Allocated segment is too small");
    newSegment.link(_currentSegment.get());
    _currentSegment = TR::ref(newSegment);
    _bytesAllocated += roundedSize;
    return _currentSegment.get().allocate(roundedSize);
}

void Region::deallocate(void *allocation, size_t) throw() {}
} // namespace TR
