/*******************************************************************************
 * Copyright IBM Corp. and others 2018
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution
 * and is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License, v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception [1] and GNU General Public
 * License, version 2 with the OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 *******************************************************************************/

#ifndef OMR_ARM64_INSTRUCTION_INCL
#define OMR_ARM64_INSTRUCTION_INCL

/*
 * The following #define and typedef must appear before any #includes in this file
 */
#ifndef OMR_INSTRUCTION_CONNECTOR
#define OMR_INSTRUCTION_CONNECTOR

namespace OMR {
namespace ARM64 {
class Instruction;
}

typedef OMR::ARM64::Instruction InstructionConnector;
} // namespace OMR
#else
#error OMR::ARM64::Instruction expected to be a primary connector, but an OMR connector is already defined
#endif

#include "compiler/codegen/OMRInstruction.hpp"

#include <stddef.h>
#include <stdint.h>
#include "codegen/InstOpCode.hpp"

namespace TR {
class CodeGenerator;
class Instruction;
class Node;
class Register;
class RegisterDependencyConditions;
} // namespace TR

namespace OMR { namespace ARM64 {

class OMR_EXTENSIBLE Instruction : public OMR::Instruction {
public:
    /**
     * @brief Constructor
     * @param[in] cg : CodeGenerator
     * @param[in] op : opcode
     * @param[in] node : node
     */
    Instruction(TR::CodeGenerator *cg, TR::InstOpCode::Mnemonic op, TR::Node *node = NULL);
    /**
     * @brief Constructor
     * @param[in] cg : CodeGenerator
     * @param[in] precedingInstruction : preceding instruction
     * @param[in] op : opcode
     * @param[in] node : node
     */
    Instruction(TR::CodeGenerator *cg, TR::Instruction *precedingInstruction, TR::InstOpCode::Mnemonic op,
        TR::Node *node = NULL);
    /**
     * @brief Constructor
     * @param[in] cg : CodeGenerator
     * @param[in] op : opcode
     * @param[in] cond : register dependency conditions
     * @param[in] node : node
     */
    Instruction(TR::CodeGenerator *cg, TR::InstOpCode::Mnemonic op, TR::RegisterDependencyConditions *cond,
        TR::Node *node = NULL);
    /**
     * @brief Constructor
     * @param[in] cg : CodeGenerator
     * @param[in] precedingInstruction : preceding instruction
     * @param[in] op : opcode
     * @param[in] cond : register dependency conditions
     * @param[in] node : node
     */
    Instruction(TR::CodeGenerator *cg, TR::Instruction *precedingInstruction, TR::InstOpCode::Mnemonic op,
        TR::RegisterDependencyConditions *cond, TR::Node *node = NULL);

    /**
     * @brief Instruction description string
     * @return description string
     */
    virtual const char *description() { return "ARM64"; }

    /**
     * @brief Gets instruction kind
     * @return instruction kind
     */
    virtual Kind getKind() { return IsNotExtended; }

    /**
     * @brief Estimates binary length
     * @param[in] currentEstimate : current estimated length
     * @return estimated binary length
     */
    virtual int32_t estimateBinaryLength(int32_t currentEstimate);

    /**
     * @brief Generates binary encoding of the instruction
     * @return instruction cursor
     */
    virtual uint8_t *generateBinaryEncoding();

    /**
     * @brief Answers if this instruction is a label or not
     * @return true if this instruction is a label, false otherwise
     */
    virtual bool isLabel() { return _opcode.getMnemonic() == TR::InstOpCode::label; }

    /**
     * @brief Gets the base register of memory access
     * @return base register
     */
    virtual TR::Register *getMemoryBase() { return NULL; }

    /**
     * @brief Gets the index register of memory access
     * @return index register
     */
    virtual TR::Register *getMemoryIndex() { return NULL; }

    /**
     * @brief Gets the offset of memory access
     * @return offset
     */
    virtual int32_t getOffset() { return 0; }

    /**
     * @brief Gets the register dependency conditions
     * @return register dependency conditions
     */
    virtual TR::RegisterDependencyConditions *getDependencyConditions() { return _conditions; }

    /**
     * @brief Sets the register dependency conditions
     * @param[in] cond : register dependency conditions
     * @return register dependency conditions
     */
    TR::RegisterDependencyConditions *setDependencyConditions(TR::RegisterDependencyConditions *cond)
    {
        return (_conditions = cond);
    }

    /**
     * @brief Sets GCMap mask
     * @param[in] cg : CodeGenerator
     * @param[in] mask : GCMap mask
     */
    void ARM64NeedsGCMap(TR::CodeGenerator *cg, uint32_t mask);

    /**
     * @brief Gets the memory data register
     * @return memory data register
     */
    virtual TR::Register *getMemoryDataRegister();

    /**
     * @brief Assigns registers
     * @param[in] kindToBeAssigned : register kind
     */
    virtual void assignRegisters(TR_RegisterKinds kindToBeAssigned);

    /**
     * @brief Answers whether this instruction references the given virtual register
     * @param[in] reg : virtual register
     * @return true when the instruction references the virtual register
     */
    virtual bool refsRegister(TR::Register *reg);

    /**
     * @brief Answers whether this instruction defines the given virtual register
     * @param[in] reg : virtual register
     * @return true when the instruction defines the virtual register
     */
    virtual bool defsRegister(TR::Register *reg);

    /**
     * @brief Answers whether this instruction uses the given virtual register
     * @param[in] reg : virtual register
     * @return true when the instruction uses the virtual register
     */
    virtual bool usesRegister(TR::Register *reg);

    /**
     * @brief Answers whether a dependency condition associated with this instruction references the given virtual
     * register
     * @param[in] reg : virtual register
     * @return true when a dependency condtion references the virtual register
     */
    virtual bool dependencyRefsRegister(TR::Register *reg);

    /**
     * @brief Calls OMR::Instruction::useRegister and remove placeholder flags if isDummy is false
     * @param[in] reg : virtual register
     * @param[in] isDummy : dummy flag
     */
    void useRegister(TR::Register *reg, bool isDummy = false);

    /*
     * Maps to TIndex in Instruction. Here we set values specific to ARM64 CodeGen.
     *
     * A 32-bit field where the lower 24-bits contain an integer that represents an
     * approximate ordering of instructions.
     *
     * The upper 8 bits are used for flags.
     * Instruction flags encoded by their bit position.  Subclasses may use any
     * available bits between LastBaseFlag and MaxBaseFlag inclusive.
     */
    enum {
        WillBePatched = 0x08000000
    };

    /**
     * @brief Answers the status of WillBePatched flag
     * @return true when WillBePatched flag is set
     */
    bool willBePatched() { return (_index & WillBePatched) != 0; }

    /**
     * @brief Sets WillBePatched flag
     * @param[in] v : flag status
     */
    void setWillBePatched(bool v = true) { v ? _index |= WillBePatched : _index &= ~WillBePatched; }

private:
    TR::RegisterDependencyConditions *_conditions;
};

}} // namespace OMR::ARM64

#endif
