/*
 * Copyright 2012-2026 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.assertj.tests.core.api.period;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.BDDAssertions.then;
import static org.assertj.core.error.ShouldHavePeriod.shouldHaveYears;
import static org.assertj.core.util.FailureMessages.actualIsNull;
import static org.assertj.tests.core.util.AssertionsUtil.expectAssertionError;

import java.time.Period;

import org.junit.jupiter.api.Test;

/**
 * @author Hayden Meloche
 */
class PeriodAssert_hasYears_Test {

  @Test
  void should_pass_if_period_has_expected_years() {
    // GIVEN
    Period period = Period.ofYears(10);
    // WHEN/THEN
    then(period).hasYears(10);
  }

  @Test
  void should_fail_when_period_is_null() {
    // GIVEN
    Period period = null;
    // WHEN
    final AssertionError code = expectAssertionError(() -> assertThat(period).hasYears(5));
    // THEN
    then(code).hasMessage(actualIsNull());
  }

  @Test
  void should_fail_if_period_does_not_have_expected_years() {
    // GIVEN
    Period period = Period.ofYears(10);
    // WHEN
    final AssertionError code = expectAssertionError(() -> assertThat(period).hasYears(15));
    // THEN
    then(code).hasMessage(shouldHaveYears(period, 10, 15).create());
  }
}
