/*
 * Copyright 2012-2026 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.assertj.core.util;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatIllegalArgumentException;
import static org.assertj.core.util.Lists.newArrayList;

import org.junit.jupiter.api.Test;

/**
 * Tests for <code>{@link Strings#join(String...)}</code>.
 * 
 * @author Alex Ruiz
 */
class Strings_join_Test {

  @Test
  void should_throw_error_if_delimiter_is_null() {
    assertThatIllegalArgumentException().isThrownBy(() -> Strings.join(null, "Uno", "Dos").with(null));
  }

  @Test
  void should_return_empty_String_if_array_to_join_is_null() {
    assertThat(Strings.join((String[]) null).with("|")).isEmpty();
  }

  @Test
  void should_join_using_delimiter() {
    assertThat(Strings.join("Luke", "Leia", "Han").with("|")).isEqualTo("Luke|Leia|Han");
  }

  @Test
  void should_join_using_delimiter_and_escape() {
    assertThat(Strings.join("Luke", "Leia", "Han").with("|", "'")).isEqualTo("'Luke'|'Leia'|'Han'");
  }

  @Test
  void should_join_using_iterable_delimiter_and_escape() {
    assertThat(Strings.join(newArrayList("Luke", "Leia", "Han")).with("|", "'")).isEqualTo("'Luke'|'Leia'|'Han'");
  }
}
