/*
 * Copyright 2012-2026 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.assertj.core.internal.floats;

import static java.lang.Float.NEGATIVE_INFINITY;
import static java.lang.Float.NaN;
import static java.lang.Float.POSITIVE_INFINITY;
import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatExceptionOfType;
import static org.assertj.core.api.Assertions.assertThatIllegalArgumentException;
import static org.assertj.core.api.Assertions.assertThatNullPointerException;
import static org.assertj.core.api.Assertions.catchThrowable;
import static org.assertj.core.api.Assertions.withinPercentage;
import static org.assertj.core.data.Percentage.withPercentage;
import static org.assertj.core.error.ShouldBeEqualWithinPercentage.shouldBeEqualWithinPercentage;
import static org.assertj.core.testkit.TestData.someInfo;
import static org.assertj.core.util.FailureMessages.actualIsNull;
import static org.mockito.Mockito.verify;

import org.assertj.core.api.AssertionInfo;
import org.assertj.core.internal.FloatsBaseTest;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;

class Floats_assertIsCloseToPercentage_Test extends FloatsBaseTest {

  private static final Float ZERO = 0f;
  private static final Float ONE = 1f;
  private static final Float TEN = 10f;

  @Test
  void should_fail_if_actual_is_null() {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(), null, ONE,
                                                                                                      withPercentage(ONE)))
                                                   .withMessage(actualIsNull());
  }

  @Test
  void should_fail_if_expected_value_is_null() {
    assertThatNullPointerException().isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(), ONE, null,
                                                                                       withPercentage(ONE)));
  }

  @Test
  void should_fail_if_percentage_is_null() {
    assertThatNullPointerException().isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(), ONE, ZERO, null));
  }

  @Test
  void should_fail_if_percentage_is_negative() {
    assertThatIllegalArgumentException().isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(), ONE, ZERO,
                                                                                           withPercentage(-1.0f)));
  }

  @ParameterizedTest
  @CsvSource({
      "1, 1, 1",
      "1, 2, 100",
      "-1, -1, 1",
      "-1, -2, 100",
      "-1, 1, 200"
  })
  void should_pass_if_difference_is_less_than_given_percentage(Float actual, Float other, Float percentage) {
    floats.assertIsCloseToPercentage(someInfo(), actual, other, withPercentage(percentage));
  }

  @ParameterizedTest
  @CsvSource({
      "1, 1, 0",
      "2, 1, 100",
      "1, 2, 50",
      "-1, -1, 0",
      "-2, -1, 100",
      "-1, -2, 50"
  })
  void should_pass_if_difference_is_equal_to_given_percentage(float actual, float other, float percentage) {
    floats.assertIsCloseToPercentage(someInfo(), actual, other, withPercentage(percentage));
  }

  @Test
  void should_fail_if_actual_is_not_close_enough_to_expected_value() {
    AssertionInfo info = someInfo();

    Throwable error = catchThrowable(() -> floats.assertIsCloseToPercentage(someInfo(), ONE, TEN, withPercentage(TEN)));

    assertThat(error).isInstanceOf(AssertionError.class);
    verify(failures).failure(info, shouldBeEqualWithinPercentage(ONE, TEN, withinPercentage(10), TEN - ONE));
  }

  @Test
  void should_fail_if_actual_is_NaN_and_expected_is_not() {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(), NaN, ONE,
                                                                                                      withPercentage(ONE)));
  }

  @Test
  void should_pass_if_actual_and_expected_are_NaN() {
    floats.assertIsCloseToPercentage(someInfo(), NaN, NaN, withPercentage(ONE));
  }

  @Test
  void should_fail_if_actual_is_POSITIVE_INFINITY_and_expected_is_not() {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(),
                                                                                                      POSITIVE_INFINITY, ONE,
                                                                                                      withPercentage(ONE)));
  }

  @Test
  void should_pass_if_actual_is_POSITIVE_INFINITY_and_expected_is_too() {
    floats.assertIsCloseToPercentage(someInfo(), POSITIVE_INFINITY, POSITIVE_INFINITY, withPercentage(ONE));
  }

  @Test
  void should_fail_if_actual_is_NEGATIVE_INFINITY_and_expected_is_not() {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(),
                                                                                                      NEGATIVE_INFINITY, ONE,
                                                                                                      withPercentage(ONE)));
  }

  @Test
  void should_pass_if_actual_is_NEGATIVE_INFINITY_and_expected_is_too() {
    floats.assertIsCloseToPercentage(someInfo(), NEGATIVE_INFINITY, NEGATIVE_INFINITY, withPercentage(ONE));
  }

  @Test
  void should_fail_if_actual_is_POSITIVE_INFINITY_and_expected_is_NEGATIVE_INFINITY() {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(),
                                                                                                      POSITIVE_INFINITY,
                                                                                                      NEGATIVE_INFINITY,
                                                                                                      withPercentage(ONE)));
  }

  @Test
  void should_fail_if_actual_is_NEGATIVE_INFINITY_and_expected_is_POSITIVE_INFINITY() {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> floats.assertIsCloseToPercentage(someInfo(),
                                                                                                      NEGATIVE_INFINITY,
                                                                                                      POSITIVE_INFINITY,
                                                                                                      withPercentage(ONE)));
  }
}
