/**
 * A {@link OptOperator} that is optimal for only the current layer being
 * rearranged.
 *
 * @packageDocumentation
 */
import { TwolayerOperator } from ".";
/**
 * How to handle large dags
 *
 * Setting this higher than `"small"` may result in very long run times or
 * crashes.
 */
export declare type LargeHandling = "small" | "medium" | "large";
/**
 * A {@link TwolayerOperator} for optimal decrossing of a single target layer
 *
 * The opt order operator orders the relevant layer to minimize the number of
 * crossings. This is expensive, but not nearly as expensive as optimizing all
 * crossings initially.
 *
 * Create with {@link opt}.
 *
 * <img alt="two layer opt example" src="media://sugi-simplex-twolayer-quad.png" width="400">
 */
export interface OptOperator extends TwolayerOperator<unknown, unknown> {
    /**
     * Set the large dag handling
     *
     * Setting to anything but `"small"` will allow running on larger dags, but
     * the layout may run forever, or crash the vm. (default: `"small"`)
     */
    large(val: LargeHandling): OptOperator;
    /** Return the handling of large graphs. */
    large(): LargeHandling;
    /**
     * Set whether to also minimize distance between nodes that share a parent or
     * child
     *
     * This adds more variables and constraints so will take longer, but will
     * likely produce a better layout. (default: false)
     */
    dist(val: boolean): OptOperator;
    /** get whether the current layout minimized distance */
    dist(): boolean;
}
/**
 * Create a default {@link OptOperator}, bundled as {@link twolayerOpt}.
 */
export declare function opt(...args: never[]): OptOperator;
