/**
 * The {@link SimplexOperator} positions nodes to maximize line straightness.
 * This is adapted from methods used for dot layout.
 *
 * @packageDocumentation
 */
import { CoordOperator } from ".";
import { DagLink } from "../../dag";
/**
 * An accessor to get how vertical a weight should be.
 *
 * A weight accessor returns three postitive numbers, where higher numbers
 * indicate than edge should be more vertical. The first number corresponds to
 * short edges, the second to medium edges, and the last one to the middle of
 * long edges. These numbers should generally be increasing.
 */
export interface WeightAccessor<NodeDatum = never, LinkDatum = never> {
    (link: DagLink<NodeDatum, LinkDatum>): readonly [number, number, number];
}
/**
 * a {@link WeightAccessor} that returns a constant value
 *
 * If using a constant value, this provides some small memory and time savings
 * over a regular accessor.
 */
export interface ConstAccessor<T extends readonly [number, number, number] = readonly [
    number,
    number,
    number
]> extends WeightAccessor<unknown, unknown> {
    (): T;
    /** the constant value */
    value: Readonly<T>;
}
/**
 * a function for creating a {@link ConstAccessor}
 */
export declare function createConstAccessor<T extends readonly [number, number, number]>(value: T): ConstAccessor<T>;
/** the operators of the simplex operator */
export interface Operators<N = never, L = never> {
    /** the weights for each edge */
    weight: WeightAccessor<N, L>;
}
/** node datum for operators */
declare type OpNodeDatum<O extends Operators> = O extends Operators<infer N, never> ? N : never;
/** link datum for operators */
declare type OpLinkDatum<O extends Operators> = O extends Operators<never, infer L> ? L : never;
/**
 * A {@link CoordOperator} that places nodes to maximize edge verticality
 *
 * The minimization mirrors that of Gansner, Emden R., et al. "A technique for
 * drawing directed graphs." IEEE Transactions on Software Engineering (1993).
 * This tries to make nodes close together, assigning greater weight for nodes
 * as part of an edge.
 *
 * Create with {@link simplex}.
 *
 * <img alt="quad example" src="media://sugi-simplex-twolayer-simplex.png" width="400">
 */
export interface SimplexOperator<Ops extends Operators> extends CoordOperator<OpNodeDatum<Ops>, OpLinkDatum<Ops>> {
    /**
     * Set the weights for how vertical edges should be.
     * The higher the weight, the more vertical an edge should be. Weights are
     * are triplets of numbers describing the weight for different parts of edge.
     * The first is between true nodes, the second is for near true nodes, and
     * the last is for the extents of long edges. Generally the number should be
     * increasing, and all must be positive. (default: () =\> [1, 2, 8])
     */
    weight<NewWeight extends WeightAccessor>(val: NewWeight): SimplexOperator<{
        /** new weight */
        weight: NewWeight;
    }>;
    /** Gets the current weight accessor */
    weight(): Ops["weight"];
}
/** default simplex operator */
export declare type DefaultSimplexOperator = SimplexOperator<{
    /** default weights taken from graphvis */
    weight: ConstAccessor<readonly [1, 2, 8]>;
}>;
/**
 * Create a default {@link SimplexOperator}, bundled as {@link coordSimplex}.
 */
export declare function simplex(...args: never[]): DefaultSimplexOperator;
export {};
