"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isSessionStorageAvailable = exports.isLocalStorageAvailable = exports.webStorageType = void 0;
exports.isWebStorageAvailable = isWebStorageAvailable;
exports.getItem = getItem;
exports.setItem = setItem;
exports.removeItem = removeItem;
var faro_core_1 = require("@grafana/faro-core");
exports.webStorageType = {
    session: 'sessionStorage',
    local: 'localStorage',
};
/**
 * Check if selected web storage mechanism is available.
 * @param type storage mechanism to test availability for.
 * @returns
 */
function isWebStorageAvailable(type) {
    var _a;
    try {
        var storage = void 0;
        storage = window[type];
        var testItem = '__faro_storage_test__';
        storage.setItem(testItem, testItem);
        storage.removeItem(testItem);
        return true;
    }
    catch (error) {
        // the above can throw
        (_a = faro_core_1.faro.internalLogger) === null || _a === void 0 ? void 0 : _a.info("Web storage of type ".concat(type, " is not available. Reason: ").concat(error));
        return false;
    }
}
/**
 * Get item from SessionStorage or LocalStorage.
 * @param key: the item key.
 * @param webStorageMechanism: wether the item shall be received form local storage or session storage. Defaults to local storage.
 */
function getItem(key, webStorageMechanism) {
    if (isWebStorageTypeAvailable(webStorageMechanism)) {
        return window[webStorageMechanism].getItem(key);
    }
    return null;
}
/**
 * Store item in SessionStorage or LocalStorage.
 * @param key: the item key.
 * @param value: the item data.
 * @param webStorageMechanism: wether the item shall be received form local storage or session storage. Defaults to local storage.
 */
function setItem(key, value, webStorageMechanism) {
    if (isWebStorageTypeAvailable(webStorageMechanism)) {
        try {
            window[webStorageMechanism].setItem(key, value);
        }
        catch (error) {
            // do nothing
        }
    }
}
/**
 * Remove item from SessionStorage or LocalStorage.
 * @param key: the item key.
 * @param webStorageMechanism: wether the item shall be received form local storage or session storage. Defaults to local storage.
 */
function removeItem(key, webStorageMechanism) {
    if (isWebStorageTypeAvailable(webStorageMechanism)) {
        window[webStorageMechanism].removeItem(key);
    }
}
exports.isLocalStorageAvailable = isWebStorageAvailable(exports.webStorageType.local);
exports.isSessionStorageAvailable = isWebStorageAvailable(exports.webStorageType.session);
function isWebStorageTypeAvailable(webStorageMechanism) {
    if (webStorageMechanism === exports.webStorageType.local) {
        return exports.isLocalStorageAvailable;
    }
    if (webStorageMechanism === exports.webStorageType.session) {
        return exports.isSessionStorageAvailable;
    }
    return false;
}
//# sourceMappingURL=webStorage.js.map