"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var __1 = require("..");
describe('Reactive', function () {
    describe('Observable', function () {
        it('Creates an observable, subscribe to it, and emit values', function () {
            var observable = new __1.Observable();
            var callback = jest.fn();
            observable.subscribe(callback);
            observable.notify(1);
            observable.notify(2);
            expect(callback).toHaveBeenCalledTimes(2);
            expect(callback).toHaveBeenNthCalledWith(1, 1);
            expect(callback).toHaveBeenNthCalledWith(2, 2);
        });
        it('Unsubscribes from an observable', function () {
            var observable = new __1.Observable();
            var callback = jest.fn();
            var subscription = observable.subscribe(callback);
            observable.notify(1);
            subscription.unsubscribe();
            observable.notify(2);
            expect(callback).toHaveBeenCalledTimes(1);
            expect(callback).toHaveBeenCalledWith(1);
        });
        it('takes emitted values until the predicate returns false', function () {
            var observable = new __1.Observable();
            var callback = jest.fn();
            observable.takeWhile(function (value) { return value <= 2; }).subscribe(callback);
            observable.notify(1);
            observable.notify(2);
            observable.notify(3);
            expect(callback).toHaveBeenCalledTimes(2);
            expect(callback).toHaveBeenNthCalledWith(1, 1);
            expect(callback).toHaveBeenNthCalledWith(2, 2);
        });
        it('subscribes to the first emitted value and unsubscribes after', function () {
            var observable = new __1.Observable();
            var callback = jest.fn();
            observable.first().subscribe(callback);
            observable.notify(1);
            observable.notify(2);
            expect(callback).toHaveBeenCalledTimes(1);
            expect(callback).toHaveBeenCalledWith(1);
        });
        it('unsubscribes a callback from the list of subscribers', function () {
            var observable = new __1.Observable();
            var callback = jest.fn();
            var sub = observable.subscribe(callback);
            observable.notify(1);
            sub.unsubscribe();
            observable.notify(2);
            expect(callback).toHaveBeenCalledTimes(1);
            expect(callback).toHaveBeenCalledWith(1);
        });
    });
    describe('Merge', function () {
        it('Merges multiple observables into a single observable using the merge() function', function () {
            var observable1 = new __1.Observable();
            var observable2 = new __1.Observable();
            var callback = jest.fn();
            var mergeObserverSub = new __1.Observable().merge(observable1, observable2).subscribe(callback);
            observable1.notify(1);
            observable2.notify('A');
            observable1.notify(2);
            observable2.notify('B');
            expect(callback).toHaveBeenCalledTimes(4);
            expect(callback).toHaveBeenNthCalledWith(1, 1);
            expect(callback).toHaveBeenNthCalledWith(2, 'A');
            expect(callback).toHaveBeenNthCalledWith(3, 2);
            expect(callback).toHaveBeenNthCalledWith(4, 'B');
            mergeObserverSub.unsubscribe();
            observable1.notify(3);
            observable2.notify('C');
            expect(callback).toHaveBeenCalledTimes(4);
            expect(callback).not.toHaveBeenNthCalledWith(5, 3);
            expect(callback).not.toHaveBeenNthCalledWith(6, 'C');
        });
        it('Unsubscribes from all observables when merge.unsubscribeAll isCalled', function () {
            var observable1 = new __1.Observable();
            var observable2 = new __1.Observable();
            var callback = jest.fn();
            var mergeObserverSub = new __1.Observable().merge(observable1, observable2);
            mergeObserverSub.subscribe(callback);
            observable1.notify(1);
            observable2.notify(2);
            mergeObserverSub.unsubscribe(callback);
            observable1.notify(3);
            observable2.notify(4);
            expect(callback).toHaveBeenCalledTimes(2);
            expect(callback).toHaveBeenNthCalledWith(1, 1);
            expect(callback).toHaveBeenNthCalledWith(2, 2);
        });
        it('Unsubscribes from all chained observables when unsubscribe is called on the final operator in the chain', function () {
            var observable = new __1.Observable().takeWhile(function (value) { return value < 3; }).filter(function (value) { return !(0, __1.isString)(value); });
            var callback = jest.fn();
            var chainedSubscription = observable.subscribe(callback);
            observable.notify(1);
            observable.notify(2);
            chainedSubscription.unsubscribe();
            observable.notify(3);
            observable.notify(4);
            observable.notify(5);
            expect(callback).toHaveBeenCalledTimes(2);
            expect(callback).toHaveBeenCalledWith(2);
            // If we would have a left over (pending) subscription, it would be called more times
            callback.mockClear();
            chainedSubscription = observable.subscribe(callback);
            observable.notify(1);
            observable.notify(2);
            chainedSubscription.unsubscribe();
            expect(callback).toHaveBeenCalledTimes(2);
            expect(callback).toHaveBeenCalledWith(2);
        });
    });
});
//# sourceMappingURL=reactive.test.js.map