/*
 * xen-destroy: Destroys a domain using libxl.  Domain will be
 * destroyed regardless of which libxl application (xl, libvirt, etc.)
 * created the domain.
 *
 * Copyright (C) 2014 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see
 * <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <xenstore.h>
#include <libxl.h>

#include "xen-utils.h"


int main(int argc, char **argv)
{
    int ret = -1;
	struct xs_handle *xsh;
    FILE *log_file;
    xentoollog_logger *logger;
    libxl_ctx *ctx = NULL;
    libxl_dominfo *dominfo = NULL;
    int i, num_doms;

	if (argc != 2) {
		fprintf(stderr, "Usage: xen-destroy <domain-name>\n");
		exit(1);
	}

	xsh = xs_daemon_open();
	if (!xsh) {
		fprintf(stderr, "Failed to open xenstore connection\n");
		exit(1);
	}

    if ((log_file = fopen("/dev/null", "a")) == NULL)  {
        printf("Failed to create null logger\n");
        goto cleanup;
    }

    logger = (xentoollog_logger *)xtl_createlogger_stdiostream(log_file,
                                                               XTL_DEBUG, 0);
    if (!logger) {
        printf("Failed to create null logger\n");
        goto cleanup;
    }

    if (libxl_ctx_alloc(&ctx, LIBXL_VERSION, 0, logger)) {
        printf("Failed to create libxl context\n");
        goto cleanup;
    }

    /* find domain ID */
    if (!(dominfo = libxl_list_domain(ctx, &num_doms))) {
        fprintf(stderr, "Failed to find domain %s\n", argv[1]);
        goto cleanup;
    }

    for (i = 0; i < num_doms; i++) {
        char *domname = domid_to_domname(xsh, dominfo[i].domid);
        if (domname == NULL)
            continue;

        if (strncmp(argv[1], domname, strlen(argv[1])) == 0) {
            if (libxl_domain_destroy(ctx, dominfo[i].domid, NULL) == 0)
                ret = 0;
            else
                fprintf(stderr, "Failed to destroy domain %s\n", argv[1]);
            free(domname);
            goto cleanup;
        }
        free(domname);
    }

    fprintf(stderr, "Failed to find domain %s\n", argv[1]);

cleanup:
    if (dominfo)
        libxl_dominfo_list_free(dominfo, num_doms);
    if (ctx)
        libxl_ctx_free(ctx);
    if (log_file)
        fclose(log_file);
    xs_daemon_close(xsh);

    return ret;
}
