/*
 * livepatch_bsc1239096
 *
 * Fix for CVE-2024-58013, bsc#1239096
 *
 *  Upstream commit:
 *  26fbd3494a7d ("Bluetooth: MGMT: Fix slab-use-after-free Read in mgmt_remove_adv_monitor_sync")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  Not affected
 *
 *  SLE15-SP4 and -SP5 commit:
 *  Not affected
 *
 *  SLE15-SP6 commit:
 *  8a1abea9f99bca9381b8904f5416f57fe176d22c
 *
 *  SLE MICRO-6-0 commit:
 *  8a1abea9f99bca9381b8904f5416f57fe176d22c
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Vincenzo Mezzela <vincenzo.mezzela@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_BT)

#if !IS_MODULE(CONFIG_BT)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from net/bluetooth/mgmt.c */
#include <linux/module.h>

/* klp-ccp: from include/asm-generic/unaligned.h */
#define __ASM_GENERIC_UNALIGNED_H

/* klp-ccp: from net/bluetooth/mgmt.c */
#include <net/bluetooth/bluetooth.h>
#include <net/bluetooth/hci_core.h>

/* klp-ccp: from include/net/bluetooth/hci_sync.h */
static int (*klpe_hci_update_passive_scan)(struct hci_dev *hdev);

/* klp-ccp: from include/net/bluetooth/hci_core.h */
static int (*klpe_hci_remove_single_adv_monitor)(struct hci_dev *hdev, u16 handle);
static int (*klpe_hci_remove_all_adv_monitor)(struct hci_dev *hdev);

/* klp-ccp: from net/bluetooth/mgmt.c */
#include <net/bluetooth/hci_sock.h>

#include <net/bluetooth/mgmt.h>

/* klp-ccp: from net/bluetooth/hci_request.h */
#include <asm/unaligned.h>

/* klp-ccp: from net/bluetooth/mgmt_util.h */
struct mgmt_pending_cmd {
	struct list_head list;
	u16 opcode;
	int index;
	void *param;
	size_t param_len;
	struct sock *sk;
	struct sk_buff *skb;
	void *user_data;
	int (*cmd_complete)(struct mgmt_pending_cmd *cmd, u8 status);
};

static int (*klpe_mgmt_cmd_complete)(struct sock *sk, u16 index, u16 cmd, u8 status,
		      void *rp, size_t rp_len);

static struct mgmt_pending_cmd *(*klpe_mgmt_pending_find)(unsigned short channel, u16 opcode,
					   struct hci_dev *hdev);

static void (*klpe_mgmt_pending_remove)(struct mgmt_pending_cmd *cmd);

/* klp-ccp: from net/bluetooth/eir.h */
#include <asm/unaligned.h>

/* klp-ccp: from net/bluetooth/mgmt.c */
static const u8 (*klpe_mgmt_status_table)[64];

static u8 (*klpe_mgmt_errno_status)(int err);

static u8 klpr_mgmt_status(int err)
{
	if (err < 0)
		return (*klpe_mgmt_errno_status)(err);

	if (err < ARRAY_SIZE((*klpe_mgmt_status_table)))
		return (*klpe_mgmt_status_table)[err];

	return MGMT_STATUS_FAILED;
}

static struct mgmt_pending_cmd *klpr_pending_find(u16 opcode, struct hci_dev *hdev)
{
	return (*klpe_mgmt_pending_find)(HCI_CHANNEL_CONTROL, opcode, hdev);
}

void klpp_mgmt_remove_adv_monitor_complete(struct hci_dev *hdev,
					     void *data, int status)
{
	struct mgmt_rp_remove_adv_monitor rp;
	struct mgmt_pending_cmd *cmd = data;
	struct mgmt_cp_remove_adv_monitor *cp;

	if (status == -ECANCELED ||
	    cmd != klpr_pending_find(MGMT_OP_REMOVE_ADV_MONITOR, hdev))
		return;

	hci_dev_lock(hdev);

	cp = cmd->param;

	rp.monitor_handle = cp->monitor_handle;

	if (!status)
		(*klpe_hci_update_passive_scan)(hdev);

	(*klpe_mgmt_cmd_complete)(cmd->sk, cmd->index, cmd->opcode,
			  klpr_mgmt_status(status), &rp, sizeof(rp));
	(*klpe_mgmt_pending_remove)(cmd);

	hci_dev_unlock(hdev);
	bt_dev_dbg(hdev, "remove monitor %d complete, status %d",
		   rp.monitor_handle, status);
}

int klpp_mgmt_remove_adv_monitor_sync(struct hci_dev *hdev, void *data)
{
	struct mgmt_pending_cmd *cmd = data;

	if (cmd != klpr_pending_find(MGMT_OP_REMOVE_ADV_MONITOR, hdev))
		return -ECANCELED;

	struct mgmt_cp_remove_adv_monitor *cp = cmd->param;
	u16 handle = __le16_to_cpu(cp->monitor_handle);

	if (!handle)
		return (*klpe_hci_remove_all_adv_monitor)(hdev);

	return (*klpe_hci_remove_single_adv_monitor)(hdev, handle);
}


#include "livepatch_bsc1239096.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "bluetooth"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "hci_remove_all_adv_monitor",
	  (void *)&klpe_hci_remove_all_adv_monitor, "bluetooth" },
	{ "hci_remove_single_adv_monitor",
	  (void *)&klpe_hci_remove_single_adv_monitor, "bluetooth" },
	{ "hci_update_passive_scan", (void *)&klpe_hci_update_passive_scan,
	  "bluetooth" },
	{ "mgmt_cmd_complete", (void *)&klpe_mgmt_cmd_complete, "bluetooth" },
	{ "mgmt_errno_status", (void *)&klpe_mgmt_errno_status, "bluetooth" },
	{ "mgmt_pending_find", (void *)&klpe_mgmt_pending_find, "bluetooth" },
	{ "mgmt_pending_remove", (void *)&klpe_mgmt_pending_remove,
	  "bluetooth" },
	{ "mgmt_status_table", (void *)&klpe_mgmt_status_table, "bluetooth" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1239096_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void livepatch_bsc1239096_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}

#endif /* IS_ENABLED(CONFIG_BT) */
