/*
 * livepatch_bsc1244631
 *
 * Fix for CVE-2024-36978, bsc#1244631
 *
 *  Upstream commit:
 *  affc18fdc694 ("net: sched: sch_multiq: fix possible OOB write in multiq_tune()")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  5dc7eca0c323543eb08556233a91c2ec1b913a6b
 *
 *  SLE15-SP4 and -SP5 commit:
 *  6416785a0a8615d34fa0105e78b5e943a070a0d7
 *
 *  SLE15-SP6 commit:
 *  3b6fd266e0669fb005e8a6475afaa5baab6459a8
 *
 *  SLE MICRO-6-0 commit:
 *  3b6fd266e0669fb005e8a6475afaa5baab6459a8
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Ali Abdallah <ali.abdallah@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/sched/sch_multiq.c */
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/skbuff.h>
#include <net/netlink.h>
#include <net/pkt_sched.h>

struct multiq_sched_data {
	u16 bands;
	u16 max_bands;
	u16 curband;
	struct tcf_proto __rcu *filter_list;
	struct tcf_block *block;
	struct Qdisc **queues;
};

int klpp_multiq_tune(struct Qdisc *sch, struct nlattr *opt,
		     struct netlink_ext_ack *extack)
{
	struct multiq_sched_data *q = qdisc_priv(sch);
	struct tc_multiq_qopt *qopt;
	struct Qdisc **removed;
	int i, n_removed = 0;

	if (!netif_is_multiqueue(qdisc_dev(sch)))
		return -EOPNOTSUPP;
	if (nla_len(opt) < sizeof(*qopt))
		return -EINVAL;

	qopt = nla_data(opt);

	qopt->bands = qdisc_dev(sch)->real_num_tx_queues;

	removed = kmalloc(sizeof(*removed) * (q->max_bands - qopt->bands),
			  GFP_KERNEL);
	if (!removed)
		return -ENOMEM;

	sch_tree_lock(sch);
	q->bands = qopt->bands;
	for (i = q->bands; i < q->max_bands; i++) {
		if (q->queues[i] != &noop_qdisc) {
			struct Qdisc *child = q->queues[i];

			q->queues[i] = &noop_qdisc;
			qdisc_purge_queue(child);
			removed[n_removed++] = child;
		}
	}

	sch_tree_unlock(sch);

	for (i = 0; i < n_removed; i++)
		qdisc_put(removed[i]);
	kfree(removed);

	for (i = 0; i < q->bands; i++) {
		if (q->queues[i] == &noop_qdisc) {
			struct Qdisc *child, *old;
			child = qdisc_create_dflt(sch->dev_queue,
						  &pfifo_qdisc_ops,
						  TC_H_MAKE(sch->handle,
							    i + 1), extack);
			if (child) {
				sch_tree_lock(sch);
				old = q->queues[i];
				q->queues[i] = child;
				if (child != &noop_qdisc)
					qdisc_hash_add(child, true);

				if (old != &noop_qdisc)
					qdisc_purge_queue(old);
				sch_tree_unlock(sch);
				qdisc_put(old);
			}
		}
	}
	return 0;
}


#include "livepatch_bsc1244631.h"

