var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { map } from "../../iters";
import { solve } from "../../simplex";
import { assert, bigrams } from "../../utils";
/** @internal */
function buildOperator(options) {
    function simplexCall(dag) {
        var _a;
        const variables = {};
        const constraints = {};
        const ints = {};
        const ids = new Map(map(dag, (node, i) => [node, i.toString()]));
        /** get node id */
        function n(node) {
            return ids.get(node);
        }
        /** get variable associated with a node */
        function variable(node) {
            return variables[n(node)];
        }
        /** enforce that first occurs before second
         *
         * @param prefix - determines a unique prefix to describe constraint
         * @param strict - strictly before or possibly equal
         */
        function before(prefix, first, second, diff = 1) {
            const fvar = variable(first);
            const svar = variable(second);
            const cons = `${prefix}: ${n(first)} -> ${n(second)}`;
            constraints[cons] = { min: diff };
            fvar[cons] = -1;
            svar[cons] = 1;
        }
        /** enforce that first and second occur on the same layer */
        function equal(prefix, first, second) {
            before(`${prefix} before`, first, second, 0);
            before(`${prefix} after`, second, first, 0);
        }
        const ranks = [];
        const groups = new Map();
        // Add node variables and fetch ranks
        for (const node of dag) {
            const nid = n(node);
            ints[nid] = 1;
            variables[nid] = { opt: 0 };
            const rank = options.rank(node);
            if (rank !== undefined) {
                ranks.push([rank, node]);
            }
            const group = options.group(node);
            if (group !== undefined) {
                const existing = groups.get(group);
                if (existing) {
                    existing.push(node);
                }
                else {
                    groups.set(group, [node]);
                }
            }
        }
        // Add link constraints
        for (const node of dag) {
            for (const [child, count] of node.ichildrenCounts()) {
                // make sure that multi nodes have at least one dummy row between them
                before("link", node, child, count > 1 ? 2 : 1);
                variable(node).opt += count;
                variable(child).opt -= count;
            }
        }
        // Add rank constraints
        const ranked = ranks.sort(([a], [b]) => a - b);
        for (const [[frank, fnode], [srank, snode]] of bigrams(ranked)) {
            if (frank < srank) {
                before("rank", fnode, snode);
            }
            else {
                equal("rank", fnode, snode);
            }
        }
        // group constraints
        for (const group of groups.values()) {
            for (const [first, second] of bigrams(group)) {
                equal("group", first, second);
            }
        }
        // NOTE bundling sets `this` to undefined, and we need it to be settable
        try {
            const assignment = solve("opt", "max", variables, constraints, ints);
            // lp solver doesn't assign some zeros
            for (const node of dag) {
                node.value = (_a = assignment[n(node)]) !== null && _a !== void 0 ? _a : 0;
            }
        }
        catch (_b) {
            assert(ranks.length || groups.size);
            throw new Error("could not find a feasible simplex layout, check that rank or group accessors are not ill-defined");
        }
    }
    function rank(newRank) {
        if (newRank === undefined) {
            return options.rank;
        }
        else {
            const { rank: _ } = options, rest = __rest(options, ["rank"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { rank: newRank }));
        }
    }
    simplexCall.rank = rank;
    function group(newGroup) {
        if (newGroup === undefined) {
            return options.group;
        }
        else {
            const { group: _ } = options, rest = __rest(options, ["group"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { group: newGroup }));
        }
    }
    simplexCall.group = group;
    return simplexCall;
}
/** @internal */
function defaultAccessor() {
    return undefined;
}
/**
 * Create a default {@link SimplexOperator}, bundled as {@link layeringSimplex}.
 */
export function simplex(...args) {
    if (args.length) {
        throw new Error(`got arguments to simplex(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({ rank: defaultAccessor, group: defaultAccessor });
}
