"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
var faro_core_1 = require("@grafana/faro-core");
var testUtils_1 = require("@grafana/faro-core/src/testUtils");
var originalWindow = window;
describe('Meta API', function () {
    var mockUrl = 'http://dummy.com';
    beforeEach(function () {
        window = Object.create(window);
        Object.defineProperty(window, 'location', {
            value: {
                href: mockUrl,
            },
            writable: true, // possibility to override
        });
    });
    beforeEach(function () {
        jest.resetModules();
        jest.clearAllMocks();
        jest.restoreAllMocks();
    });
    afterAll(function () {
        window = originalWindow;
    });
    describe('setView', function () {
        it('updates the view meta if the new view meta is different to the previous one', function () {
            var api = (0, faro_core_1.initializeFaro)((0, testUtils_1.mockConfig)()).api;
            var view = { name: 'my-view' };
            api.setView(view);
            var previousView = api.getView();
            expect(previousView).toEqual(view);
            var newView = { name: 'my-new-view' };
            api.setView(newView);
            previousView = api.getView();
            expect(previousView).toEqual(newView);
        });
        it('does not update the view meta if the new view meta is identical to the previous one', function () {
            var api = (0, faro_core_1.initializeFaro)((0, testUtils_1.mockConfig)()).api;
            var view = { name: 'my-view' };
            api.setView(view);
            var previousView = api.getView();
            expect(previousView).toEqual(view);
            var newView = { name: 'my-view' };
            api.setView(newView);
            previousView = api.getView();
            expect(previousView).toEqual(view);
        });
    });
    describe('setSession', function () {
        it('adds overrides to the session meta if provided via the setView() function call', function () {
            var initialSession = { id: 'my-session' };
            var api = (0, faro_core_1.initializeFaro)((0, testUtils_1.mockConfig)({ sessionTracking: { enabled: false, session: initialSession } })).api;
            expect(api.getSession()).toEqual(initialSession);
            var overrides = { serviceName: 'service-1' };
            var newSession = { id: 'my-new-session', attributes: { hello: 'world' } };
            api.setSession(newSession, { overrides: overrides });
            expect(api.getSession()).toEqual(__assign(__assign({}, newSession), { overrides: overrides }));
            overrides = { serviceName: 'service-2' };
            api.setSession({}, { overrides: overrides });
            expect(api.getSession()).toEqual({ overrides: overrides });
            overrides = { serviceName: 'service-3' };
            api.setSession(undefined, { overrides: overrides });
            expect(api.getSession()).toEqual({ overrides: overrides });
        });
        it('merges the new overrides with the existing session meta overrides', function () {
            var _a, _b, _c, _d, _e, _f, _g;
            var initialSession = { id: 'my-session' };
            var mc = (0, testUtils_1.mockConfig)({
                sessionTracking: {
                    session: initialSession,
                },
                trackGeolocation: false,
            });
            // mockConfig is the result of calling makeCoreConfig in faro-web-sdk package.
            // It it reads the geoLocationTracking properties it adds them to the sessionTracking.session.overrides object.
            mc.sessionTracking.session.overrides = { geoLocationTrackingEnabled: false };
            var api = (0, faro_core_1.initializeFaro)(mc).api;
            expect((_a = api.getSession()) === null || _a === void 0 ? void 0 : _a.id).toEqual(initialSession.id);
            expect((_b = api.getSession()) === null || _b === void 0 ? void 0 : _b.overrides).toBeDefined();
            expect((_c = api.getSession()) === null || _c === void 0 ? void 0 : _c.overrides).toStrictEqual({ geoLocationTrackingEnabled: false });
            var overrides = { serviceName: 'service-1' };
            var newSession = { id: 'my-new-session' };
            api.setSession(newSession, { overrides: overrides });
            expect((_d = api.getSession()) === null || _d === void 0 ? void 0 : _d.id).toEqual(newSession.id);
            expect((_e = api.getSession()) === null || _e === void 0 ? void 0 : _e.overrides).toStrictEqual(__assign(__assign({}, overrides), { geoLocationTrackingEnabled: false }));
            var newOverrides = { serviceName: 'service-2' };
            api.setSession(newSession, { overrides: newOverrides });
            expect((_f = api.getSession()) === null || _f === void 0 ? void 0 : _f.id).toEqual(newSession.id);
            expect((_g = api.getSession()) === null || _g === void 0 ? void 0 : _g.overrides).toStrictEqual(__assign(__assign({}, newOverrides), { geoLocationTrackingEnabled: false }));
        });
    });
    describe('setPage / getPage', function () {
        it('updates the page meta when setPage(meta) is called', function () {
            var api = (0, faro_core_1.initializeFaro)((0, testUtils_1.mockConfig)()).api;
            var page = { url: 'http://example.com/my-page', id: 'my-page' };
            api.setPage(page);
            expect(api.getPage()).toEqual(page);
            var newPage = { url: 'http://example.com/my-new-page', id: 'my-new-page' };
            api.setPage(newPage);
            expect(api.getPage()).toEqual(newPage);
        });
        it('updates the page id if the parameter of setPage is a string', function () {
            var _a;
            var api = (0, faro_core_1.initializeFaro)((0, testUtils_1.mockConfig)()).api;
            var initialPage = { url: 'http://example.com/my-page', id: 'my-page', attributes: { hello: 'world' } };
            api.setPage(initialPage);
            expect(api.getPage()).toStrictEqual(initialPage);
            var newPageId = 'my-new-page-id';
            api.setPage(newPageId);
            expect((_a = api.getPage()) === null || _a === void 0 ? void 0 : _a.id).toEqual(newPageId);
        });
        it('gets the page meta when getPage(meta) is called', function () {
            var api = (0, faro_core_1.initializeFaro)((0, testUtils_1.mockConfig)()).api;
            var page = { url: 'http://example.com/my-page', id: 'my-page' };
            api.setPage(page);
            expect(api.getPage()).toEqual(page);
        });
        // Note: there's an integration test in the web-sdk that tests the following scenario:
        // >>> it'sets the page meta correctly when setPage() is called and the locally cached meta is not set <<<
        // This is because it needs web-sdk functions to be able to test the integration
        // you can find it in the pageMeta test file: https://github.com/grafana/faro-web-sdk/blob/3c2ba0f8ea8bfdfb39cd79b704d9a6c07bc7834e/packages/web-sdk/src/metas/page/meta.test.ts#L10
    });
});
//# sourceMappingURL=initilialize.test.js.map