import { TransportItemType } from '../../transports';
import { deepEqual, getCurrentTimestamp, isArray, isEmpty, isError, isNull, isObject, stringifyExternalJson, stringifyObjectValues, } from '../../utils';
import { timestampToIsoString } from '../../utils/date';
import { USER_ACTION_START } from '../const';
import { shouldIgnoreEvent } from '../utils';
import { defaultExceptionType } from './const';
let stacktraceParser;
export function initializeExceptionsAPI({ internalLogger, config, metas, transports, tracesApi, actionBuffer, getMessage, }) {
    var _a;
    internalLogger.debug('Initializing exceptions API');
    let lastPayload = null;
    stacktraceParser = (_a = config.parseStacktrace) !== null && _a !== void 0 ? _a : stacktraceParser;
    const changeStacktraceParser = (newStacktraceParser) => {
        internalLogger.debug('Changing stacktrace parser');
        stacktraceParser = newStacktraceParser !== null && newStacktraceParser !== void 0 ? newStacktraceParser : stacktraceParser;
    };
    const getStacktraceParser = () => stacktraceParser;
    const { ignoreErrors = [], preserveOriginalError } = config;
    const pushError = (error, { skipDedupe, stackFrames, type, context, spanContext, timestampOverwriteMs, originalError } = {}) => {
        if (isErrorIgnored(ignoreErrors, originalError !== null && originalError !== void 0 ? originalError : error)) {
            return;
        }
        try {
            const ctx = stringifyObjectValues(Object.assign(Object.assign({}, parseCause(originalError !== null && originalError !== void 0 ? originalError : error)), (context !== null && context !== void 0 ? context : {})));
            const item = {
                meta: metas.value,
                payload: Object.assign(Object.assign({ type: type || error.name || defaultExceptionType, value: error.message, timestamp: timestampOverwriteMs ? timestampToIsoString(timestampOverwriteMs) : getCurrentTimestamp(), trace: spanContext
                        ? {
                            trace_id: spanContext.traceId,
                            span_id: spanContext.spanId,
                        }
                        : tracesApi.getTraceContext() }, (isEmpty(ctx) ? {} : { context: ctx })), (preserveOriginalError ? { originalError } : {})),
                type: TransportItemType.EXCEPTION,
            };
            stackFrames = stackFrames !== null && stackFrames !== void 0 ? stackFrames : (error.stack ? stacktraceParser === null || stacktraceParser === void 0 ? void 0 : stacktraceParser(error).frames : undefined);
            if (stackFrames === null || stackFrames === void 0 ? void 0 : stackFrames.length) {
                item.payload.stacktrace = {
                    frames: stackFrames,
                };
            }
            const testingPayload = {
                type: item.payload.type,
                value: item.payload.value,
                stackTrace: item.payload.stacktrace,
                context: item.payload.context,
            };
            if (!skipDedupe && config.dedupe && !isNull(lastPayload) && deepEqual(testingPayload, lastPayload)) {
                internalLogger.debug('Skipping error push because it is the same as the last one\n', item.payload);
                return;
            }
            lastPayload = testingPayload;
            internalLogger.debug('Pushing exception\n', item);
            const msg = getMessage();
            if (msg && msg.type === USER_ACTION_START) {
                actionBuffer.addItem(item);
            }
            else {
                transports.execute(item);
            }
        }
        catch (err) {
            internalLogger.error('Error pushing event', err);
        }
    };
    changeStacktraceParser(config.parseStacktrace);
    return {
        changeStacktraceParser,
        getStacktraceParser,
        pushError,
    };
}
function parseCause(error) {
    let cause = error.cause;
    if (isError(cause)) {
        cause = error.cause.toString();
        // typeof operator on null returns "object". This is a well-known quirk in JavaScript and is considered a bug that cannot be fixed due to backward compatibility issues.
        // MDN: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/typeof#typeof_null
    }
    else if (cause !== null && (isObject(error.cause) || isArray(error.cause))) {
        cause = stringifyExternalJson(error.cause);
    }
    else if (cause != null) {
        cause = error.cause.toString();
    }
    return cause == null ? {} : { cause };
}
function isErrorIgnored(ignoreErrors, error) {
    const { message, name, stack } = error;
    return shouldIgnoreEvent(ignoreErrors, message + ' ' + name + ' ' + stack);
}
//# sourceMappingURL=initialize.js.map