package network

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/aquasecurity/trivy/internal/testutil"
	"github.com/aquasecurity/trivy/pkg/iac/adapters/common"
	"github.com/aquasecurity/trivy/pkg/iac/adapters/terraform/tftestutil"
	"github.com/aquasecurity/trivy/pkg/iac/providers/azure/network"
	iacTypes "github.com/aquasecurity/trivy/pkg/iac/types"
)

func Test_Adapt(t *testing.T) {
	tests := []struct {
		name      string
		terraform string
		expected  network.Network
	}{
		{
			name: "defined",
			terraform: `
			resource "azurerm_network_security_rule" "example" {
				name                        = "example_security_rule"
				network_security_group_name = azurerm_network_security_group.example.name
				direction                   = "Inbound"
				access                      = "Allow"
				protocol                    = "TCP"
				source_port_range           = "*"
				destination_port_ranges     = ["3389"]
				source_address_prefix       = "4.53.160.75"
				destination_address_prefix  = "*"
		   }
		   
		   resource "azurerm_network_security_group" "example" {
			 name                = "tf-appsecuritygroup"
		   }

		   resource "azurerm_network_watcher_flow_log" "example" {
			resource_group_name  = azurerm_resource_group.example.name
			name                 = "example-log"
			enabled              = true
		  
			retention_policy {
			  enabled = true
			  days    = 7
			}		  
		  }
`,
			expected: network.Network{
				SecurityGroups: []network.SecurityGroup{
					{
						Rules: []network.SecurityGroupRule{
							{
								Allow: iacTypes.BoolTest(true),
								SourceAddresses: []iacTypes.StringValue{
									iacTypes.StringTest("4.53.160.75"),
								},
								DestinationAddresses: []iacTypes.StringValue{
									iacTypes.StringTest("*"),
								},
								SourcePorts: []common.PortRange{
									{
										Start: iacTypes.IntTest(0),
										End:   iacTypes.IntTest(65535),
									},
								},
								DestinationPorts: []common.PortRange{
									{
										Start: iacTypes.IntTest(3389),
										End:   iacTypes.IntTest(3389),
									},
								},
								Protocol: iacTypes.StringTest("TCP"),
							},
						},
					},
				},
				NetworkWatcherFlowLogs: []network.NetworkWatcherFlowLog{
					{
						Enabled: iacTypes.BoolTest(true),
						RetentionPolicy: network.RetentionPolicy{
							Enabled: iacTypes.BoolTest(true),
							Days:    iacTypes.IntTest(7),
						},
					},
				},
			},
		},
		{
			name: "defaults",
			terraform: `resource "azurerm_network_security_group" "example" {
	name = "tf-appsecuritygroup"
	security_rule {}
}
`,
			expected: network.Network{
				SecurityGroups: []network.SecurityGroup{
					{
						Rules: []network.SecurityGroupRule{
							{
								Allow: iacTypes.BoolTest(true),
							},
						},
					},
				},
			},
		},
		{
			name: "network interface",
			terraform: `resource "azurerm_network_interface" "example" {
	name                = "example-nic"
	location            = "eastus"
	resource_group_name = "example-rg"

	ip_configuration {
		name = "primary-ip"
		primary = true
		subnet_id = "subnet-primary-id"
		public_ip_address_id = "public-ip-primary-id"
	}

	ip_configuration {
		name = "secondary-ip"
		subnet_id = "subnet-secondary-id"
	}
}
`,
			expected: network.Network{
				NetworkInterfaces: []network.NetworkInterface{
					{
						Metadata: iacTypes.NewTestMetadata(),
						// legacy fields filled from primary
						SubnetID:        iacTypes.String("subnet-primary-id", iacTypes.NewTestMetadata()),
						HasPublicIP:     iacTypes.Bool(true, iacTypes.NewTestMetadata()),
						PublicIPAddress: iacTypes.String("public-ip-primary-id", iacTypes.NewTestMetadata()),

						IPConfigurations: []network.IPConfiguration{
							{
								Metadata:        iacTypes.NewTestMetadata(),
								SubnetID:        iacTypes.String("subnet-primary-id", iacTypes.NewTestMetadata()),
								Primary:         iacTypes.Bool(true, iacTypes.NewTestMetadata()),
								HasPublicIP:     iacTypes.Bool(true, iacTypes.NewTestMetadata()),
								PublicIPAddress: iacTypes.String("public-ip-primary-id", iacTypes.NewTestMetadata()),
							},
							{
								Metadata:        iacTypes.NewTestMetadata(),
								SubnetID:        iacTypes.String("subnet-secondary-id", iacTypes.NewTestMetadata()),
								Primary:         iacTypes.Bool(false, iacTypes.NewTestMetadata()),
								HasPublicIP:     iacTypes.Bool(false, iacTypes.NewTestMetadata()),
								PublicIPAddress: iacTypes.String("", iacTypes.NewTestMetadata()),
							},
						},
					},
				},
			},
		},
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			modules := tftestutil.CreateModulesFromSource(t, test.terraform, ".tf")
			adapted := Adapt(modules)
			testutil.AssertDefsecEqual(t, test.expected, adapted)
		})
	}
}

func Test_adaptWatcherLog(t *testing.T) {
	tests := []struct {
		name      string
		terraform string
		expected  network.NetworkWatcherFlowLog
	}{
		{
			name: "defined",
			terraform: `
			resource "azurerm_network_watcher_flow_log" "watcher" {		
				enabled = true
				retention_policy {
					enabled = true
					days = 90
				}
			}
`,
			expected: network.NetworkWatcherFlowLog{
				Enabled: iacTypes.BoolTest(true),
				RetentionPolicy: network.RetentionPolicy{
					Enabled: iacTypes.BoolTest(true),
					Days:    iacTypes.IntTest(90),
				},
			},
		},
		{
			name: "defaults",
			terraform: `
			resource "azurerm_network_watcher_flow_log" "watcher" {
				retention_policy {
				}
			}
`,
			expected: network.NetworkWatcherFlowLog{
				Enabled: iacTypes.BoolTest(false),
				RetentionPolicy: network.RetentionPolicy{
					Enabled: iacTypes.BoolTest(false),
					Days:    iacTypes.IntTest(0),
				},
			},
		},
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			modules := tftestutil.CreateModulesFromSource(t, test.terraform, ".tf")
			adapted := adaptWatcherLog(modules.GetBlocks()[0])
			testutil.AssertDefsecEqual(t, test.expected, adapted)
		})
	}
}

func TestLines(t *testing.T) {
	src := `
	resource "azurerm_network_security_group" "example" {
		name                = "tf-appsecuritygroup"
	}
   
	resource "azurerm_network_security_rule" "example" {
		name                        = "example_security_rule"
		network_security_group_name = azurerm_network_security_group.example.name
		direction                   = "Inbound"
		access                      = "Allow"
		protocol                    = "TCP"
		source_port_range           = "*"
		destination_port_ranges     = ["3389"]
		source_address_prefix       = "4.53.160.75"
		destination_address_prefix  = "*"
   }
   
   resource "azurerm_network_watcher_flow_log" "example" {
	resource_group_name  = azurerm_resource_group.example.name
	name                 = "example-log"
  
	retention_policy {
	  enabled = true
	  days    = 7
	}		  
  	}`

	modules := tftestutil.CreateModulesFromSource(t, src, ".tf")
	adapted := Adapt(modules)

	require.Len(t, adapted.SecurityGroups, 1)
	require.Len(t, adapted.NetworkWatcherFlowLogs, 1)

	securityGroup := adapted.SecurityGroups[0]
	rule := securityGroup.Rules[0]
	watcher := adapted.NetworkWatcherFlowLogs[0]

	assert.Equal(t, 2, securityGroup.Metadata.Range().GetStartLine())
	assert.Equal(t, 4, securityGroup.Metadata.Range().GetEndLine())

	assert.Equal(t, 6, rule.Metadata.Range().GetStartLine())
	assert.Equal(t, 16, rule.Metadata.Range().GetEndLine())

	assert.Equal(t, 9, rule.Outbound.GetMetadata().Range().GetStartLine())
	assert.Equal(t, 9, rule.Outbound.GetMetadata().Range().GetEndLine())

	assert.Equal(t, 10, rule.Allow.GetMetadata().Range().GetStartLine())
	assert.Equal(t, 10, rule.Allow.GetMetadata().Range().GetEndLine())

	assert.Equal(t, 11, rule.Protocol.GetMetadata().Range().GetStartLine())
	assert.Equal(t, 11, rule.Protocol.GetMetadata().Range().GetEndLine())

	assert.Equal(t, 12, rule.SourcePorts[0].Metadata.Range().GetStartLine())
	assert.Equal(t, 12, rule.SourcePorts[0].Metadata.Range().GetEndLine())

	assert.Equal(t, 13, rule.DestinationPorts[0].Metadata.Range().GetStartLine())
	assert.Equal(t, 13, rule.DestinationPorts[0].Metadata.Range().GetEndLine())

	assert.Equal(t, 14, rule.SourceAddresses[0].GetMetadata().Range().GetStartLine())
	assert.Equal(t, 14, rule.SourceAddresses[0].GetMetadata().Range().GetEndLine())

	assert.Equal(t, 15, rule.DestinationAddresses[0].GetMetadata().Range().GetStartLine())
	assert.Equal(t, 15, rule.DestinationAddresses[0].GetMetadata().Range().GetEndLine())

	assert.Equal(t, 18, watcher.Metadata.Range().GetStartLine())
	assert.Equal(t, 26, watcher.Metadata.Range().GetEndLine())

	assert.Equal(t, 22, watcher.RetentionPolicy.Metadata.Range().GetStartLine())
	assert.Equal(t, 25, watcher.RetentionPolicy.Metadata.Range().GetEndLine())

	assert.Equal(t, 23, watcher.RetentionPolicy.Enabled.GetMetadata().Range().GetStartLine())
	assert.Equal(t, 23, watcher.RetentionPolicy.Enabled.GetMetadata().Range().GetEndLine())

	assert.Equal(t, 24, watcher.RetentionPolicy.Days.GetMetadata().Range().GetStartLine())
	assert.Equal(t, 24, watcher.RetentionPolicy.Days.GetMetadata().Range().GetEndLine())
}
