/*
 * livepatch_bsc1243648
 *
 * Fix for CVE-2024-56558, bsc#1243648
 *
 *  Upstream commit:
 *  be8f982c369c ("nfsd: make sure exp active before svc_export_show")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  e6fe56ba830601d4f9e76f428eb178c4ee83e9cd
 *
 *  SLE15-SP4 and -SP5 commit:
 *  cbb730b5b99597d866d86675147dc5c158241b51
 *
 *  SLE15-SP6 commit:
 *  2126f1262944cdda3156a1a11e33f429ac80c773
 *
 *  SLE MICRO-6-0 commit:
 *  2126f1262944cdda3156a1a11e33f429ac80c773
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Ali Abdallah <ali.abdallah@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


/* klp-ccp: from fs/nfsd/export.c */
#include <linux/slab.h>
#include <linux/namei.h>
#include <linux/module.h>

#include <linux/sunrpc/svc_xprt.h>

/* klp-ccp: from include/linux/exportfs.h */
#define LINUX_EXPORTFS_H 1

/* klp-ccp: from include/linux/sunrpc/cache.h */
static int (*klpe_cache_check)(struct cache_detail *detail,
		       struct cache_head *h, struct cache_req *rqstp);

/* klp-ccp: from fs/nfsd/nfsd.h */
#include <linux/types.h>
#include <linux/mount.h>

/* klp-ccp: from include/linux/nfs4.h */
#define _LINUX_NFS4_H

/* klp-ccp: from fs/nfsd/nfsd.h */
#include <linux/sunrpc/svc.h>
#include <linux/sunrpc/svc_xprt.h>
#include <linux/sunrpc/msg_prot.h>

#include <uapi/linux/nfsd/debug.h>

/* klp-ccp: from fs/nfsd/netns.h */
#include <net/net_namespace.h>

#include <linux/percpu_counter.h>

/* klp-ccp: from fs/nfsd/export.h */
#include <linux/sunrpc/cache.h>
#include <linux/percpu_counter.h>
#include <uapi/linux/nfsd/export.h>
#include <linux/nfs4.h>

struct nfsd4_fs_locations {
	uint32_t locations_count;
	struct nfsd4_fs_location *locations;
/* If we're not actually serving this data ourselves (only providing a
 * list of replicas that do serve it) then we set "migrated": */
	int migrated;
};

#define MAX_SECINFO_LIST	8

struct exp_flavor_info {
	u32	pseudoflavor;
	u32	flags;
};

struct svc_export {
	struct cache_head	h;
	struct auth_domain *	ex_client;
	int			ex_flags;
	int			ex_fsid;
	struct path		ex_path;
	kuid_t			ex_anon_uid;
	kgid_t			ex_anon_gid;
	unsigned char *		ex_uuid; /* 16 byte fsid */
	struct nfsd4_fs_locations ex_fslocs;
	uint32_t		ex_nflavors;
	struct exp_flavor_info	ex_flavors[MAX_SECINFO_LIST];
	u32			ex_layout_types;
	struct nfsd4_deviceid_map *ex_devid_map;
	struct cache_detail	*cd;
	struct rcu_head		ex_rcu;
	unsigned long		ex_xprtsec_modes;
	struct export_stats	*ex_stats;
};

static inline void exp_put(struct svc_export *exp)
{
	cache_put(&exp->h, exp->cd);
}

/* klp-ccp: from fs/nfsd/stats.h */
#include <uapi/linux/nfsd/stats.h>
#include <linux/percpu_counter.h>

/* klp-ccp: from fs/nfsd/nfsfh.h */
#include <linux/sunrpc/svc.h>

#include <linux/exportfs.h>

/* klp-ccp: from fs/nfsd/pnfs.h */
#ifdef CONFIG_NFSD_V4
#include <linux/exportfs.h>
#include <linux/nfsd/export.h>

/* klp-ccp: from fs/nfsd/state.h */
#include <linux/idr.h>
#include <linux/refcount.h>
#include <linux/sunrpc/svc_xprt.h>

#else
#error "klp-ccp: a preceeding branch should have been taken"
/* klp-ccp: from fs/nfsd/pnfs.h */
#endif /* CONFIG_NFSD_V4 */

/* klp-ccp: from fs/nfsd/trace.h */
#if !defined(_NFSD_TRACE_H) || defined(TRACE_HEADER_MULTI_READ)

/* klp-ccp: from fs/nfsd/vfs.h */
#include <linux/fs.h>

/* klp-ccp: from fs/nfsd/cache.h */
#include <linux/sunrpc/svc.h>

#else
#error "klp-ccp: a preceeding branch should have been taken"
/* klp-ccp: from fs/nfsd/trace.h */
#endif /* _NFSD_TRACE_H */

#include <trace/define_trace.h>

/* klp-ccp: from fs/nfsd/export.c */
static int is_export_stats_file(struct seq_file *m)
{
	/*
	 * The export_stats file uses the same ops as the exports file.
	 * We use the file's name to determine the reported info per export.
	 * There is no rename in nsfdfs, so d_name.name is stable.
	 */
	return !strcmp(m->file->f_path.dentry->d_name.name, "export_stats");
}

static int (*klpe_svc_export_show)(struct seq_file *m,
			   struct cache_detail *cd,
			   struct cache_head *h);

int klpp_e_show(struct seq_file *m, void *p)
{
	struct cache_head *cp = p;
	struct svc_export *exp = container_of(cp, struct svc_export, h);
	struct cache_detail *cd = m->private;
	bool export_stats = is_export_stats_file(m);

	if (p == SEQ_START_TOKEN) {
		seq_puts(m, "# Version 1.1\n");
		if (export_stats)
			seq_puts(m, "# Path Client Start-time\n#\tStats\n");
		else
			seq_puts(m, "# Path Client(Flags) # IPs\n");
		return 0;
	}

	if (!cache_get_rcu(&exp->h))
		return 0;

	if ((*klpe_cache_check)(cd, &exp->h, NULL))
		return 0;

	exp_put(exp);
	return (*klpe_svc_export_show)(m, cd, cp);
}


#include "livepatch_bsc1243648.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "nfsd"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "cache_check", (void *)&klpe_cache_check, "sunrpc" },
	{ "svc_export_show", (void *)&klpe_svc_export_show, "nfsd" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1243648_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void livepatch_bsc1243648_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}
