/*
 * livepatch_bsc1251228
 *
 * Fix for CVE-2022-50432, bsc#1251228
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Lidong Zhong <lidong.zhong@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


/* klp-ccp: from fs/kernfs/dir.c */
#include <linux/sched.h>
#include <linux/fs.h>

#include <linux/idr.h>
#include <linux/slab.h>

#include <linux/hash.h>

/* klp-ccp: from fs/kernfs/kernfs-internal.h */
#include <linux/lockdep.h>
#include <linux/fs.h>
#include <linux/mutex.h>
#include <linux/xattr.h>

#include <linux/kernfs.h>

/* klp-ccp: from fs/kernfs/kernfs-internal.h */
struct kernfs_iattrs {
	kuid_t			ia_uid;
	kgid_t			ia_gid;
	struct timespec64	ia_atime;
	struct timespec64	ia_mtime;
	struct timespec64	ia_ctime;

	struct simple_xattrs	xattrs;
};

#define KN_DEACTIVATED_BIAS		(INT_MIN + 1)

static inline struct kernfs_root *kernfs_root(struct kernfs_node *kn)
{
	/* if parent exists, it's always a dir; otherwise, @sd is a dir */
	if (kn->parent)
		kn = kn->parent;
	return kn->dir.root;
}

static struct mutex (*klpe_kernfs_mutex);

static void (*klpe_kernfs_drain_open_files)(struct kernfs_node *kn);

#define rb_to_kn(X) rb_entry((X), struct kernfs_node, rb)

static bool klpr_kernfs_active(struct kernfs_node *kn)
{
	lockdep_assert_held(&(*klpe_kernfs_mutex));
	return atomic_read(&kn->active) >= 0;
}

static bool kernfs_lockdep(struct kernfs_node *kn)
{
#ifdef CONFIG_DEBUG_LOCK_ALLOC
#error "klp-ccp: non-taken branch"
#else
	return false;
#endif
}

static bool (*klpe_kernfs_unlink_sibling)(struct kernfs_node *kn);

static void klpr_kernfs_drain(struct kernfs_node *kn)
	__releases(&kernfs_mutex) __acquires(&kernfs_mutex)
{
	struct kernfs_root *root = kernfs_root(kn);

	lockdep_assert_held(&(*klpe_kernfs_mutex));
	WARN_ON_ONCE(klpr_kernfs_active(kn));

	mutex_unlock(&(*klpe_kernfs_mutex));

	if (kernfs_lockdep(kn)) {
		rwsem_acquire(&kn->dep_map, 0, 0, _RET_IP_);
		if (atomic_read(&kn->active) != KN_DEACTIVATED_BIAS)
			lock_contended(&kn->dep_map, _RET_IP_);
	}

	/* but everyone should wait for draining */
	wait_event(root->deactivate_waitq,
		   atomic_read(&kn->active) == KN_DEACTIVATED_BIAS);

	if (kernfs_lockdep(kn)) {
		lock_acquired(&kn->dep_map, _RET_IP_);
		rwsem_release(&kn->dep_map, _RET_IP_);
	}

	(*klpe_kernfs_drain_open_files)(kn);

	mutex_lock(&(*klpe_kernfs_mutex));
}

void kernfs_get(struct kernfs_node *kn);

extern typeof(kernfs_get) kernfs_get;

void kernfs_put(struct kernfs_node *kn);

extern typeof(kernfs_put) kernfs_put;

static struct kernfs_node *(*klpe_kernfs_find_ns)(struct kernfs_node *parent,
					  const unsigned char *name,
					  const void *ns);

static struct kernfs_node *kernfs_leftmost_descendant(struct kernfs_node *pos)
{
	struct kernfs_node *last;

	while (true) {
		struct rb_node *rbn;

		last = pos;

		if (kernfs_type(pos) != KERNFS_DIR)
			break;

		rbn = rb_first(&pos->dir.children);
		if (!rbn)
			break;

		pos = rb_to_kn(rbn);
	}

	return last;
}

static struct kernfs_node *(*klpe_kernfs_next_descendant_post)(struct kernfs_node *pos,
						       struct kernfs_node *root);

static void klpr___kernfs_remove(struct kernfs_node *kn)
{
	struct kernfs_node *pos;

	lockdep_assert_held(&(*klpe_kernfs_mutex));

	/*
	 * Short-circuit if non-root @kn has already finished removal.
	 * This is for kernfs_remove_self() which plays with active ref
	 * after removal.
	 */
	if (!kn || (kn->parent && RB_EMPTY_NODE(&kn->rb)))
		return;

	pr_debug("kernfs %s: removing\n", kn->name);

	/* prevent any new usage under @kn by deactivating all nodes */
	pos = NULL;
	while ((pos = (*klpe_kernfs_next_descendant_post)(pos, kn)))
		if (klpr_kernfs_active(pos))
			atomic_add(KN_DEACTIVATED_BIAS, &pos->active);

	/* deactivate and unlink the subtree node-by-node */
	do {
		pos = kernfs_leftmost_descendant(kn);

		/*
		 * kernfs_drain() drops kernfs_mutex temporarily and @pos's
		 * base ref could have been put by someone else by the time
		 * the function returns.  Make sure it doesn't go away
		 * underneath us.
		 */
		kernfs_get(pos);

		/*
		 * Drain iff @kn was activated.  This avoids draining and
		 * its lockdep annotations for nodes which have never been
		 * activated and allows embedding kernfs_remove() in create
		 * error paths without worrying about draining.
		 */
		if (kn->flags & KERNFS_ACTIVATED)
			klpr_kernfs_drain(pos);
		else
			WARN_ON_ONCE(atomic_read(&kn->active) != KN_DEACTIVATED_BIAS);

		/*
		 * kernfs_unlink_sibling() succeeds once per node.  Use it
		 * to decide who's responsible for cleanups.
		 */
		if (!pos->parent || (*klpe_kernfs_unlink_sibling)(pos)) {
			struct kernfs_iattrs *ps_iattr =
				pos->parent ? pos->parent->iattr : NULL;

			/* update timestamps on the parent */
			if (ps_iattr) {
				ktime_get_real_ts64(&ps_iattr->ia_ctime);
				ps_iattr->ia_mtime = ps_iattr->ia_ctime;
			}

			kernfs_put(pos);
		}

		kernfs_put(pos);
	} while (pos != kn);
}

int klpp_kernfs_remove_by_name_ns(struct kernfs_node *parent, const char *name,
			     const void *ns)
{
	struct kernfs_node *kn;

	if (!parent) {
		WARN(1, KERN_WARNING "kernfs: can not remove '%s', no directory\n",
			name);
		return -ENOENT;
	}

	mutex_lock(&(*klpe_kernfs_mutex));

	kn = (*klpe_kernfs_find_ns)(parent, name, ns);
	if (kn) {
		kernfs_get(kn);
		klpr___kernfs_remove(kn);
		kernfs_put(kn);
	}

	mutex_unlock(&(*klpe_kernfs_mutex));

	if (kn)
		return 0;
	else
		return -ENOENT;
}


#include "livepatch_bsc1251228.h"

#include <linux/kernel.h>
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "kernfs_drain_open_files", (void *)&klpe_kernfs_drain_open_files },
	{ "kernfs_find_ns", (void *)&klpe_kernfs_find_ns },
	{ "kernfs_mutex", (void *)&klpe_kernfs_mutex },
	{ "kernfs_next_descendant_post",
	  (void *)&klpe_kernfs_next_descendant_post },
	{ "kernfs_unlink_sibling", (void *)&klpe_kernfs_unlink_sibling },
};

int livepatch_bsc1251228_init(void)
{
	return __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

