package typescript

import (
	"github.com/grafana/codejen"
	"github.com/grafana/cog/internal/ast"
	"github.com/grafana/cog/internal/ast/compiler"
	"github.com/grafana/cog/internal/jennies/common"
	"github.com/grafana/cog/internal/languages"
	"github.com/grafana/cog/internal/tools"
)

const LanguageRef = "typescript"

type Config struct {
	PathPrefix string `yaml:"path_prefix"`

	// SkipRuntime disables runtime-related code generation when enabled.
	// Note: builders can NOT be generated with this flag turned on, as they
	// rely on the runtime to function.
	SkipRuntime bool `yaml:"skip_runtime"`

	// SkipIndex disables the generation of `index.ts` files.
	SkipIndex bool `yaml:"skip_index"`

	// BuilderTemplatesDirectories holds a list of directories containing templates
	// to be used to override parts of builders.
	BuilderTemplatesDirectories []string `yaml:"builder_templates"`

	// PackagesImportMap associates package names to their import path.
	PackagesImportMap map[string]string `yaml:"packages_import_map"`

	// EnumsAsUnionTypes generates enums as a union of values instead of using
	// an actual `enum` declaration.
	// If EnumsAsUnionTypes is false, an enum will be generated as:
	// ```ts
	// enum Direction {
	//   Up = "up",
	//   Down = "down",
	//   Left = "left",
	//   Right = "right",
	// }
	// ```
	// If EnumsAsUnionTypes is true, the same enum will be generated as:
	// ```ts
	// type Direction = "up" | "down" | "left" | "right";
	// ```
	EnumsAsUnionTypes bool `yaml:"enums_as_union_types"`
}

func (config *Config) InterpolateParameters(interpolator func(input string) string) {
	config.BuilderTemplatesDirectories = tools.Map(config.BuilderTemplatesDirectories, interpolator)
	for pkg, importPath := range config.PackagesImportMap {
		config.PackagesImportMap[pkg] = interpolator(importPath)
	}
}

func (config *Config) enumFormatter(packageMapper packageMapper) enumFormatter {
	if config.EnumsAsUnionTypes {
		return &enumAsDisjunctionFormatter{}
	}

	return &enumAsTypeFormatter{packageMapper: packageMapper}
}

type Language struct {
	config          Config
	apiRefCollector *common.APIReferenceCollector
}

func New(config Config) *Language {
	return &Language{
		config:          config,
		apiRefCollector: common.NewAPIReferenceCollector(),
	}
}

func (language *Language) Name() string {
	return LanguageRef
}

func (language *Language) Jennies(globalConfig languages.Config) *codejen.JennyList[languages.Context] {
	tmpl := initTemplates(language.config.BuilderTemplatesDirectories)

	jenny := codejen.JennyListWithNamer[languages.Context](func(_ languages.Context) string {
		return LanguageRef
	})
	jenny.AppendOneToMany(
		common.If[languages.Context](!language.config.SkipRuntime, Runtime{}),

		common.If[languages.Context](globalConfig.Types, RawTypes{config: language.config}),
		common.If[languages.Context](!language.config.SkipRuntime && globalConfig.Builders, &Builder{
			config:          language.config,
			tmpl:            tmpl,
			apiRefCollector: language.apiRefCollector,
		}),

		common.If[languages.Context](!language.config.SkipIndex, Index{Targets: globalConfig}),

		common.If[languages.Context](globalConfig.APIReference, common.APIReference{
			Collector: language.apiRefCollector,
			Language:  LanguageRef,
			Formatter: apiReferenceFormatter(language.config),
			Tmpl:      tmpl,
		}),
	)
	jenny.AddPostprocessors(common.GeneratedCommentHeader(globalConfig))

	return jenny
}

func (language *Language) CompilerPasses() compiler.Passes {
	return compiler.Passes{
		&compiler.RenameNumericEnumValues{},
	}
}

func (language *Language) NullableKinds() languages.NullableConfig {
	return languages.NullableConfig{
		Kinds:              []ast.Kind{ast.KindMap, ast.KindArray, ast.KindRef, ast.KindStruct},
		ProtectArrayAppend: true,
		AnyIsNullable:      true,
	}
}
