// SPDX-License-Identifier: GPL-2.0 OR BSD-3-Clause
//
// cs35l45-tables.c -- CS35L45 ALSA SoC audio driver
//
// Copyright 2019-2022 Cirrus Logic, Inc.
//
// Author: James Schulman <james.schulman@cirrus.com>

#include <linux/module.h>
#include <linux/regmap.h>

#include "cs35l45.h"

static const struct reg_sequence cs35l45_patch[] = {
	{ 0x00000040,			0x00000055 },
	{ 0x00000040,			0x000000AA },
	{ 0x00000044,			0x00000055 },
	{ 0x00000044,			0x000000AA },
	{ 0x00006480,			0x0830500A },
	{ 0x00007C60,			0x1000850B },
	{ CS35L45_BOOST_OV_CFG,		0x007000D0 },
	{ CS35L45_LDPM_CONFIG,		0x0001B636 },
	{ 0x00002C08,			0x00000009 },
	{ 0x00006850,			0x0A30FFC4 },
	{ 0x00003820,			0x00040100 },
	{ 0x00003824,			0x00000000 },
	{ 0x00007CFC,			0x62870004 },
	{ 0x00007C60,			0x1001850B },
	{ 0x00000040,			0x00000000 },
	{ 0x00000044,			0x00000000 },
	{ CS35L45_BOOST_CCM_CFG,	0xF0000003 },
	{ CS35L45_BOOST_DCM_CFG,	0x08710220 },
	{ CS35L45_ERROR_RELEASE,	0x00200000 },
};

int cs35l45_apply_patch(struct cs35l45_private *cs35l45)
{
	return regmap_register_patch(cs35l45->regmap, cs35l45_patch,
				     ARRAY_SIZE(cs35l45_patch));
}
EXPORT_SYMBOL_NS_GPL(cs35l45_apply_patch, SND_SOC_CS35L45);

static const struct reg_default cs35l45_defaults[] = {
	{ CS35L45_BLOCK_ENABLES,		0x00003323 },
	{ CS35L45_BLOCK_ENABLES2,		0x00000010 },
	{ CS35L45_SYNC_GPIO1,			0x00000007 },
	{ CS35L45_INTB_GPIO2_MCLK_REF,		0x00000005 },
	{ CS35L45_GPIO3,			0x00000005 },
	{ CS35L45_PWRMGT_CTL,			0x00000000 },
	{ CS35L45_WAKESRC_CTL,			0x00000008 },
	{ CS35L45_WKI2C_CTL,			0x00000030 },
	{ CS35L45_REFCLK_INPUT,			0x00000510 },
	{ CS35L45_GLOBAL_SAMPLE_RATE,		0x00000003 },
	{ CS35L45_ASP_ENABLES1,			0x00000000 },
	{ CS35L45_ASP_CONTROL1,			0x00000028 },
	{ CS35L45_ASP_CONTROL2,			0x18180200 },
	{ CS35L45_ASP_CONTROL3,			0x00000002 },
	{ CS35L45_ASP_FRAME_CONTROL1,		0x03020100 },
	{ CS35L45_ASP_FRAME_CONTROL2,		0x00000004 },
	{ CS35L45_ASP_FRAME_CONTROL5,		0x00000100 },
	{ CS35L45_ASP_DATA_CONTROL1,		0x00000018 },
	{ CS35L45_ASP_DATA_CONTROL5,		0x00000018 },
	{ CS35L45_DACPCM1_INPUT,		0x00000008 },
	{ CS35L45_ASPTX1_INPUT,			0x00000018 },
	{ CS35L45_ASPTX2_INPUT,			0x00000019 },
	{ CS35L45_ASPTX3_INPUT,			0x00000020 },
	{ CS35L45_ASPTX4_INPUT,			0x00000028 },
	{ CS35L45_ASPTX5_INPUT,			0x00000048 },
	{ CS35L45_DSP1_RX1_RATE,		0x00000001 },
	{ CS35L45_DSP1_RX2_RATE,		0x00000001 },
	{ CS35L45_DSP1_RX3_RATE,		0x00000001 },
	{ CS35L45_DSP1_RX4_RATE,		0x00000001 },
	{ CS35L45_DSP1_RX5_RATE,		0x00000001 },
	{ CS35L45_DSP1_RX6_RATE,		0x00000001 },
	{ CS35L45_DSP1_RX7_RATE,		0x00000001 },
	{ CS35L45_DSP1_RX8_RATE,		0x00000001 },
	{ CS35L45_DSP1_TX1_RATE,		0x00000001 },
	{ CS35L45_DSP1_TX2_RATE,		0x00000001 },
	{ CS35L45_DSP1_TX3_RATE,		0x00000001 },
	{ CS35L45_DSP1_TX4_RATE,		0x00000001 },
	{ CS35L45_DSP1_TX5_RATE,		0x00000001 },
	{ CS35L45_DSP1_TX6_RATE,		0x00000001 },
	{ CS35L45_DSP1_TX7_RATE,		0x00000001 },
	{ CS35L45_DSP1_TX8_RATE,		0x00000001 },
	{ CS35L45_DSP1RX1_INPUT,		0x00000008 },
	{ CS35L45_DSP1RX2_INPUT,		0x00000009 },
	{ CS35L45_DSP1RX3_INPUT,		0x00000018 },
	{ CS35L45_DSP1RX4_INPUT,		0x00000019 },
	{ CS35L45_DSP1RX5_INPUT,		0x00000020 },
	{ CS35L45_DSP1RX6_INPUT,		0x00000028 },
	{ CS35L45_DSP1RX7_INPUT,		0x0000003A },
	{ CS35L45_DSP1RX8_INPUT,		0x00000028 },
	{ CS35L45_AMP_PCM_CONTROL,		0x00100000 },
	{ CS35L45_IRQ1_CFG,			0x00000000 },
	{ CS35L45_IRQ1_MASK_1,			0xBFEFFFBF },
	{ CS35L45_IRQ1_MASK_2,			0xFFFFFFFF },
	{ CS35L45_IRQ1_MASK_3,			0xFFFF87FF },
	{ CS35L45_IRQ1_MASK_4,			0xF8FFFFFF },
	{ CS35L45_IRQ1_MASK_5,			0x0EF80000 },
	{ CS35L45_IRQ1_MASK_6,			0x00000000 },
	{ CS35L45_IRQ1_MASK_7,			0xFFFFFF78 },
	{ CS35L45_IRQ1_MASK_8,			0x00003FFF },
	{ CS35L45_IRQ1_MASK_9,			0x00000000 },
	{ CS35L45_IRQ1_MASK_10,			0x00000000 },
	{ CS35L45_IRQ1_MASK_11,			0x00000000 },
	{ CS35L45_IRQ1_MASK_12,			0x00000000 },
	{ CS35L45_IRQ1_MASK_13,			0x00000000 },
	{ CS35L45_IRQ1_MASK_14,			0x00000001 },
	{ CS35L45_IRQ1_MASK_15,			0x00000000 },
	{ CS35L45_IRQ1_MASK_16,			0x00000000 },
	{ CS35L45_IRQ1_MASK_17,			0x00000000 },
	{ CS35L45_IRQ1_MASK_18,			0x3FE5D0FF },
	{ CS35L45_GPIO1_CTRL1,			0x81000001 },
	{ CS35L45_GPIO2_CTRL1,			0x81000001 },
	{ CS35L45_GPIO3_CTRL1,			0x81000001 },
};

static bool cs35l45_readable_reg(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case CS35L45_DEVID ... CS35L45_OTPID:
	case CS35L45_SFT_RESET:
	case CS35L45_GLOBAL_ENABLES:
	case CS35L45_BLOCK_ENABLES:
	case CS35L45_BLOCK_ENABLES2:
	case CS35L45_ERROR_RELEASE:
	case CS35L45_SYNC_GPIO1:
	case CS35L45_INTB_GPIO2_MCLK_REF:
	case CS35L45_GPIO3:
	case CS35L45_PWRMGT_CTL:
	case CS35L45_WAKESRC_CTL:
	case CS35L45_WKI2C_CTL:
	case CS35L45_PWRMGT_STS:
	case CS35L45_REFCLK_INPUT:
	case CS35L45_GLOBAL_SAMPLE_RATE:
	case CS35L45_ASP_ENABLES1:
	case CS35L45_ASP_CONTROL1:
	case CS35L45_ASP_CONTROL2:
	case CS35L45_ASP_CONTROL3:
	case CS35L45_ASP_FRAME_CONTROL1:
	case CS35L45_ASP_FRAME_CONTROL2:
	case CS35L45_ASP_FRAME_CONTROL5:
	case CS35L45_ASP_DATA_CONTROL1:
	case CS35L45_ASP_DATA_CONTROL5:
	case CS35L45_DACPCM1_INPUT:
	case CS35L45_ASPTX1_INPUT:
	case CS35L45_ASPTX2_INPUT:
	case CS35L45_ASPTX3_INPUT:
	case CS35L45_ASPTX4_INPUT:
	case CS35L45_ASPTX5_INPUT:
	case CS35L45_DSP1RX1_INPUT:
	case CS35L45_DSP1RX2_INPUT:
	case CS35L45_DSP1RX3_INPUT:
	case CS35L45_DSP1RX4_INPUT:
	case CS35L45_DSP1RX5_INPUT:
	case CS35L45_DSP1RX6_INPUT:
	case CS35L45_DSP1RX7_INPUT:
	case CS35L45_DSP1RX8_INPUT:
	case CS35L45_AMP_PCM_CONTROL:
	case CS35L45_AMP_PCM_HPF_TST:
	case CS35L45_IRQ1_CFG:
	case CS35L45_IRQ1_STATUS:
	case CS35L45_IRQ1_EINT_1 ... CS35L45_IRQ1_EINT_18:
	case CS35L45_IRQ1_STS_1 ... CS35L45_IRQ1_STS_18:
	case CS35L45_IRQ1_MASK_1 ... CS35L45_IRQ1_MASK_18:
	case CS35L45_GPIO_STATUS1:
	case CS35L45_GPIO1_CTRL1:
	case CS35L45_GPIO2_CTRL1:
	case CS35L45_GPIO3_CTRL1:
	case CS35L45_DSP_MBOX_1:
	case CS35L45_DSP_MBOX_2:
	case CS35L45_DSP_VIRT1_MBOX_1 ... CS35L45_DSP_VIRT1_MBOX_4:
	case CS35L45_DSP_VIRT2_MBOX_1 ... CS35L45_DSP_VIRT2_MBOX_4:
	case CS35L45_DSP1_SYS_ID:
	case CS35L45_DSP1_CLOCK_FREQ:
	case CS35L45_DSP1_RX1_RATE:
	case CS35L45_DSP1_RX2_RATE:
	case CS35L45_DSP1_RX3_RATE:
	case CS35L45_DSP1_RX4_RATE:
	case CS35L45_DSP1_RX5_RATE:
	case CS35L45_DSP1_RX6_RATE:
	case CS35L45_DSP1_RX7_RATE:
	case CS35L45_DSP1_RX8_RATE:
	case CS35L45_DSP1_TX1_RATE:
	case CS35L45_DSP1_TX2_RATE:
	case CS35L45_DSP1_TX3_RATE:
	case CS35L45_DSP1_TX4_RATE:
	case CS35L45_DSP1_TX5_RATE:
	case CS35L45_DSP1_TX6_RATE:
	case CS35L45_DSP1_TX7_RATE:
	case CS35L45_DSP1_TX8_RATE:
	case CS35L45_DSP1_SCRATCH1:
	case CS35L45_DSP1_SCRATCH2:
	case CS35L45_DSP1_SCRATCH3:
	case CS35L45_DSP1_SCRATCH4:
	case CS35L45_DSP1_CCM_CORE_CONTROL:
	case CS35L45_DSP1_XMEM_PACK_0 ... CS35L45_DSP1_XMEM_PACK_4607:
	case CS35L45_DSP1_XMEM_UNPACK32_0 ... CS35L45_DSP1_XMEM_UNPACK32_3071:
	case CS35L45_DSP1_XMEM_UNPACK24_0 ... CS35L45_DSP1_XMEM_UNPACK24_6143:
	case CS35L45_DSP1_YMEM_PACK_0 ... CS35L45_DSP1_YMEM_PACK_1532:
	case CS35L45_DSP1_YMEM_UNPACK32_0 ... CS35L45_DSP1_YMEM_UNPACK32_1022:
	case CS35L45_DSP1_YMEM_UNPACK24_0 ... CS35L45_DSP1_YMEM_UNPACK24_2043:
	case CS35L45_DSP1_PMEM_0 ... CS35L45_DSP1_PMEM_3834:
		return true;
	default:
		return false;
	}
}

static bool cs35l45_volatile_reg(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case CS35L45_DEVID ... CS35L45_OTPID:
	case CS35L45_SFT_RESET:
	case CS35L45_GLOBAL_ENABLES:
	case CS35L45_ERROR_RELEASE:
	case CS35L45_AMP_PCM_HPF_TST:	/* not cachable */
	case CS35L45_PWRMGT_STS:
	case CS35L45_IRQ1_STATUS:
	case CS35L45_IRQ1_EINT_1 ... CS35L45_IRQ1_EINT_18:
	case CS35L45_IRQ1_STS_1 ... CS35L45_IRQ1_STS_18:
	case CS35L45_GPIO_STATUS1:
	case CS35L45_DSP_MBOX_1:
	case CS35L45_DSP_MBOX_2:
	case CS35L45_DSP_VIRT1_MBOX_1 ... CS35L45_DSP_VIRT1_MBOX_4:
	case CS35L45_DSP_VIRT2_MBOX_1 ... CS35L45_DSP_VIRT2_MBOX_4:
	case CS35L45_DSP1_SYS_ID:
	case CS35L45_DSP1_CLOCK_FREQ:
	case CS35L45_DSP1_SCRATCH1:
	case CS35L45_DSP1_SCRATCH2:
	case CS35L45_DSP1_SCRATCH3:
	case CS35L45_DSP1_SCRATCH4:
	case CS35L45_DSP1_CCM_CORE_CONTROL:
	case CS35L45_DSP1_XMEM_PACK_0 ... CS35L45_DSP1_XMEM_PACK_4607:
	case CS35L45_DSP1_XMEM_UNPACK32_0 ... CS35L45_DSP1_XMEM_UNPACK32_3071:
	case CS35L45_DSP1_XMEM_UNPACK24_0 ... CS35L45_DSP1_XMEM_UNPACK24_6143:
	case CS35L45_DSP1_YMEM_PACK_0 ... CS35L45_DSP1_YMEM_PACK_1532:
	case CS35L45_DSP1_YMEM_UNPACK32_0 ... CS35L45_DSP1_YMEM_UNPACK32_1022:
	case CS35L45_DSP1_YMEM_UNPACK24_0 ... CS35L45_DSP1_YMEM_UNPACK24_2043:
	case CS35L45_DSP1_PMEM_0 ... CS35L45_DSP1_PMEM_3834:
		return true;
	default:
		return false;
	}
}

const struct regmap_config cs35l45_i2c_regmap = {
	.reg_bits = 32,
	.val_bits = 32,
	.reg_stride = 4,
	.reg_format_endian = REGMAP_ENDIAN_BIG,
	.val_format_endian = REGMAP_ENDIAN_BIG,
	.max_register = CS35L45_LASTREG,
	.reg_defaults = cs35l45_defaults,
	.num_reg_defaults = ARRAY_SIZE(cs35l45_defaults),
	.volatile_reg = cs35l45_volatile_reg,
	.readable_reg = cs35l45_readable_reg,
	.cache_type = REGCACHE_RBTREE,
};
EXPORT_SYMBOL_NS_GPL(cs35l45_i2c_regmap, SND_SOC_CS35L45);

const struct regmap_config cs35l45_spi_regmap = {
	.reg_bits = 32,
	.val_bits = 32,
	.pad_bits = 16,
	.reg_stride = 4,
	.reg_format_endian = REGMAP_ENDIAN_BIG,
	.val_format_endian = REGMAP_ENDIAN_BIG,
	.max_register = CS35L45_LASTREG,
	.reg_defaults = cs35l45_defaults,
	.num_reg_defaults = ARRAY_SIZE(cs35l45_defaults),
	.volatile_reg = cs35l45_volatile_reg,
	.readable_reg = cs35l45_readable_reg,
	.cache_type = REGCACHE_RBTREE,
};
EXPORT_SYMBOL_NS_GPL(cs35l45_spi_regmap, SND_SOC_CS35L45);

static const struct {
	u8 cfg_id;
	u32 freq;
} cs35l45_pll_refclk_freq[] = {
	{ 0x0C,   128000 },
	{ 0x0F,   256000 },
	{ 0x11,   384000 },
	{ 0x12,   512000 },
	{ 0x15,   768000 },
	{ 0x17,  1024000 },
	{ 0x19,  1411200 },
	{ 0x1B,  1536000 },
	{ 0x1C,  2116800 },
	{ 0x1D,  2048000 },
	{ 0x1E,  2304000 },
	{ 0x1F,  2822400 },
	{ 0x21,  3072000 },
	{ 0x23,  4233600 },
	{ 0x24,  4096000 },
	{ 0x25,  4608000 },
	{ 0x26,  5644800 },
	{ 0x27,  6000000 },
	{ 0x28,  6144000 },
	{ 0x29,  6350400 },
	{ 0x2A,  6912000 },
	{ 0x2D,  7526400 },
	{ 0x2E,  8467200 },
	{ 0x2F,  8192000 },
	{ 0x30,  9216000 },
	{ 0x31, 11289600 },
	{ 0x33, 12288000 },
	{ 0x37, 16934400 },
	{ 0x38, 18432000 },
	{ 0x39, 22579200 },
	{ 0x3B, 24576000 },
};

unsigned int cs35l45_get_clk_freq_id(unsigned int freq)
{
	int i;

	if (freq == 0)
		return -EINVAL;

	for (i = 0; i < ARRAY_SIZE(cs35l45_pll_refclk_freq); ++i) {
		if (cs35l45_pll_refclk_freq[i].freq == freq)
			return cs35l45_pll_refclk_freq[i].cfg_id;
	}

	return -EINVAL;
}
EXPORT_SYMBOL_NS_GPL(cs35l45_get_clk_freq_id, SND_SOC_CS35L45);
