/*
 * livepatch_bsc1245795
 *
 * Fix for CVE-2025-21756, bsc#1245795
 *
 *  Upstream commit:
 *  fcdd2242c023 ("vsock: Keep the binding until socket destruction")
 *  78dafe1cf3af ("vsock: Orphan socket after transport release")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  Not affected
 *
 *  SLE15-SP4 and -SP5 commit:
 *  a3adf03fb524cd6d0ea050caf6d853cb9eb63e13
 *  7e393283772ec2d48469ea0ca4490ddb20243698
 *
 *  SLE15-SP6 commit:
 *  545191eac053f51d1e1c31c29893eba75006708f
 *
 *  SLE MICRO-6-0 commit:
 *  545191eac053f51d1e1c31c29893eba75006708f
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Ali Abdallah <ali.abdallah@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/vmw_vsock/af_vsock.c */
#include <linux/compat.h>
#include <linux/types.h>
#include <linux/bitops.h>
#include <linux/cred.h>
#include <linux/errqueue.h>
#include <linux/init.h>

#include <linux/kernel.h>
#include <linux/sched/signal.h>
#include <linux/kmod.h>
#include <linux/list.h>

/* klp-ccp: from net/vmw_vsock/af_vsock.c */
#include <linux/list.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/net.h>
#include <linux/poll.h>
#include <linux/random.h>
#include <linux/skbuff.h>
#include <linux/smp.h>
#include <linux/socket.h>
#include <linux/stddef.h>
#include <linux/unistd.h>
#include <linux/wait.h>
#include <linux/workqueue.h>
#include <net/sock.h>
#include <net/af_vsock.h>

/* klp-ccp: from include/net/af_vsock.h */
static void (*klpe_vsock_remove_bound)(struct vsock_sock *vsk);
static void (*klpe_vsock_remove_connected)(struct vsock_sock *vsk);
static struct sock *(*klpe_vsock_dequeue_accept)(struct sock *listener);

void klpp_vsock_remove_sock(struct vsock_sock *vsk)
{
	/* Transport reassignment must not remove the binding. */
	if (sock_flag(sk_vsock(vsk), SOCK_DEAD))
		(*klpe_vsock_remove_bound)(vsk);

	(*klpe_vsock_remove_connected)(vsk);
}

static bool sock_type_connectible(u16 type)
{
	return (type == SOCK_STREAM) || (type == SOCK_SEQPACKET);
}

void klpp___vsock_release(struct sock *sk, int level)
{
	if (sk) {
		struct sock *pending;
		struct vsock_sock *vsk;

		vsk = vsock_sk(sk);
		pending = NULL;	/* Compiler warning. */

		/* When "level" is SINGLE_DEPTH_NESTING, use the nested
		 * version to avoid the warning "possible recursive locking
		 * detected". When "level" is 0, lock_sock_nested(sk, level)
		 * is the same as lock_sock(sk).
		 */
		lock_sock_nested(sk, level);

		/* Indicate to vsock_remove_sock() that the socket is being released and
		 * can be removed from the bound_table. Unlike transport reassignment
		 * case, where the socket must remain bound despite vsock_remove_sock()
		 * being called from the transport release() callback.
		 */
		sock_set_flag(sk, SOCK_DEAD);

		if (vsk->transport)
			vsk->transport->release(vsk);
		else if (sock_type_connectible(sk->sk_type))
			klpp_vsock_remove_sock(vsk);

		sock_orphan(sk);
		sk->sk_shutdown = SHUTDOWN_MASK;

		skb_queue_purge(&sk->sk_receive_queue);

		/* Clean up any sockets that never were accepted. */
		while ((pending = (*klpe_vsock_dequeue_accept)(sk)) != NULL) {
			klpp___vsock_release(pending, SINGLE_DEPTH_NESTING);
			sock_put(pending);
		}

		release_sock(sk);
		sock_put(sk);
	}
}


#include "livepatch_bsc1245795.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "vsock"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "vsock_dequeue_accept", (void *)&klpe_vsock_dequeue_accept,
	  "vsock" },
	{ "vsock_remove_bound", (void *)&klpe_vsock_remove_bound, "vsock" },
	{ "vsock_remove_connected", (void *)&klpe_vsock_remove_connected,
	  "vsock" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1245795_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void livepatch_bsc1245795_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}
