/*
 * livepatch_bsc1235769
 *
 * Fix for CVE-2024-57793, bsc#1235769
 *
 *  Upstream commit:
 *  27834971f616 ("virt: tdx-guest: Just leak decrypted memory on unrecoverable errors")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  Not affected
 *
 *  SLE15-SP4 and -SP5 commit:
 *  Not affected
 *
 *  SLE15-SP6 commit:
 *  0fbd2e184ff4d63a526332da09f3b93f3abfd722
 *
 *  SLE MICRO-6-0 commit:
 *  0fbd2e184ff4d63a526332da09f3b93f3abfd722
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Fernando Gonzalez <fernando.gonzalez@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_TDX_GUEST_DRIVER)

#if !IS_MODULE(CONFIG_TDX_GUEST_DRIVER)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from drivers/virt/coco/tdx-guest/tdx-guest.c */
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/mod_devicetable.h>
#include <linux/string.h>
#include <linux/uaccess.h>
#include <linux/set_memory.h>
#include <linux/tsm.h>

/* klp-ccp: from include/linux/tsm.h */
static int (*klpe_tsm_register)(const struct tsm_ops *ops, void *priv,
		 const struct config_item_type *type);

/* klp-ccp: from drivers/virt/coco/tdx-guest/tdx-guest.c */
#include <linux/sizes.h>
#include <asm/cpu_device_id.h>

#define GET_QUOTE_BUF_SIZE		SZ_8K

static void *(*klpe_quote_data);

static void (*klpe_free_quote_buf)(void *buf);

static void *klpp_alloc_quote_buf(void)
{
	size_t len = PAGE_ALIGN(GET_QUOTE_BUF_SIZE);
	unsigned int count = len >> PAGE_SHIFT;
	void *addr;

	addr = alloc_pages_exact(len, GFP_KERNEL | __GFP_ZERO);
	if (!addr)
		return NULL;

	if (set_memory_decrypted((unsigned long)addr, count))
		return NULL;

	return addr;
}

static struct miscdevice (*klpe_tdx_misc_dev);

static const struct x86_cpu_id (*klpe_tdx_guest_ids)[];

static const struct tsm_ops (*klpe_tdx_tsm_ops);

int klpp_tdx_guest_init(void)
{
	int ret;

	if (!x86_match_cpu((*klpe_tdx_guest_ids)))
		return -ENODEV;

	ret = misc_register(&(*klpe_tdx_misc_dev));
	if (ret)
		return ret;

	(*klpe_quote_data) = klpp_alloc_quote_buf();
	if (!(*klpe_quote_data)) {
		pr_err("Failed to allocate Quote buffer\n");
		ret = -ENOMEM;
		goto free_misc;
	}

	ret = (*klpe_tsm_register)(&(*klpe_tdx_tsm_ops), NULL, NULL);
	if (ret)
		goto free_quote;

	return 0;

free_quote:
	(*klpe_free_quote_buf)((*klpe_quote_data));
free_misc:
	misc_deregister(&(*klpe_tdx_misc_dev));

	return ret;
}


#include "livepatch_bsc1235769.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "tdx_guest"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "free_quote_buf", (void *)&klpe_free_quote_buf, "tdx_guest" },
	{ "quote_data", (void *)&klpe_quote_data, "tdx_guest" },
	{ "tdx_guest_ids", (void *)&klpe_tdx_guest_ids, "tdx_guest" },
	{ "tdx_misc_dev", (void *)&klpe_tdx_misc_dev, "tdx_guest" },
	{ "tdx_tsm_ops", (void *)&klpe_tdx_tsm_ops, "tdx_guest" },
	{ "tsm_register", (void *)&klpe_tsm_register, "tsm" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1235769_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void livepatch_bsc1235769_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}

#endif /* IS_ENABLED(CONFIG_TDX_GUEST_DRIVER) */
