/*
 * livepatch_bsc1231353
 *
 * Fix for bsc#1231353
 *
 *  Upstream commit:
 *  none yet
 *
 *  SLE12-SP5 commit:
 *  none yet
 *
 *  SLE15-SP2 and -SP3 commit:
 *  none yet
 *
 *  SLE15-SP4 commit:
 *  2b5943c82e10e7a1b35790f7bbc8e60e6ab37b19
 *
 *  SLE15-SP5 commit:
 *  none yet
 *
 *  SLE15-SP6 commit:
 *  none yet
 *
 *
 *  Copyright (c) 2024 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if !IS_MODULE(CONFIG_NFS_FS)
#error "Live patch supports only CONFIG_NFS_FS=m"
#endif

/* klp-ccp: from fs/nfs/super.c */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/time.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/unistd.h>
#include <linux/sunrpc/clnt.h>

/* klp-ccp: from include/linux/sunrpc/debug.h */
#if IS_ENABLED(CONFIG_SUNRPC_DEBUG)

static unsigned int		(*klpe_nfs_debug);

# define klpr_dfprintk(fac, fmt, ...)					\
do {									\
	klpr_ifdebug(fac)							\
		printk(KERN_DEFAULT fmt, ##__VA_ARGS__);		\
} while (0)

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

/* klp-ccp: from include/linux/sunrpc/auth.h */
static int			(*klpe_rpcauth_get_gssinfo)(rpc_authflavor_t,
				struct rpcsec_gss_info *);

/* klp-ccp: from fs/nfs/super.c */
#include <linux/sunrpc/addr.h>
#include <linux/sunrpc/stats.h>
#include <linux/nfs_fs.h>

/* klp-ccp: from include/linux/nfs_fs.h */
#ifdef CONFIG_NFS_DEBUG
# define NFS_DEBUG
#else
#error "klp-ccp: non-taken branch"
#endif

# ifdef NFS_DEBUG
#  define klpr_ifdebug(fac)		if (unlikely((*klpe_nfs_debug) & NFSDBG_##fac))
#  define NFS_IFDEBUG(x)	x
# else
#error "klp-ccp: non-taken branch"
#endif

/* klp-ccp: from fs/nfs/super.c */
#include <linux/seq_file.h>
#include <linux/mount.h>
#include <linux/inet.h>
#include <linux/in6.h>
#include <linux/slab.h>
#include <net/ipv6.h>
#include <linux/netdevice.h>
#include <linux/nfs_xdr.h>
#include <linux/magic.h>
#include <linux/nsproxy.h>
#include <linux/rcupdate.h>
#include <linux/uaccess.h>

/* klp-ccp: from fs/nfs/nfs4_fs.h */
#if IS_ENABLED(CONFIG_NFS_V4)

#include <linux/seqlock.h>

#else /* CONFIG_NFS_V4 */
#error "klp-ccp: non-taken branch"
#endif /* CONFIG_NFS_V4 */

/* klp-ccp: from fs/nfs/callback.h */
#include <linux/sunrpc/svc.h>
/* klp-ccp: from fs/nfs/iostat.h */
#include <linux/percpu.h>
#include <linux/cache.h>
/* klp-ccp: from fs/nfs/internal.h */
#include <linux/fs_context.h>
#include <linux/security.h>
#include <linux/crc32.h>
#include <linux/sunrpc/addr.h>

/* klp-ccp: from fs/nfs/internal.h */
#include <linux/wait_bit.h>

#define NFS_MAX_SECFLAVORS	(12)

#define NFS_UNSPEC_PORT		(-1)

#define NFS_UNSPEC_RETRANS	(UINT_MAX)
#define NFS_UNSPEC_TIMEO	(UINT_MAX)

struct nfs_fs_context {
	bool			internal;
	bool			skip_reconfig_option_check;
	bool			need_mount;
	bool			sloppy;
	unsigned int		flags;		/* NFS{,4}_MOUNT_* flags */
	unsigned int		rsize, wsize;
	unsigned int		timeo, retrans;
	unsigned int		acregmin, acregmax;
	unsigned int		acdirmin, acdirmax;
	unsigned int		namlen;
	unsigned int		options;
	unsigned int		bsize;
	struct nfs_auth_info	auth_info;
	rpc_authflavor_t	selected_flavor;
	char			*client_address;
	unsigned int		version;
	unsigned int		minorversion;
	char			*fscache_uniq;
	unsigned short		protofamily;
	unsigned short		mountfamily;
	bool			has_sec_mnt_opts;

	struct {
		union {
			struct sockaddr	address;
			struct sockaddr_storage	_address;
		};
		size_t			addrlen;
		char			*hostname;
		u32			version;
		int			port;
		unsigned short		protocol;
	} mount_server;

	struct {
		union {
			struct sockaddr	address;
			struct sockaddr_storage	_address;
		};
		size_t			addrlen;
		char			*hostname;
		char			*export_path;
		int			port;
		unsigned short		protocol;
		unsigned short		nconnect;
		unsigned short		export_path_len;
	} nfs_server;

	struct nfs_fh		*mntfh;
	struct nfs_server	*server;
	struct nfs_subversion	*nfs_mod;

	/* Information for a cloned mount. */
	struct nfs_clone_mount {
		struct super_block	*sb;
		struct dentry		*dentry;
		struct nfs_fattr	*fattr;
		unsigned int		inherited_bsize;
	} clone_data;
};

static inline struct nfs_fs_context *nfs_fc2context(const struct fs_context *fc)
{
	return fc->fs_private;
}

struct nfs_mount_request {
	struct sockaddr		*sap;
	size_t			salen;
	char			*hostname;
	char			*dirpath;
	u32			version;
	unsigned short		protocol;
	struct nfs_fh		*fh;
	int			noresvport;
	unsigned int		*auth_flav_len;
	rpc_authflavor_t	*auth_flavs;
	struct net		*net;
};

static int (*klpe_nfs_mount)(struct nfs_mount_request *info, int timeo, int retrans);

static bool (*klpe_nfs_auth_info_match)(const struct nfs_auth_info *, rpc_authflavor_t);
int klpp_nfs_try_get_tree(struct fs_context *);
static int (*klpe_nfs_get_tree_common)(struct fs_context *);

static inline void nfs_set_port(struct sockaddr *sap, int *port,
				const unsigned short default_port)
{
	if (*port == NFS_UNSPEC_PORT)
		*port = default_port;

	rpc_set_port(sap, *port);
}

/* klp-ccp: from fs/nfs/fscache.h */
#include <linux/nfs_fs.h>
#include <linux/nfs_mount.h>
#include <linux/nfs4_mount.h>
/* klp-ccp: from fs/nfs/pnfs.h */
#include <linux/refcount.h>
#include <linux/nfs_fs.h>
#include <linux/nfs_page.h>
#include <linux/workqueue.h>
/* klp-ccp: from fs/nfs/nfs.h */
#include <linux/fs.h>
#include <linux/sunrpc/sched.h>
#include <linux/nfs_xdr.h>

struct nfs_subversion {
	struct module *owner;	/* THIS_MODULE pointer */
	struct file_system_type *nfs_fs;	/* NFS filesystem type */
	const struct rpc_version *rpc_vers;	/* NFS version information */
	const struct nfs_rpc_ops *rpc_ops;	/* NFS operations */
	const struct super_operations *sops;	/* NFS Super operations */
	const struct xattr_handler **xattr;	/* NFS xattr handlers */
	struct list_head list;		/* List of NFS versions */
};

/* klp-ccp: from fs/nfs/super.c */
static int klpr_nfs_verify_authflavors(struct nfs_fs_context *ctx,
				  rpc_authflavor_t *server_authlist,
				  unsigned int count)
{
	rpc_authflavor_t flavor = RPC_AUTH_MAXFLAVOR;
	bool found_auth_null = false;
	unsigned int i;

	/*
	 * If the sec= mount option is used, the specified flavor or AUTH_NULL
	 * must be in the list returned by the server.
	 *
	 * AUTH_NULL has a special meaning when it's in the server list - it
	 * means that the server will ignore the rpc creds, so any flavor
	 * can be used but still use the sec= that was specified.
	 *
	 * Note also that the MNT procedure in MNTv1 does not return a list
	 * of supported security flavors. In this case, nfs_mount() fabricates
	 * a security flavor list containing just AUTH_NULL.
	 */
	for (i = 0; i < count; i++) {
		flavor = server_authlist[i];

		if ((*klpe_nfs_auth_info_match)(&ctx->auth_info, flavor))
			goto out;

		if (flavor == RPC_AUTH_NULL)
			found_auth_null = true;
	}

	if (found_auth_null) {
		flavor = ctx->auth_info.flavors[0];
		goto out;
	}

	klpr_dfprintk(MOUNT,
		 "NFS: specified auth flavors not supported by server\n");
	return -EACCES;

out:
	ctx->selected_flavor = flavor;
	klpr_dfprintk(MOUNT, "NFS: using auth flavor %u\n", ctx->selected_flavor);
	return 0;
}

static int klpp_nfs_request_mount(struct fs_context *fc,
			     struct nfs_fh *root_fh,
			     rpc_authflavor_t *server_authlist,
			     unsigned int *server_authlist_len)
{
	struct nfs_fs_context *ctx = nfs_fc2context(fc);
	struct nfs_mount_request request = {
		.sap		= (struct sockaddr *)
						&ctx->mount_server.address,
		.dirpath	= ctx->nfs_server.export_path,
		.protocol	= ctx->mount_server.protocol,
		.fh		= root_fh,
		.noresvport	= ctx->flags & NFS_MOUNT_NORESVPORT,
		.auth_flav_len	= server_authlist_len,
		.auth_flavs	= server_authlist,
		.net		= fc->net_ns,
	};
	int status;

	if (ctx->mount_server.version == 0) {
		switch (ctx->version) {
			default:
				ctx->mount_server.version = NFS_MNT3_VERSION;
				break;
			case 2:
				ctx->mount_server.version = NFS_MNT_VERSION;
		}
	}
	request.version = ctx->mount_server.version;

	if (ctx->mount_server.hostname)
		request.hostname = ctx->mount_server.hostname;
	else
		request.hostname = ctx->nfs_server.hostname;

	/*
	 * Construct the mount server's address.
	 */
	if (ctx->mount_server.address.sa_family == AF_UNSPEC) {
		memcpy(request.sap, &ctx->nfs_server.address,
		       ctx->nfs_server.addrlen);
		ctx->mount_server.addrlen = ctx->nfs_server.addrlen;
	}
	request.salen = ctx->mount_server.addrlen;
	nfs_set_port(request.sap, &ctx->mount_server.port, 0);

	/*
	 * Now ask the mount server to map our export path
	 * to a file handle.
	 */
	if ((request.protocol == XPRT_TRANSPORT_UDP) ==
	    !(ctx->flags & NFS_MOUNT_TCP))
		/*
		 * NFS protocol and mount protocol are both UDP or neither UDP
		 * so timeouts are compatible.  Use NFS timeouts for MOUNT
		 */
		status = (*klpe_nfs_mount)(&request, ctx->timeo, ctx->retrans);
	else
		status = (*klpe_nfs_mount)(&request, NFS_UNSPEC_TIMEO, NFS_UNSPEC_RETRANS);
	if (status != 0) {
		klpr_dfprintk(MOUNT, "NFS: unable to mount server %s, error %d\n",
				request.hostname, status);
		return status;
	}

	return 0;
}

static struct nfs_server *klpp_nfs_try_mount_request(struct fs_context *fc)
{
	struct nfs_fs_context *ctx = nfs_fc2context(fc);
	int status;
	unsigned int i;
	bool tried_auth_unix = false;
	bool auth_null_in_list = false;
	struct nfs_server *server = ERR_PTR(-EACCES);
	rpc_authflavor_t authlist[NFS_MAX_SECFLAVORS];
	unsigned int authlist_len = ARRAY_SIZE(authlist);

	status = klpp_nfs_request_mount(fc, ctx->mntfh, authlist, &authlist_len);
	if (status)
		return ERR_PTR(status);

	/*
	 * Was a sec= authflavor specified in the options? First, verify
	 * whether the server supports it, and then just try to use it if so.
	 */
	if (ctx->auth_info.flavor_len > 0) {
		status = klpr_nfs_verify_authflavors(ctx, authlist, authlist_len);
		klpr_dfprintk(MOUNT, "NFS: using auth flavor %u\n",
			 ctx->selected_flavor);
		if (status)
			return ERR_PTR(status);
		return ctx->nfs_mod->rpc_ops->create_server(fc);
	}

	/*
	 * No sec= option was provided. RFC 2623, section 2.7 suggests we
	 * SHOULD prefer the flavor listed first. However, some servers list
	 * AUTH_NULL first. Avoid ever choosing AUTH_NULL.
	 */
	for (i = 0; i < authlist_len; ++i) {
		rpc_authflavor_t flavor;
		struct rpcsec_gss_info info;

		flavor = authlist[i];
		switch (flavor) {
		case RPC_AUTH_UNIX:
			tried_auth_unix = true;
			break;
		case RPC_AUTH_NULL:
			auth_null_in_list = true;
			continue;
		default:
			if ((*klpe_rpcauth_get_gssinfo)(flavor, &info) != 0)
				continue;
			break;
		}
		klpr_dfprintk(MOUNT, "NFS: attempting to use auth flavor %u\n", flavor);
		ctx->selected_flavor = flavor;
		server = ctx->nfs_mod->rpc_ops->create_server(fc);
		if (!IS_ERR(server))
			return server;
	}

	/*
	 * Nothing we tried so far worked. At this point, give up if we've
	 * already tried AUTH_UNIX or if the server's list doesn't contain
	 * AUTH_NULL
	 */
	if (tried_auth_unix || !auth_null_in_list)
		return server;

	/* Last chance! Try AUTH_UNIX */
	klpr_dfprintk(MOUNT, "NFS: attempting to use auth flavor %u\n", RPC_AUTH_UNIX);
	ctx->selected_flavor = RPC_AUTH_UNIX;
	return ctx->nfs_mod->rpc_ops->create_server(fc);
}

int klpp_nfs_try_get_tree(struct fs_context *fc)
{
	struct nfs_fs_context *ctx = nfs_fc2context(fc);

	if (ctx->need_mount)
		ctx->server = klpp_nfs_try_mount_request(fc);
	else
		ctx->server = ctx->nfs_mod->rpc_ops->create_server(fc);

	return (*klpe_nfs_get_tree_common)(fc);
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1231353.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "nfs"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "nfs_auth_info_match", (void *)&klpe_nfs_auth_info_match, "nfs" },
	{ "nfs_get_tree_common", (void *)&klpe_nfs_get_tree_common, "nfs" },
	{ "nfs_mount", (void *)&klpe_nfs_mount, "nfs" },
	{ "nfs_debug", (void *)&klpe_nfs_debug, "sunrpc" },
	{ "rpcauth_get_gssinfo", (void *)&klpe_rpcauth_get_gssinfo, "sunrpc" },
};

static int livepatch_bsc1231353_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1231353_module_nb = {
	.notifier_call = livepatch_bsc1231353_module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1231353_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&livepatch_bsc1231353_module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LIVEPATCHED_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						  ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&livepatch_bsc1231353_module_nb);

	module_put(mod);
	return ret;
}

void livepatch_bsc1231353_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1231353_module_nb);
}
