/*
 * livepatch_bsc1242579
 *
 * Fix for CVE-2025-21999, bsc#1242579
 *
 *  Upstream commit:
 *  654b33ada4ab ("proc: fix UAF in proc_get_inode()")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  Not affected
 *
 *  SLE15-SP4 and -SP5 commit:
 *  8fb79443f171c25aa2bc5c937b7a3d1b47231ba2
 *
 *  SLE15-SP6 commit:
 *  15e810e82439d49cdc4d92417d8417c077c79cba
 *
 *  SLE MICRO-6-0 commit:
 *  15e810e82439d49cdc4d92417d8417c077c79cba
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Ali Abdallah <ali.abdallah@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from fs/proc/inode.c */
#include <linux/cache.h>
#include <linux/time.h>
#include <linux/proc_fs.h>

/* klp-ccp: from include/linux/fs.h */
static void (*klpe_make_empty_dir_inode)(struct inode *inode);

/* klp-ccp: from fs/proc/inode.c */
#include <linux/kernel.h>
#include <linux/pid_namespace.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/completion.h>

/* klp-ccp: from include/linux/sysctl.h */
#define _LINUX_SYSCTL_H

/* klp-ccp: from fs/proc/inode.c */
#include <linux/printk.h>

#include <linux/limits.h>
#include <linux/init.h>

#include <linux/sysctl.h>

#include <linux/uaccess.h>

/* klp-ccp: from fs/proc/internal.h */
#include <linux/proc_fs.h>

#include <linux/refcount.h>
#include <linux/spinlock.h>
#include <linux/atomic.h>

#include <linux/sched/coredump.h>

struct proc_dir_entry {
    /*
     * number of callers into module in progress;
     * negative -> it's going away RSN
     */
    atomic_t in_use;
    refcount_t refcnt;
    struct list_head pde_openers;    /* who did ->open, but not ->release */
    /* protects ->pde_openers and all struct pde_opener instances */
    spinlock_t pde_unload_lock;
    struct completion *pde_unload_completion;
    const struct inode_operations *proc_iops;
    const struct file_operations *proc_fops;
    const struct dentry_operations *proc_dops;
    union {
        const struct seq_operations *seq_ops;
        int (*single_show)(struct seq_file *, void *);
    };
    proc_write_t write;
    void *data;
    unsigned int state_size;
    unsigned int low_ino;
    nlink_t nlink;
    kuid_t uid;
    kgid_t gid;
    loff_t size;
    struct proc_dir_entry *parent;
    struct rb_root subdir;
    struct rb_node subdir_node;
    char *name;
    umode_t mode;
    u8 namelen;
    char inline_name[];
} __randomize_layout;

union proc_op {
    int (*proc_get_link)(struct dentry *, struct path *);
    int (*proc_show)(struct seq_file *m,
        struct pid_namespace *ns, struct pid *pid,
        struct task_struct *task);
    const char *lsm;
};

struct proc_inode {
    struct pid *pid;
    unsigned int fd;
    union proc_op op;
    struct proc_dir_entry *pde;
    struct ctl_table_header *sysctl;
    struct ctl_table *sysctl_entry;
    struct hlist_node sysctl_inodes;
    const struct proc_ns_operations *ns_ops;
    struct inode vfs_inode;
} __randomize_layout;

static inline struct proc_inode *PROC_I(const struct inode *inode)
{
    return container_of(inode, struct proc_inode, vfs_inode);
}

static void (*klpe_pde_put)(struct proc_dir_entry *);

static inline bool is_empty_pde(const struct proc_dir_entry *pde)
{
    return S_ISDIR(pde->mode) && !pde->proc_iops;
}

/* klp-ccp: from fs/proc/inode.c */
static inline int use_pde(struct proc_dir_entry *pde)
{
    return likely(atomic_inc_unless_negative(&pde->in_use));
}

enum {BIAS = -1U<<31};

#ifdef CONFIG_COMPAT
static void unuse_pde(struct proc_dir_entry *pde)
{
    if (unlikely(atomic_dec_return(&pde->in_use) == BIAS))
        complete(pde->pde_unload_completion);
}
#endif

static const struct file_operations (*klpe_proc_reg_file_ops);

#ifdef CONFIG_COMPAT
static const struct file_operations (*klpe_proc_reg_file_ops_no_compat);
#endif

struct inode *klpp_proc_get_inode(struct super_block *sb, struct proc_dir_entry *de)
{
    struct inode *inode = new_inode(sb);

    if (inode) {
        inode->i_ino = de->low_ino;
        inode->i_mtime = inode->i_atime = inode->i_ctime = current_time(inode);
        PROC_I(inode)->pde = de;

        if (is_empty_pde(de)) {
            (*klpe_make_empty_dir_inode)(inode);
            return inode;
        }
        if (de->mode) {
            inode->i_mode = de->mode;
            inode->i_uid = de->uid;
            inode->i_gid = de->gid;
        }
        if (de->size)
            inode->i_size = de->size;
        if (de->nlink)
            set_nlink(inode, de->nlink);
        WARN_ON(!de->proc_iops);
        inode->i_op = de->proc_iops;
        if (de->proc_fops) {
            if (S_ISREG(inode->i_mode)) {
#ifdef CONFIG_COMPAT
                if (use_pde(de)) {
                    if (!de->proc_fops->compat_ioctl)
                        inode->i_fop =
                            &(*klpe_proc_reg_file_ops_no_compat);
                    else
                        inode->i_fop = &(*klpe_proc_reg_file_ops);
                    unuse_pde(de);
                } else {
                    inode->i_fop = &(*klpe_proc_reg_file_ops);
                }
#else /* CONFIG_COMPAT */
                inode->i_fop = &(*klpe_proc_reg_file_ops);
#endif
            } else {
                inode->i_fop = de->proc_fops;
            }
        }
    } else
           (*klpe_pde_put)(de);
    return inode;
}


#include "livepatch_bsc1242579.h"

#include <linux/kernel.h>
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
    { "make_empty_dir_inode", (void *)&klpe_make_empty_dir_inode },
    { "pde_put", (void *)&klpe_pde_put },
    { "proc_reg_file_ops", (void *)&klpe_proc_reg_file_ops },
#ifdef CONFIG_COMPAT
    { "proc_reg_file_ops_no_compat",
      (void *)&klpe_proc_reg_file_ops_no_compat },
#endif
};

int livepatch_bsc1242579_init(void)
{
    return __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

