#!/usr/bin/env bash
RED='\e[31m'
GREEN='\e[32m'
BLUE='\e[34m'
BOLD='\e[1m'
RESET='\e[0m'

log() {
  # $1 - optional: string: "+" for green color, "-" for red color
  # $2 - message
  colorcode="$BLUE"
  [[ "$1" == '+' ]] && colorcode="$GREEN" && shift
  [[ "$1" == '-' ]] && colorcode="$RED" && shift
  echo -e "$colorcode${1}$RESET"
}
fail() {
  # $1 - message
  echo -e "$RED$BOLD${1}$RESET"
  exit 1
}
[[ -z "$DEPLOY_KEY" ]] && fail "The DEPLOY_KEY environment variable is unset. To clone/push the target repo, this script needs an SSH private key as a deploy key."
log "Creating .ssh directory"
SSH_DIR="/root/.ssh"
mkdir -p "$SSH_DIR"
chmod 700 "$SSH_DIR"
KEY_FILE="$SSH_DIR/id_rsa_deploy"
log "Running ssh-agent"
eval $(ssh-agent -s)
log "Outputting SSH key to a file"
echo -e "$DEPLOY_KEY" > "$KEY_FILE"
chmod 0600 "$KEY_FILE"
log "Displaying fingerprint of SSH key to be added"
ssh-keygen -lf "$KEY_FILE"
log "Adding SSH key"
ssh-add "$KEY_FILE"
log "Displaying SSH keys in keyring"
ssh-add -l
log "Adding github.com to known_hosts"
ssh-keyscan github.com 2>/dev/null >> "$SSH_DIR/known_hosts"
chmod 644 "$SSH_DIR/known_hosts"
log "Setting up default git username and email"
git config --global user.name "SUSE Docs Bot"
git config --global user.email "doc-team+docbot@suse.com"
CLONE_REPO="SUSEdoc/release-notes"
CLONE_BRANCH="gh-pages"
CLONE_DIR="susedoc-release-notes"
MAX_COMMITS=35
log "Cloning target repository '$CLONE_REPO' to '$CLONE_DIR'"
git clone --branch "$CLONE_BRANCH" "git@github.com:${CLONE_REPO}.git" "$CLONE_DIR"
TARGET_DIR="${CLONE_DIR}/${PROJECT_NAME}-${PROJECT_BRANCH}/html/release-notes"
log "Creating target directory '$TARGET_DIR'"
mkdir -p "$TARGET_DIR"
log "Copying files to repository directory"
cp -r "public/${PROJECT_BRANCH}/." "$TARGET_DIR"
if [[ "$MAX_COMMITS" -gt 0 && $(PAGER=cat $git log --oneline --format='%h' | wc -l) -ge "$MAX_COMMITS" ]]; then
  log "Resetting target repository, so it does not become too large"
  # nicked from: https://stackoverflow.com/questions/13716658
  $git checkout --orphan new-branch
  $git add -A . >/dev/null
  $git commit -am "Automatic repo reset via CI"
  $git branch -D "$branch"
  $git branch -m "$branch"
  $git push -f origin "$branch" || fail "Target repository could not be force-pushed to."
elif [[ "$MAX_COMMITS" -gt 0 ]]; then
  log "Not resetting target repository, as there are fewer than $MAX_COMMITS commits in the target repository."
else
  log "Not resetting target repository, because 'repo-reset-after=0' is set."
fi
log "Committing and pushing"
git -C "$CLONE_DIR" add .
git -C "$CLONE_DIR" commit --allow-empty -m "Automatic rebuild after $CI_PROJECT_NAME commit $CI_COMMIT_SHORT_SHA"
git -C "$CLONE_DIR" push
log "Done"