/*
 * livepatch_bsc1232271
 *
 * Fix for CVE-2024-49867, bsc#1232271
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from fs/btrfs/disk-io.c */
#include <linux/fs.h>
#include <linux/blkdev.h>
#include <linux/radix-tree.h>
#include <linux/writeback.h>
#include <linux/workqueue.h>
#include <linux/kthread.h>
#include <linux/slab.h>
#include <linux/migrate.h>
#include <linux/ratelimit.h>
#include <linux/uuid.h>
#include <linux/semaphore.h>
#include <linux/error-injection.h>

/* klp-ccp: from include/linux/crc32c.h */
#define _LINUX_CRC32C_H

/* klp-ccp: from fs/btrfs/disk-io.c */
#include <linux/sched/mm.h>

/* klp-ccp: from include/asm-generic/unaligned.h */
#define __ASM_GENERIC_UNALIGNED_H

/* klp-ccp: from fs/btrfs/ctree.h */
#include <linux/mm.h>
#include <linux/sched/signal.h>
#include <linux/highmem.h>
#include <linux/fs.h>
#include <linux/rwsem.h>
#include <linux/semaphore.h>
#include <linux/completion.h>

#include <linux/wait.h>
#include <linux/slab.h>
#include <trace/events/btrfs.h>
#include <asm/unaligned.h>
#include <linux/pagemap.h>
#include <linux/btrfs.h>
#include <linux/btrfs_tree.h>
#include <linux/workqueue.h>

#include <linux/sizes.h>
#include <linux/dynamic_debug.h>
#include <linux/refcount.h>
#include <linux/crc32c.h>

/* klp-ccp: from fs/btrfs/misc.h */
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/math64.h>
#include <linux/rbtree.h>

#define ENUM_BIT(name)                                  \
	__ ## name ## _BIT,                             \
	name = (1U << __ ## name ## _BIT),              \
	__ ## name ## _SEQ = __ ## name ## _BIT

/* klp-ccp: from fs/btrfs/extent-io-tree.h */
enum {
	ENUM_BIT(EXTENT_DIRTY),
	ENUM_BIT(EXTENT_UPTODATE),
	ENUM_BIT(EXTENT_LOCKED),
	ENUM_BIT(EXTENT_NEW),
	ENUM_BIT(EXTENT_DELALLOC),
	ENUM_BIT(EXTENT_DEFRAG),
	ENUM_BIT(EXTENT_BOUNDARY),
	ENUM_BIT(EXTENT_NODATASUM),
	ENUM_BIT(EXTENT_CLEAR_META_RESV),
	ENUM_BIT(EXTENT_NEED_WAIT),
	ENUM_BIT(EXTENT_NORESERVE),
	ENUM_BIT(EXTENT_QGROUP_RESERVED),
	ENUM_BIT(EXTENT_CLEAR_DATA_RESV),
	/*
	 * Must be cleared only during ordered extent completion or on error
	 * paths if we did not manage to submit bios and create the ordered
	 * extents for the range.  Should not be cleared during page release
	 * and page invalidation (if there is an ordered extent in flight),
	 * that is left for the ordered extent completion.
	 */
	ENUM_BIT(EXTENT_DELALLOC_NEW),
	/*
	 * When an ordered extent successfully completes for a region marked as
	 * a new delalloc range, use this flag when clearing a new delalloc
	 * range to indicate that the VFS' inode number of bytes should be
	 * incremented and the inode's new delalloc bytes decremented, in an
	 * atomic way to prevent races with stat(2).
	 */
	ENUM_BIT(EXTENT_ADD_INODE_BYTES),
	/*
	 * Set during truncate when we're clearing an entire range and we just
	 * want the extent states to go away.
	 */
	ENUM_BIT(EXTENT_CLEAR_ALL_BITS),

	/*
	 * This must be last.
	 *
	 * Bit not representing a state but a request for NOWAIT semantics,
	 * e.g. when allocating memory, and must be masked out from the other
	 * bits.
	 */
	ENUM_BIT(EXTENT_NOWAIT)
};

struct extent_io_tree {
	struct rb_root state;
	struct btrfs_fs_info *fs_info;
	/* Inode associated with this tree, or NULL. */
	struct btrfs_inode *inode;

	/* Who owns this io tree, should be one of IO_TREE_* */
	u8 owner;

	spinlock_t lock;
};

bool find_first_extent_bit(struct extent_io_tree *tree, u64 start,
			   u64 *start_ret, u64 *end_ret, u32 bits,
			   struct extent_state **cached_state);

/* klp-ccp: from fs/btrfs/extent_io.h */
#include <linux/rbtree.h>
#include <linux/refcount.h>

#include <linux/btrfs_tree.h>

/* klp-ccp: from fs/btrfs/compression.h */
#include <linux/sizes.h>

/* klp-ccp: from fs/btrfs/bio.h */
#include <linux/bio.h>
#include <linux/workqueue.h>

/* klp-ccp: from fs/btrfs/tree-checker.h */
#include <uapi/linux/btrfs_tree.h>

/* klp-ccp: from fs/btrfs/ulist.h */
#include <linux/list.h>
#include <linux/rbtree.h>

/* klp-ccp: from fs/btrfs/extent_map.h */
#include <linux/rbtree.h>
#include <linux/refcount.h>

/* klp-ccp: from fs/btrfs/async-thread.h */
#include <linux/workqueue.h>

typedef void (*btrfs_func_t)(struct btrfs_work *arg);

struct btrfs_work {
	btrfs_func_t func;
	btrfs_func_t ordered_func;
	btrfs_func_t ordered_free;

	/* Don't touch things below */
	struct work_struct normal_work;
	struct list_head ordered_list;
	struct btrfs_workqueue *wq;
	unsigned long flags;
};

void btrfs_flush_workqueue(struct btrfs_workqueue *wq);

/* klp-ccp: from fs/btrfs/block-rsv.h */
enum btrfs_rsv_type {
	BTRFS_BLOCK_RSV_GLOBAL,
	BTRFS_BLOCK_RSV_DELALLOC,
	BTRFS_BLOCK_RSV_TRANS,
	BTRFS_BLOCK_RSV_CHUNK,
	BTRFS_BLOCK_RSV_DELOPS,
	BTRFS_BLOCK_RSV_DELREFS,
	BTRFS_BLOCK_RSV_EMPTY,
	BTRFS_BLOCK_RSV_TEMP,
};

struct btrfs_block_rsv {
	u64 size;
	u64 reserved;
	struct btrfs_space_info *space_info;
	spinlock_t lock;
	bool full;
	bool failfast;
	/* Block reserve type, one of BTRFS_BLOCK_RSV_* */
	enum btrfs_rsv_type type:8;

	/*
	 * Qgroup equivalent for @size @reserved
	 *
	 * Unlike normal @size/@reserved for inode rsv, qgroup doesn't care
	 * about things like csum size nor how many tree blocks it will need to
	 * reserve.
	 *
	 * Qgroup cares more about net change of the extent usage.
	 *
	 * So for one newly inserted file extent, in worst case it will cause
	 * leaf split and level increase, nodesize for each file extent is
	 * already too much.
	 *
	 * In short, qgroup_size/reserved is the upper limit of possible needed
	 * qgroup metadata reservation.
	 */
	u64 qgroup_rsv_size;
	u64 qgroup_rsv_reserved;
};

/* klp-ccp: from fs/btrfs/locking.h */
#include <linux/atomic.h>
#include <linux/wait.h>
#include <linux/percpu_counter.h>

/* klp-ccp: from fs/btrfs/fs.h */
#include <linux/blkdev.h>
#include <linux/fs.h>
#include <linux/btrfs_tree.h>
#include <linux/sizes.h>

enum {
	BTRFS_FS_CLOSING_START,
	BTRFS_FS_CLOSING_DONE,
	BTRFS_FS_LOG_RECOVERING,
	BTRFS_FS_OPEN,
	BTRFS_FS_QUOTA_ENABLED,
	BTRFS_FS_UPDATE_UUID_TREE_GEN,
	BTRFS_FS_CREATING_FREE_SPACE_TREE,
	BTRFS_FS_BTREE_ERR,
	BTRFS_FS_LOG1_ERR,
	BTRFS_FS_LOG2_ERR,
	BTRFS_FS_QUOTA_OVERRIDE,
	/* Used to record internally whether fs has been frozen */
	BTRFS_FS_FROZEN,
	/*
	 * Indicate that balance has been set up from the ioctl and is in the
	 * main phase. The fs_info::balance_ctl is initialized.
	 */
	BTRFS_FS_BALANCE_RUNNING,

	/*
	 * Indicate that relocation of a chunk has started, it's set per chunk
	 * and is toggled between chunks.
	 */
	BTRFS_FS_RELOC_RUNNING,

	/* Indicate that the cleaner thread is awake and doing something. */
	BTRFS_FS_CLEANER_RUNNING,

	/*
	 * The checksumming has an optimized version and is considered fast,
	 * so we don't need to offload checksums to workqueues.
	 */
	BTRFS_FS_CSUM_IMPL_FAST,

	/* Indicate that the discard workqueue can service discards. */
	BTRFS_FS_DISCARD_RUNNING,

	/* Indicate that we need to cleanup space cache v1 */
	BTRFS_FS_CLEANUP_SPACE_CACHE_V1,

	/* Indicate that we can't trust the free space tree for caching yet */
	BTRFS_FS_FREE_SPACE_TREE_UNTRUSTED,

	/* Indicate whether there are any tree modification log users */
	BTRFS_FS_TREE_MOD_LOG_USERS,

	/* Indicate that we want the transaction kthread to commit right now. */
	BTRFS_FS_COMMIT_TRANS,

	/* Indicate we have half completed snapshot deletions pending. */
	BTRFS_FS_UNFINISHED_DROPS,

	/* Indicate we have to finish a zone to do next allocation. */
	BTRFS_FS_NEED_ZONE_FINISH,

	/* Indicate that we want to commit the transaction. */
	BTRFS_FS_NEED_TRANS_COMMIT,

	/* This is set when active zone tracking is needed. */
	BTRFS_FS_ACTIVE_ZONE_TRACKING,

	/*
	 * Indicate if we have some features changed, this is mostly for
	 * cleaner thread to update the sysfs interface.
	 */
	BTRFS_FS_FEATURE_CHANGED,

#if BITS_PER_LONG == 32
#error "klp-ccp: non-taken branch"
#endif
};

struct btrfs_dev_replace {
	/* See #define above */
	u64 replace_state;
	/* Seconds since 1-Jan-1970 */
	time64_t time_started;
	/* Seconds since 1-Jan-1970 */
	time64_t time_stopped;
	atomic64_t num_write_errors;
	atomic64_t num_uncorrectable_read_errors;

	u64 cursor_left;
	u64 committed_cursor_left;
	u64 cursor_left_last_write_of_item;
	u64 cursor_right;

	/* See #define above */
	u64 cont_reading_from_srcdev_mode;

	int is_valid;
	int item_needs_writeback;
	struct btrfs_device *srcdev;
	struct btrfs_device *tgtdev;

	struct mutex lock_finishing_cancel_unmount;
	struct rw_semaphore rwsem;

	struct btrfs_scrub_progress scrub_progress;

	struct percpu_counter bio_counter;
	wait_queue_head_t replace_wait;
};

struct btrfs_free_cluster {
	spinlock_t lock;
	spinlock_t refill_lock;
	struct rb_root root;

	/* Largest extent in this cluster */
	u64 max_size;

	/* First extent starting offset */
	u64 window_start;

	/* We did a full search and couldn't create a cluster */
	bool fragmented;

	struct btrfs_block_group *block_group;
	/*
	 * When a cluster is allocated from a block group, we put the cluster
	 * onto a list in the block group so that it can be freed before the
	 * block group is freed.
	 */
	struct list_head block_group_list;
};

#define BTRFS_NR_DISCARD_LISTS		3

struct btrfs_discard_ctl {
	struct workqueue_struct *discard_workers;
	struct delayed_work work;
	spinlock_t lock;
	struct btrfs_block_group *block_group;
	struct list_head discard_list[BTRFS_NR_DISCARD_LISTS];
	u64 prev_discard;
	u64 prev_discard_time;
	atomic_t discardable_extents;
	atomic64_t discardable_bytes;
	u64 max_discard_size;
	u64 delay_ms;
	u32 iops_limit;
	u32 kbps_limit;
	u64 discard_extent_bytes;
	u64 discard_bitmap_bytes;
	atomic64_t discard_bytes_saved;
};

enum btrfs_exclusive_operation {
	BTRFS_EXCLOP_NONE,
	BTRFS_EXCLOP_BALANCE_PAUSED,
	BTRFS_EXCLOP_BALANCE,
	BTRFS_EXCLOP_DEV_ADD,
	BTRFS_EXCLOP_DEV_REMOVE,
	BTRFS_EXCLOP_DEV_REPLACE,
	BTRFS_EXCLOP_RESIZE,
	BTRFS_EXCLOP_SWAP_ACTIVATE,
};

struct btrfs_commit_stats {
	/* Total number of commits */
	u64 commit_count;
	/* The maximum commit duration so far in ns */
	u64 max_commit_dur;
	/* The last commit duration in ns */
	u64 last_commit_dur;
	/* The total commit duration in ns */
	u64 total_commit_dur;
};

struct btrfs_fs_info {
	u8 chunk_tree_uuid[BTRFS_UUID_SIZE];
	unsigned long flags;
	struct btrfs_root *tree_root;
	struct btrfs_root *chunk_root;
	struct btrfs_root *dev_root;
	struct btrfs_root *fs_root;
	struct btrfs_root *quota_root;
	struct btrfs_root *uuid_root;
	struct btrfs_root *data_reloc_root;
	struct btrfs_root *block_group_root;

	/* The log root tree is a directory of all the other log roots */
	struct btrfs_root *log_root_tree;

	/* The tree that holds the global roots (csum, extent, etc) */
	rwlock_t global_root_lock;
	struct rb_root global_root_tree;

	spinlock_t fs_roots_radix_lock;
	struct radix_tree_root fs_roots_radix;

	/* Block group cache stuff */
	rwlock_t block_group_cache_lock;
	struct rb_root_cached block_group_cache_tree;

	/* Keep track of unallocated space */
	atomic64_t free_chunk_space;

	/* Track ranges which are used by log trees blocks/logged data extents */
	struct extent_io_tree excluded_extents;

	/* logical->physical extent mapping */
	struct rb_root_cached mapping_tree;
	rwlock_t mapping_tree_lock;

	/*
	 * Block reservation for extent, checksum, root tree and delayed dir
	 * index item.
	 */
	struct btrfs_block_rsv global_block_rsv;
	/* Block reservation for metadata operations */
	struct btrfs_block_rsv trans_block_rsv;
	/* Block reservation for chunk tree */
	struct btrfs_block_rsv chunk_block_rsv;
	/* Block reservation for delayed operations */
	struct btrfs_block_rsv delayed_block_rsv;
	/* Block reservation for delayed refs */
	struct btrfs_block_rsv delayed_refs_rsv;

	struct btrfs_block_rsv empty_block_rsv;

	u64 generation;
	u64 last_trans_committed;
	/*
	 * Generation of the last transaction used for block group relocation
	 * since the filesystem was last mounted (or 0 if none happened yet).
	 * Must be written and read while holding btrfs_fs_info::commit_root_sem.
	 */
	u64 last_reloc_trans;

	/*
	 * This is updated to the current trans every time a full commit is
	 * required instead of the faster short fsync log commits
	 */
	u64 last_trans_log_full_commit;
	unsigned long mount_opt;

	unsigned long compress_type:4;
	unsigned int compress_level;
	u32 commit_interval;
	/*
	 * It is a suggestive number, the read side is safe even it gets a
	 * wrong number because we will write out the data into a regular
	 * extent. The write side(mount/remount) is under ->s_umount lock,
	 * so it is also safe.
	 */
	u64 max_inline;

	struct btrfs_transaction *running_transaction;
	wait_queue_head_t transaction_throttle;
	wait_queue_head_t transaction_wait;
	wait_queue_head_t transaction_blocked_wait;
	wait_queue_head_t async_submit_wait;

	/*
	 * Used to protect the incompat_flags, compat_flags, compat_ro_flags
	 * when they are updated.
	 *
	 * Because we do not clear the flags for ever, so we needn't use
	 * the lock on the read side.
	 *
	 * We also needn't use the lock when we mount the fs, because
	 * there is no other task which will update the flag.
	 */
	spinlock_t super_lock;
	struct btrfs_super_block *super_copy;
	struct btrfs_super_block *super_for_commit;
	struct super_block *sb;
	struct inode *btree_inode;
	struct mutex tree_log_mutex;
	struct mutex transaction_kthread_mutex;
	struct mutex cleaner_mutex;
	struct mutex chunk_mutex;

	/*
	 * This is taken to make sure we don't set block groups ro after the
	 * free space cache has been allocated on them.
	 */
	struct mutex ro_block_group_mutex;

	/*
	 * This is used during read/modify/write to make sure no two ios are
	 * trying to mod the same stripe at the same time.
	 */
	struct btrfs_stripe_hash_table *stripe_hash_table;

	/*
	 * This protects the ordered operations list only while we are
	 * processing all of the entries on it.  This way we make sure the
	 * commit code doesn't find the list temporarily empty because another
	 * function happens to be doing non-waiting preflush before jumping
	 * into the main commit.
	 */
	struct mutex ordered_operations_mutex;

	struct rw_semaphore commit_root_sem;

	struct rw_semaphore cleanup_work_sem;

	struct rw_semaphore subvol_sem;

	spinlock_t trans_lock;
	/*
	 * The reloc mutex goes with the trans lock, it is taken during commit
	 * to protect us from the relocation code.
	 */
	struct mutex reloc_mutex;

	struct list_head trans_list;
	struct list_head dead_roots;
	struct list_head caching_block_groups;

	spinlock_t delayed_iput_lock;
	struct list_head delayed_iputs;
	atomic_t nr_delayed_iputs;
	wait_queue_head_t delayed_iputs_wait;

	atomic64_t tree_mod_seq;

	/* This protects tree_mod_log and tree_mod_seq_list */
	rwlock_t tree_mod_log_lock;
	struct rb_root tree_mod_log;
	struct list_head tree_mod_seq_list;

	atomic_t async_delalloc_pages;

	/* This is used to protect the following list -- ordered_roots. */
	spinlock_t ordered_root_lock;

	/*
	 * All fs/file tree roots in which there are data=ordered extents
	 * pending writeback are added into this list.
	 *
	 * These can span multiple transactions and basically include every
	 * dirty data page that isn't from nodatacow.
	 */
	struct list_head ordered_roots;

	struct mutex delalloc_root_mutex;
	spinlock_t delalloc_root_lock;
	/* All fs/file tree roots that have delalloc inodes. */
	struct list_head delalloc_roots;

	/*
	 * There is a pool of worker threads for checksumming during writes and
	 * a pool for checksumming after reads.  This is because readers can
	 * run with FS locks held, and the writers may be waiting for those
	 * locks.  We don't want ordering in the pending list to cause
	 * deadlocks, and so the two are serviced separately.
	 *
	 * A third pool does submit_bio to avoid deadlocking with the other two.
	 */
	struct btrfs_workqueue *workers;
	struct btrfs_workqueue *hipri_workers;
	struct btrfs_workqueue *delalloc_workers;
	struct btrfs_workqueue *flush_workers;
	struct workqueue_struct *endio_workers;
	struct workqueue_struct *endio_meta_workers;
	struct workqueue_struct *rmw_workers;
	struct workqueue_struct *compressed_write_workers;
	struct btrfs_workqueue *endio_write_workers;
	struct btrfs_workqueue *endio_freespace_worker;
	struct btrfs_workqueue *caching_workers;

	/*
	 * Fixup workers take dirty pages that didn't properly go through the
	 * cow mechanism and make them safe to write.  It happens for the
	 * sys_munmap function call path.
	 */
	struct btrfs_workqueue *fixup_workers;
	struct btrfs_workqueue *delayed_workers;

	struct task_struct *transaction_kthread;
	struct task_struct *cleaner_kthread;
	u32 thread_pool_size;

	struct kobject *space_info_kobj;
	struct kobject *qgroups_kobj;
	struct kobject *discard_kobj;

	/* Used to keep from writing metadata until there is a nice batch */
	struct percpu_counter dirty_metadata_bytes;
	struct percpu_counter delalloc_bytes;
	struct percpu_counter ordered_bytes;
	s32 dirty_metadata_batch;
	s32 delalloc_batch;

	struct list_head dirty_cowonly_roots;

	struct btrfs_fs_devices *fs_devices;

	/*
	 * The space_info list is effectively read only after initial setup.
	 * It is populated at mount time and cleaned up after all block groups
	 * are removed.  RCU is used to protect it.
	 */
	struct list_head space_info;

	struct btrfs_space_info *data_sinfo;

	struct reloc_control *reloc_ctl;

	/* data_alloc_cluster is only used in ssd_spread mode */
	struct btrfs_free_cluster data_alloc_cluster;

	/* All metadata allocations go through this cluster. */
	struct btrfs_free_cluster meta_alloc_cluster;

	/* Auto defrag inodes go here. */
	spinlock_t defrag_inodes_lock;
	struct rb_root defrag_inodes;
	atomic_t defrag_running;

	/* Used to protect avail_{data, metadata, system}_alloc_bits */
	seqlock_t profiles_lock;
	/*
	 * These three are in extended format (availability of single chunks is
	 * denoted by BTRFS_AVAIL_ALLOC_BIT_SINGLE bit, other types are denoted
	 * by corresponding BTRFS_BLOCK_GROUP_* bits)
	 */
	u64 avail_data_alloc_bits;
	u64 avail_metadata_alloc_bits;
	u64 avail_system_alloc_bits;

	/* Balance state */
	spinlock_t balance_lock;
	struct mutex balance_mutex;
	atomic_t balance_pause_req;
	atomic_t balance_cancel_req;
	struct btrfs_balance_control *balance_ctl;
	wait_queue_head_t balance_wait_q;

	/* Cancellation requests for chunk relocation */
	atomic_t reloc_cancel_req;

	u32 data_chunk_allocations;
	u32 metadata_ratio;

	void *bdev_holder;

	/* Private scrub information */
	struct mutex scrub_lock;
	atomic_t scrubs_running;
	atomic_t scrub_pause_req;
	atomic_t scrubs_paused;
	atomic_t scrub_cancel_req;
	wait_queue_head_t scrub_pause_wait;
	/*
	 * The worker pointers are NULL iff the refcount is 0, ie. scrub is not
	 * running.
	 */
	refcount_t scrub_workers_refcnt;
	struct workqueue_struct *scrub_workers;
	struct workqueue_struct *scrub_wr_completion_workers;
	struct btrfs_subpage_info *subpage_info;

	struct btrfs_discard_ctl discard_ctl;

#ifdef CONFIG_BTRFS_FS_CHECK_INTEGRITY
#error "klp-ccp: non-taken branch"
#endif
	u64 qgroup_flags;

	/* Holds configuration and tracking. Protected by qgroup_lock. */
	struct rb_root qgroup_tree;
	spinlock_t qgroup_lock;

	/*
	 * Used to avoid frequently calling ulist_alloc()/ulist_free()
	 * when doing qgroup accounting, it must be protected by qgroup_lock.
	 */
	struct ulist *qgroup_ulist;

	/*
	 * Protect user change for quota operations. If a transaction is needed,
	 * it must be started before locking this lock.
	 */
	struct mutex qgroup_ioctl_lock;

	/* List of dirty qgroups to be written at next commit. */
	struct list_head dirty_qgroups;

	/* Used by qgroup for an efficient tree traversal. */
	u64 qgroup_seq;

	/* Qgroup rescan items. */
	/* Protects the progress item */
	struct mutex qgroup_rescan_lock;
	struct btrfs_key qgroup_rescan_progress;
	struct btrfs_workqueue *qgroup_rescan_workers;
	struct completion qgroup_rescan_completion;
	struct btrfs_work qgroup_rescan_work;
	/* Protected by qgroup_rescan_lock */
	bool qgroup_rescan_running;
	u8 qgroup_drop_subtree_thres;

	/*
	 * If this is not 0, then it indicates a serious filesystem error has
	 * happened and it contains that error (negative errno value).
	 */
	int fs_error;

	/* Filesystem state */
	unsigned long fs_state;

	struct btrfs_delayed_root *delayed_root;

	/* Extent buffer radix tree */
	spinlock_t buffer_lock;
	/* Entries are eb->start / sectorsize */
	struct radix_tree_root buffer_radix;

	/* Next backup root to be overwritten */
	int backup_root_index;

	/* Device replace state */
	struct btrfs_dev_replace dev_replace;

	struct semaphore uuid_tree_rescan_sem;

	/* Used to reclaim the metadata space in the background. */
	struct work_struct async_reclaim_work;
	struct work_struct async_data_reclaim_work;
	struct work_struct preempt_reclaim_work;

	/* Reclaim partially filled block groups in the background */
	struct work_struct reclaim_bgs_work;
	struct list_head reclaim_bgs;
	int bg_reclaim_threshold;

	spinlock_t unused_bgs_lock;
	struct list_head unused_bgs;
	struct mutex unused_bg_unpin_mutex;
	/* Protect block groups that are going to be deleted */
	struct mutex reclaim_bgs_lock;

	/* Cached block sizes */
	u32 nodesize;
	u32 sectorsize;
	/* ilog2 of sectorsize, use to avoid 64bit division */
	u32 sectorsize_bits;
	u32 csum_size;
	u32 csums_per_leaf;
	u32 stripesize;

	/*
	 * Maximum size of an extent. BTRFS_MAX_EXTENT_SIZE on regular
	 * filesystem, on zoned it depends on the device constraints.
	 */
	u64 max_extent_size;

	/* Block groups and devices containing active swapfiles. */
	spinlock_t swapfile_pins_lock;
	struct rb_root swapfile_pins;

	struct crypto_shash *csum_shash;

	/* Type of exclusive operation running, protected by super_lock */
	enum btrfs_exclusive_operation exclusive_operation;

	/*
	 * Zone size > 0 when in ZONED mode, otherwise it's used for a check
	 * if the mode is enabled
	 */
	u64 zone_size;

	/* Constraints for ZONE_APPEND commands: */
	struct queue_limits limits;
	u64 max_zone_append_size;

	struct mutex zoned_meta_io_lock;
	spinlock_t treelog_bg_lock;
	u64 treelog_bg;

	/*
	 * Start of the dedicated data relocation block group, protected by
	 * relocation_bg_lock.
	 */
	spinlock_t relocation_bg_lock;
	u64 data_reloc_bg;
	struct mutex zoned_data_reloc_io_lock;

	struct btrfs_block_group *active_meta_bg;
	struct btrfs_block_group *active_system_bg;

	u64 nr_global_roots;

	spinlock_t zone_active_bgs_lock;
	struct list_head zone_active_bgs;

	/* Updates are not protected by any lock */
	struct btrfs_commit_stats commit_stats;

	/*
	 * Last generation where we dropped a non-relocation root.
	 * Use btrfs_set_last_root_drop_gen() and btrfs_get_last_root_drop_gen()
	 * to change it and to read it, respectively.
	 */
	u64 last_root_drop_gen;

	/*
	 * Annotations for transaction events (structures are empty when
	 * compiled without lockdep).
	 */
	struct lockdep_map btrfs_trans_num_writers_map;
	struct lockdep_map btrfs_trans_num_extwriters_map;
	struct lockdep_map btrfs_state_change_map[4];
	struct lockdep_map btrfs_trans_pending_ordered_map;
	struct lockdep_map btrfs_ordered_extent_map;

#ifdef CONFIG_BTRFS_FS_REF_VERIFY
#error "klp-ccp: non-taken branch"
#endif

#ifdef CONFIG_BTRFS_DEBUG
#error "klp-ccp: non-taken branch"
#endif
};

static inline void btrfs_wake_unfinished_drop(struct btrfs_fs_info *fs_info)
{
	clear_and_wake_up_bit(BTRFS_FS_UNFINISHED_DROPS, &fs_info->flags);
}

#define BTRFS_FS_ERROR(fs_info)	(READ_ONCE((fs_info)->fs_error))

/* klp-ccp: from fs/btrfs/disk-io.h */
int btrfs_commit_super(struct btrfs_fs_info *fs_info);

void btrfs_free_fs_roots(struct btrfs_fs_info *fs_info);

void btrfs_cleanup_one_transaction(struct btrfs_transaction *trans,
				  struct btrfs_fs_info *fs_info);

/* klp-ccp: from fs/btrfs/transaction.h */
#include <linux/refcount.h>

/* klp-ccp: from fs/btrfs/btrfs_inode.h */
#include <linux/hash.h>
#include <linux/refcount.h>

/* klp-ccp: from fs/btrfs/delayed-inode.h */
#include <linux/rbtree.h>
#include <linux/spinlock.h>
#include <linux/mutex.h>
#include <linux/list.h>
#include <linux/wait.h>
#include <linux/atomic.h>
#include <linux/refcount.h>

/* klp-ccp: from fs/btrfs/btrfs_inode.h */
void btrfs_run_delayed_iputs(struct btrfs_fs_info *fs_info);

/* klp-ccp: from fs/btrfs/delayed-ref.h */
#include <linux/refcount.h>

struct btrfs_delayed_ref_root {
	/* head ref rbtree */
	struct rb_root_cached href_root;

	/* dirty extent records */
	struct rb_root dirty_extent_root;

	/* this spin lock protects the rbtree and the entries inside */
	spinlock_t lock;

	/* how many delayed ref updates we've queued, used by the
	 * throttling code
	 */
	atomic_t num_entries;

	/* total number of head nodes in tree */
	unsigned long num_heads;

	/* total number of head nodes ready for processing */
	unsigned long num_heads_ready;

	u64 pending_csums;

	unsigned long flags;

	u64 run_delayed_start;

	/*
	 * To make qgroup to skip given root.
	 * This is for snapshot, as btrfs_qgroup_inherit() will manually
	 * modify counters for snapshot and its source, so we should skip
	 * the snapshot in new_root/old_roots or it will get calculated twice
	 */
	u64 qgroup_to_skip;
};

/* klp-ccp: from fs/btrfs/transaction.h */
enum btrfs_trans_state {
	TRANS_STATE_RUNNING,
	TRANS_STATE_COMMIT_START,
	TRANS_STATE_COMMIT_DOING,
	TRANS_STATE_UNBLOCKED,
	TRANS_STATE_SUPER_COMMITTED,
	TRANS_STATE_COMPLETED,
	TRANS_STATE_MAX,
};

struct btrfs_transaction {
	u64 transid;
	/*
	 * total external writers(USERSPACE/START/ATTACH) in this
	 * transaction, it must be zero before the transaction is
	 * being committed
	 */
	atomic_t num_extwriters;
	/*
	 * total writers in this transaction, it must be zero before the
	 * transaction can end
	 */
	atomic_t num_writers;
	refcount_t use_count;

	unsigned long flags;

	/* Be protected by fs_info->trans_lock when we want to change it. */
	enum btrfs_trans_state state;
	int aborted;
	struct list_head list;
	struct extent_io_tree dirty_pages;
	time64_t start_time;
	wait_queue_head_t writer_wait;
	wait_queue_head_t commit_wait;
	struct list_head pending_snapshots;
	struct list_head dev_update_list;
	struct list_head switch_commits;
	struct list_head dirty_bgs;

	/*
	 * There is no explicit lock which protects io_bgs, rather its
	 * consistency is implied by the fact that all the sites which modify
	 * it do so under some form of transaction critical section, namely:
	 *
	 * - btrfs_start_dirty_block_groups - This function can only ever be
	 *   run by one of the transaction committers. Refer to
	 *   BTRFS_TRANS_DIRTY_BG_RUN usage in btrfs_commit_transaction
	 *
	 * - btrfs_write_dirty_blockgroups - this is called by
	 *   commit_cowonly_roots from transaction critical section
	 *   (TRANS_STATE_COMMIT_DOING)
	 *
	 * - btrfs_cleanup_dirty_bgs - called on transaction abort
	 */
	struct list_head io_bgs;
	struct list_head dropped_roots;
	struct extent_io_tree pinned_extents;

	/*
	 * we need to make sure block group deletion doesn't race with
	 * free space cache writeout.  This mutex keeps them from stomping
	 * on each other
	 */
	struct mutex cache_write_mutex;
	spinlock_t dirty_bgs_lock;
	/* Protected by spin lock fs_info->unused_bgs_lock. */
	struct list_head deleted_bgs;
	spinlock_t dropped_roots_lock;
	struct btrfs_delayed_ref_root delayed_refs;
	struct btrfs_fs_info *fs_info;

	/*
	 * Number of ordered extents the transaction must wait for before
	 * committing. These are ordered extents started by a fast fsync.
	 */
	atomic_t pending_ordered;
	wait_queue_head_t pending_wait;
};

void btrfs_put_transaction(struct btrfs_transaction *transaction);

/* klp-ccp: from fs/btrfs/messages.h */
#include <linux/types.h>

#ifdef CONFIG_PRINTK

#define btrfs_printk(fs_info, fmt, args...)				\
	_btrfs_printk(fs_info, fmt, ##args)

__printf(2, 3)
void _btrfs_printk(const struct btrfs_fs_info *fs_info, const char *fmt, ...);

#else
#error "klp-ccp: non-taken branch"
#endif

#define btrfs_err(fs_info, fmt, args...) \
	btrfs_printk(fs_info, KERN_ERR fmt, ##args)
#define btrfs_warn(fs_info, fmt, args...) \
	btrfs_printk(fs_info, KERN_WARNING fmt, ##args)

#define btrfs_info(fs_info, fmt, args...) \
	btrfs_printk(fs_info, KERN_INFO fmt, ##args)

#ifdef CONFIG_BTRFS_ASSERT
void __noreturn btrfs_assertfail(const char *expr, const char *file, int line);

#define ASSERT(expr)						\
	(likely(expr) ? (void)0 : btrfs_assertfail(#expr, __FILE__, __LINE__))
#else
#error "klp-ccp: non-taken branch"
#endif

/* klp-ccp: from fs/btrfs/dev-replace.h */
void btrfs_dev_replace_suspend_for_unmount(struct btrfs_fs_info *fs_info);

/* klp-ccp: from fs/btrfs/raid56.h */
#include <linux/workqueue.h>

/* klp-ccp: from fs/btrfs/volumes.h */
#include <linux/btrfs.h>

void btrfs_mapping_tree_free(struct btrfs_fs_info *fs_info);

void btrfs_close_devices(struct btrfs_fs_devices *fs_devices);

int btrfs_pause_balance(struct btrfs_fs_info *fs_info);

/* klp-ccp: from fs/btrfs/sysfs.h */
#include <linux/kobject.h>

void btrfs_sysfs_remove_fsid(struct btrfs_fs_devices *fs_devs);

void btrfs_sysfs_remove_mounted(struct btrfs_fs_info *fs_info);

/* klp-ccp: from fs/btrfs/qgroup.h */
#include <linux/spinlock.h>
#include <linux/rbtree.h>
#include <linux/kobject.h>

int btrfs_qgroup_wait_for_completion(struct btrfs_fs_info *fs_info,
				     bool interruptible);

void btrfs_free_qgroup_config(struct btrfs_fs_info *fs_info);

bool btrfs_check_quota_leak(struct btrfs_fs_info *fs_info);

/* klp-ccp: from fs/btrfs/block-group.h */
void btrfs_delete_unused_bgs(struct btrfs_fs_info *fs_info);

void btrfs_put_block_group_cache(struct btrfs_fs_info *info);
int btrfs_free_block_groups(struct btrfs_fs_info *info);

/* klp-ccp: from fs/btrfs/discard.h */
#include <linux/sizes.h>

void btrfs_discard_cleanup(struct btrfs_fs_info *fs_info);

/* klp-ccp: from fs/btrfs/zoned.h */
#include <linux/types.h>
#include <linux/blkdev.h>

/* klp-ccp: from fs/btrfs/subpage.h */
#include <linux/spinlock.h>

/* klp-ccp: from fs/btrfs/defrag.h */
void btrfs_cleanup_defrag_inodes(struct btrfs_fs_info *fs_info);

/* klp-ccp: from fs/btrfs/scrub.h */
int btrfs_scrub_cancel(struct btrfs_fs_info *info);

/* klp-ccp: from fs/btrfs/disk-io.c */
extern int btrfs_cleanup_transaction(struct btrfs_fs_info *fs_info);
static void btrfs_error_commit_super(struct btrfs_fs_info *fs_info);

extern void btrfs_stop_all_workers(struct btrfs_fs_info *fs_info);

extern void free_root_pointers(struct btrfs_fs_info *info, bool free_chunk_root);

void btrfs_free_fs_roots(struct btrfs_fs_info *fs_info);

int btrfs_commit_super(struct btrfs_fs_info *fs_info);

#include "../klp_trace.h"

KLPR_TRACE_EVENT(btrfs, btrfs_transaction_commit,
		 TP_PROTO(const struct btrfs_fs_info *fs_info),
		 TP_ARGS(fs_info)
);

static void warn_about_uncommitted_trans(struct btrfs_fs_info *fs_info)
{
	struct btrfs_transaction *trans;
	struct btrfs_transaction *tmp;
	bool found = false;

	if (list_empty(&fs_info->trans_list))
		return;

	/*
	 * This function is only called at the very end of close_ctree(),
	 * thus no other running transaction, no need to take trans_lock.
	 */
	ASSERT(test_bit(BTRFS_FS_CLOSING_DONE, &fs_info->flags));
	list_for_each_entry_safe(trans, tmp, &fs_info->trans_list, list) {
		struct extent_state *cached = NULL;
		u64 dirty_bytes = 0;
		u64 cur = 0;
		u64 found_start;
		u64 found_end;

		found = true;
		while (find_first_extent_bit(&trans->dirty_pages, cur,
			&found_start, &found_end, EXTENT_DIRTY, &cached)) {
			dirty_bytes += found_end + 1 - found_start;
			cur = found_end + 1;
		}
		btrfs_warn(fs_info,
	"transaction %llu (with %llu dirty metadata bytes) is not committed",
			   trans->transid, dirty_bytes);
		btrfs_cleanup_one_transaction(trans, fs_info);

		if (trans == fs_info->running_transaction)
			fs_info->running_transaction = NULL;
		list_del_init(&trans->list);

		btrfs_put_transaction(trans);
		klpr_trace_btrfs_transaction_commit(fs_info);
	}
	ASSERT(!found);
}

void klpp_close_ctree(struct btrfs_fs_info *fs_info)
{
	int ret;

	set_bit(BTRFS_FS_CLOSING_START, &fs_info->flags);

	/*
	 * If we had UNFINISHED_DROPS we could still be processing them, so
	 * clear that bit and wake up relocation so it can stop.
	 * We must do this before stopping the block group reclaim task, because
	 * at btrfs_relocate_block_group() we wait for this bit, and after the
	 * wait we stop with -EINTR if btrfs_fs_closing() returns non-zero - we
	 * have just set BTRFS_FS_CLOSING_START, so btrfs_fs_closing() will
	 * return 1.
	 */
	btrfs_wake_unfinished_drop(fs_info);

	/*
	 * We may have the reclaim task running and relocating a data block group,
	 * in which case it may create delayed iputs. So stop it before we park
	 * the cleaner kthread otherwise we can get new delayed iputs after
	 * parking the cleaner, and that can make the async reclaim task to hang
	 * if it's waiting for delayed iputs to complete, since the cleaner is
	 * parked and can not run delayed iputs - this will make us hang when
	 * trying to stop the async reclaim task.
	 */
	cancel_work_sync(&fs_info->reclaim_bgs_work);
	/*
	 * We don't want the cleaner to start new transactions, add more delayed
	 * iputs, etc. while we're closing. We can't use kthread_stop() yet
	 * because that frees the task_struct, and the transaction kthread might
	 * still try to wake up the cleaner.
	 */
	kthread_park(fs_info->cleaner_kthread);

	/* wait for the qgroup rescan worker to stop */
	btrfs_qgroup_wait_for_completion(fs_info, false);

	/* wait for the uuid_scan task to finish */
	down(&fs_info->uuid_tree_rescan_sem);
	/* avoid complains from lockdep et al., set sem back to initial state */
	up(&fs_info->uuid_tree_rescan_sem);

	/* pause restriper - we want to resume on mount */
	btrfs_pause_balance(fs_info);

	btrfs_dev_replace_suspend_for_unmount(fs_info);

	btrfs_scrub_cancel(fs_info);

	/* wait for any defraggers to finish */
	wait_event(fs_info->transaction_wait,
		   (atomic_read(&fs_info->defrag_running) == 0));

	/* clear out the rbtree of defraggable inodes */
	btrfs_cleanup_defrag_inodes(fs_info);

	/*
	 * Wait for any fixup workers to complete.
	 * If we don't wait for them here and they are still running by the time
	 * we call kthread_stop() against the cleaner kthread further below, we
	 * get an use-after-free on the cleaner because the fixup worker adds an
	 * inode to the list of delayed iputs and then attempts to wakeup the
	 * cleaner kthread, which was already stopped and destroyed. We parked
	 * already the cleaner, but below we run all pending delayed iputs.
	 */
	btrfs_flush_workqueue(fs_info->fixup_workers);

	/*
	 * After we parked the cleaner kthread, ordered extents may have
	 * completed and created new delayed iputs. If one of the async reclaim
	 * tasks is running and in the RUN_DELAYED_IPUTS flush state, then we
	 * can hang forever trying to stop it, because if a delayed iput is
	 * added after it ran btrfs_run_delayed_iputs() and before it called
	 * btrfs_wait_on_delayed_iputs(), it will hang forever since there is
	 * no one else to run iputs.
	 *
	 * So wait for all ongoing ordered extents to complete and then run
	 * delayed iputs. This works because once we reach this point no one
	 * can either create new ordered extents nor create delayed iputs
	 * through some other means.
	 *
	 * Also note that btrfs_wait_ordered_roots() is not safe here, because
	 * it waits for BTRFS_ORDERED_COMPLETE to be set on an ordered extent,
	 * but the delayed iput for the respective inode is made only when doing
	 * the final btrfs_put_ordered_extent() (which must happen at
	 * btrfs_finish_ordered_io() when we are unmounting).
	 */
	btrfs_flush_workqueue(fs_info->endio_write_workers);
	/* Ordered extents for free space inodes. */
	btrfs_flush_workqueue(fs_info->endio_freespace_worker);
	btrfs_run_delayed_iputs(fs_info);

	cancel_work_sync(&fs_info->async_reclaim_work);
	cancel_work_sync(&fs_info->async_data_reclaim_work);
	cancel_work_sync(&fs_info->preempt_reclaim_work);

	/* Cancel or finish ongoing discard work */
	btrfs_discard_cleanup(fs_info);

	if (!sb_rdonly(fs_info->sb)) {
		/*
		 * The cleaner kthread is stopped, so do one final pass over
		 * unused block groups.
		 */
		btrfs_delete_unused_bgs(fs_info);

		/*
		 * There might be existing delayed inode workers still running
		 * and holding an empty delayed inode item. We must wait for
		 * them to complete first because they can create a transaction.
		 * This happens when someone calls btrfs_balance_delayed_items()
		 * and then a transaction commit runs the same delayed nodes
		 * before any delayed worker has done something with the nodes.
		 * We must wait for any worker here and not at transaction
		 * commit time since that could cause a deadlock.
		 * This is a very rare case.
		 */
		btrfs_flush_workqueue(fs_info->delayed_workers);

		ret = btrfs_commit_super(fs_info);
		if (ret)
			btrfs_err(fs_info, "commit super ret %d", ret);
	}

	if (BTRFS_FS_ERROR(fs_info))
		btrfs_error_commit_super(fs_info);

	kthread_stop(fs_info->transaction_kthread);
	kthread_stop(fs_info->cleaner_kthread);

	ASSERT(list_empty(&fs_info->delayed_iputs));
	set_bit(BTRFS_FS_CLOSING_DONE, &fs_info->flags);

	if (btrfs_check_quota_leak(fs_info)) {
		WARN_ON(IS_ENABLED(CONFIG_BTRFS_DEBUG));
		btrfs_err(fs_info, "qgroup reserved space leaked");
	}

	btrfs_free_qgroup_config(fs_info);
	ASSERT(list_empty(&fs_info->delalloc_roots));

	if (percpu_counter_sum(&fs_info->delalloc_bytes)) {
		btrfs_info(fs_info, "at unmount delalloc count %lld",
		       percpu_counter_sum(&fs_info->delalloc_bytes));
	}

	if (percpu_counter_sum(&fs_info->ordered_bytes))
		btrfs_info(fs_info, "at unmount dio bytes count %lld",
			   percpu_counter_sum(&fs_info->ordered_bytes));

	btrfs_sysfs_remove_mounted(fs_info);
	btrfs_sysfs_remove_fsid(fs_info->fs_devices);

	btrfs_put_block_group_cache(fs_info);

	/*
	 * we must make sure there is not any read request to
	 * submit after we stopping all workers.
	 */
	invalidate_inode_pages2(fs_info->btree_inode->i_mapping);
	btrfs_stop_all_workers(fs_info);

	/* We shouldn't have any transaction open at this point */
	warn_about_uncommitted_trans(fs_info);

	clear_bit(BTRFS_FS_OPEN, &fs_info->flags);
	free_root_pointers(fs_info, true);
	btrfs_free_fs_roots(fs_info);

	/*
	 * We must free the block groups after dropping the fs_roots as we could
	 * have had an IO error and have left over tree log blocks that aren't
	 * cleaned up until the fs roots are freed.  This makes the block group
	 * accounting appear to be wrong because there's pending reserved bytes,
	 * so make sure we do the block group cleanup afterwards.
	 */
	btrfs_free_block_groups(fs_info);

	iput(fs_info->btree_inode);

#ifdef CONFIG_BTRFS_FS_CHECK_INTEGRITY
#error "klp-ccp: non-taken branch"
#endif
	btrfs_mapping_tree_free(fs_info);
	btrfs_close_devices(fs_info->fs_devices);
}

static void btrfs_error_commit_super(struct btrfs_fs_info *fs_info)
{
	/* cleanup FS via transaction */
	btrfs_cleanup_transaction(fs_info);

	mutex_lock(&fs_info->cleaner_mutex);
	btrfs_run_delayed_iputs(fs_info);
	mutex_unlock(&fs_info->cleaner_mutex);

	down_write(&fs_info->cleanup_work_sem);
	up_write(&fs_info->cleanup_work_sem);
}


#include "livepatch_bsc1232271.h"

#include <linux/livepatch.h>

extern typeof(_btrfs_printk) _btrfs_printk
	 KLP_RELOC_SYMBOL(btrfs, btrfs, _btrfs_printk);
extern typeof(btrfs_assertfail) btrfs_assertfail
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_assertfail);
extern typeof(btrfs_check_quota_leak) btrfs_check_quota_leak
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_check_quota_leak);
extern typeof(btrfs_cleanup_defrag_inodes) btrfs_cleanup_defrag_inodes
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_cleanup_defrag_inodes);
extern typeof(btrfs_cleanup_one_transaction) btrfs_cleanup_one_transaction
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_cleanup_one_transaction);
extern typeof(btrfs_cleanup_transaction) btrfs_cleanup_transaction
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_cleanup_transaction);
extern typeof(btrfs_close_devices) btrfs_close_devices
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_close_devices);
extern typeof(btrfs_commit_super) btrfs_commit_super
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_commit_super);
extern typeof(btrfs_delete_unused_bgs) btrfs_delete_unused_bgs
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_delete_unused_bgs);
extern typeof(btrfs_dev_replace_suspend_for_unmount)
	 btrfs_dev_replace_suspend_for_unmount
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_dev_replace_suspend_for_unmount);
extern typeof(btrfs_discard_cleanup) btrfs_discard_cleanup
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_discard_cleanup);
extern typeof(btrfs_flush_workqueue) btrfs_flush_workqueue
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_flush_workqueue);
extern typeof(btrfs_free_block_groups) btrfs_free_block_groups
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_free_block_groups);
extern typeof(btrfs_free_fs_roots) btrfs_free_fs_roots
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_free_fs_roots);
extern typeof(btrfs_free_qgroup_config) btrfs_free_qgroup_config
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_free_qgroup_config);
extern typeof(btrfs_mapping_tree_free) btrfs_mapping_tree_free
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_mapping_tree_free);
extern typeof(btrfs_pause_balance) btrfs_pause_balance
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_pause_balance);
extern typeof(btrfs_put_block_group_cache) btrfs_put_block_group_cache
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_put_block_group_cache);
extern typeof(btrfs_put_transaction) btrfs_put_transaction
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_put_transaction);
extern typeof(btrfs_qgroup_wait_for_completion) btrfs_qgroup_wait_for_completion
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_qgroup_wait_for_completion);
extern typeof(btrfs_run_delayed_iputs) btrfs_run_delayed_iputs
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_run_delayed_iputs);
extern typeof(btrfs_scrub_cancel) btrfs_scrub_cancel
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_scrub_cancel);
extern typeof(btrfs_stop_all_workers) btrfs_stop_all_workers
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_stop_all_workers);
extern typeof(btrfs_sysfs_remove_fsid) btrfs_sysfs_remove_fsid
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_sysfs_remove_fsid);
extern typeof(btrfs_sysfs_remove_mounted) btrfs_sysfs_remove_mounted
	 KLP_RELOC_SYMBOL(btrfs, btrfs, btrfs_sysfs_remove_mounted);
extern typeof(find_first_extent_bit) find_first_extent_bit
	 KLP_RELOC_SYMBOL(btrfs, btrfs, find_first_extent_bit);
extern typeof(free_root_pointers) free_root_pointers
	 KLP_RELOC_SYMBOL(btrfs, btrfs, free_root_pointers);
