/**
 * Utility to sanitize a tree
 *
 * @param {Node} node Hast tree to sanitize
 * @param {Schema} [schema] Schema defining how to sanitize - defaults to Github style sanitation
 */
export function sanitize(node: Node, schema?: Schema): Node
export type Parent = import('hast').Parent
export type Root = import('hast').Root
export type Properties = import('hast').Properties
export type Node = Parent['children'][number] | Root
/**
 * Possible property values
 */
export type PropertyValue = Properties[string]
/**
 * Possible primitive HTML attribute values
 */
export type PrimitivePropertyValue = string | number | boolean
export type AttributeValue = string | [string, ...PrimitivePropertyValue[]]
export type AttributeMap = {
  [x: string]: Array<PrimitivePropertyValue>
}
/**
 * Sanitization configuration
 */
export type Schema = {
  /**
   * Map of tag names to allowed property names. The special '*' key defines property names allowed on all elements
   */
  attributes?: {
    [x: string]: Array<AttributeValue>
  }
  /**
   * Map of tag names to required property names and their default property value
   */
  required?: {
    [x: string]: {
      [x: string]: PropertyValue
    }
  }
  /**
   * List of allowed tag names
   */
  tagNames?: Array<string>
  /**
   * Map of protocols to allow in property values
   */
  protocols?: {
    [x: string]: Array<string>
  }
  /**
   * Map of tag names to their required ancestor elements
   */
  ancestors?: {
    [x: string]: Array<string>
  }
  /**
   * List of allowed property names which can clobber
   */
  clobber?: Array<string>
  /**
   * Prefix to use before potentially clobbering property names
   */
  clobberPrefix?: string
  /**
   * Names of elements to strip from the tree
   */
  strip?: Array<string>
  /**
   * Whether to allow comments
   */
  allowComments?: boolean
  /**
   * Whether to allow doctypes
   */
  allowDoctypes?: boolean
}
export type Handler = (
  schema: Schema,
  value: unknown,
  node: Node,
  stack: Array<string>
) => unknown
export type NodeDefinition = {
  [x: string]: Handler
}
export type NodeDefinitionGetter = (
  schema: Schema,
  node: Node
) => NodeDefinition
export type NodeSchema = {
  [x: string]: NodeDefinition | NodeDefinitionGetter
}
