/*
 * livepatch_bsc1236244
 *
 * Fix for CVE-2024-53208, bsc#1236244
 *
 *  Upstream commit:
 *  0b882940665c ("Bluetooth: MGMT: Fix slab-use-after-free Read in set_powered_sync")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  Not affected
 *
 *  SLE15-SP4 and -SP5 commit:
 *  Not affected
 *
 *  SLE15-SP6 commit:
 *  0e6f9cb7fa5473e70449f2ea7f600c1497b398d2
 *
 *  SLE MICRO-6-0 commit:
 *  0e6f9cb7fa5473e70449f2ea7f600c1497b398d2
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Ali Abdallah <ali.abdallah@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_BT)

#if !IS_MODULE(CONFIG_BT)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from net/bluetooth/mgmt.c */
#include <linux/module.h>

/* klp-ccp: from include/asm-generic/unaligned.h */
#define __ASM_GENERIC_UNALIGNED_H

/* klp-ccp: from net/bluetooth/mgmt.c */
#include <net/bluetooth/bluetooth.h>
#include <net/bluetooth/hci_core.h>

/* klp-ccp: from include/net/bluetooth/hci_sync.h */
static int (*klpe_hci_update_passive_scan)(struct hci_dev *hdev);

static int (*klpe_hci_set_powered_sync)(struct hci_dev *hdev, u8 val);

/* klp-ccp: from net/bluetooth/mgmt.c */
#include <net/bluetooth/hci_sock.h>

#include <net/bluetooth/mgmt.h>

/* klp-ccp: from net/bluetooth/hci_request.h */
#include <asm/unaligned.h>

/* klp-ccp: from net/bluetooth/mgmt_util.h */
struct mgmt_pending_cmd {
	struct list_head list;
	u16 opcode;
	int index;
	void *param;
	size_t param_len;
	struct sock *sk;
	struct sk_buff *skb;
	void *user_data;
	int (*cmd_complete)(struct mgmt_pending_cmd *cmd, u8 status);
};

static int (*klpe_mgmt_cmd_status)(struct sock *sk, u16 index, u16 cmd, u8 status);

static struct mgmt_pending_cmd *(*klpe_mgmt_pending_find)(unsigned short channel, u16 opcode,
					   struct hci_dev *hdev);

static void (*klpe_mgmt_pending_remove)(struct mgmt_pending_cmd *cmd);

/* klp-ccp: from net/bluetooth/eir.h */
#include <asm/unaligned.h>

/* klp-ccp: from net/bluetooth/mgmt.c */
static const u8 (*klpe_mgmt_status_table)[64];

static u8 (*klpe_mgmt_errno_status)(int err);

static u8 klpr_mgmt_status(int err)
{
	if (err < 0)
		return (*klpe_mgmt_errno_status)(err);

	if (err < ARRAY_SIZE((*klpe_mgmt_status_table)))
		return (*klpe_mgmt_status_table)[err];

	return MGMT_STATUS_FAILED;
}

static struct mgmt_pending_cmd *klpr_pending_find(u16 opcode, struct hci_dev *hdev)
{
	return (*klpe_mgmt_pending_find)(HCI_CHANNEL_CONTROL, opcode, hdev);
}

static int (*klpe_send_settings_rsp)(struct sock *sk, u16 opcode, struct hci_dev *hdev);

static void (*klpe_restart_le_actions)(struct hci_dev *hdev);

static int (*klpe_new_settings)(struct hci_dev *hdev, struct sock *skip);

void klpp_mgmt_set_powered_complete(struct hci_dev *hdev, void *data, int err)
{
	struct mgmt_pending_cmd *cmd = data;
	struct mgmt_mode *cp;

	/* Make sure cmd still outstanding. */
	if (err == -ECANCELED ||
	    cmd != klpr_pending_find(MGMT_OP_SET_POWERED, hdev))
		return;

	cp = cmd->param;

	bt_dev_dbg(hdev, "err %d", err);

	if (!err) {
		if (cp->val) {
			hci_dev_lock(hdev);
			(*klpe_restart_le_actions)(hdev);
			(*klpe_hci_update_passive_scan)(hdev);
			hci_dev_unlock(hdev);
		}

		(*klpe_send_settings_rsp)(cmd->sk, cmd->opcode, hdev);

		/* Only call new_setting for power on as power off is deferred
		 * to hdev->power_off work which does call hci_dev_do_close.
		 */
		if (cp->val)
			(*klpe_new_settings)(hdev, cmd->sk);
	} else {
		(*klpe_mgmt_cmd_status)(cmd->sk, hdev->id, MGMT_OP_SET_POWERED,
				klpr_mgmt_status(err));
	}

	(*klpe_mgmt_pending_remove)(cmd);
}

int klpp_set_powered_sync(struct hci_dev *hdev, void *data)
{
	struct mgmt_pending_cmd *cmd = data;
	struct mgmt_mode *cp;

	/* Make sure cmd still outstanding. */
	if (cmd != klpr_pending_find(MGMT_OP_SET_POWERED, hdev))
		return -ECANCELED;

	cp = cmd->param;

	BT_DBG("%s", hdev->name);

	return (*klpe_hci_set_powered_sync)(hdev, cp->val);
}


#include "livepatch_bsc1236244.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "bluetooth"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "hci_set_powered_sync", (void *)&klpe_hci_set_powered_sync,
	  "bluetooth" },
	{ "hci_update_passive_scan", (void *)&klpe_hci_update_passive_scan,
	  "bluetooth" },
	{ "mgmt_cmd_status", (void *)&klpe_mgmt_cmd_status, "bluetooth" },
	{ "mgmt_errno_status", (void *)&klpe_mgmt_errno_status, "bluetooth" },
	{ "mgmt_pending_find", (void *)&klpe_mgmt_pending_find, "bluetooth" },
	{ "mgmt_pending_remove", (void *)&klpe_mgmt_pending_remove,
	  "bluetooth" },
	{ "mgmt_status_table", (void *)&klpe_mgmt_status_table, "bluetooth" },
	{ "new_settings", (void *)&klpe_new_settings, "bluetooth" },
	{ "restart_le_actions", (void *)&klpe_restart_le_actions,
	  "bluetooth" },
	{ "send_settings_rsp", (void *)&klpe_send_settings_rsp, "bluetooth" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1236244_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void livepatch_bsc1236244_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}

#endif /* IS_ENABLED(CONFIG_BT) */
