/*
 * livepatch_bsc1233023
 *
 * Fix for CVE-2022-49025, bsc#1233023
 *
 *  Upstream commit:
 *  52c795af0444 ("net/mlx5e: Fix use-after-free when reverting termination table")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  9ce031e4f5e7e271ddfe931118ce7c51a5b0d81e
 *
 *  SLE15-SP4 and -SP5 commit:
 *  cb783bf6504b0fd96536e98d4959e6be2695e687
 *
 *  SLE15-SP6 commit:
 *  Not affected
 *
 *  SLE MICRO-6-0 commit:
 *  Not affected
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Vincenzo MEZZELA <vincenzo.mezzela@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/eswitch_offloads_termtbl.c */
#include <linux/mlx5/fs.h>

/* klp-ccp: from include/linux/mlx5/fs.h */
static struct mlx5_flow_namespace *
(*klpe_mlx5_get_flow_namespace)(struct mlx5_core_dev *dev,
			enum mlx5_flow_namespace_type type);

static struct mlx5_flow_table *
(*klpe_mlx5_create_auto_grouped_flow_table)(struct mlx5_flow_namespace *ns,
				    struct mlx5_flow_table_attr *ft_attr);

static int (*klpe_mlx5_destroy_flow_table)(struct mlx5_flow_table *ft);

static struct mlx5_flow_handle *
(*klpe_mlx5_add_flow_rules)(struct mlx5_flow_table *ft,
		    const struct mlx5_flow_spec *spec,
		    struct mlx5_flow_act *flow_act,
		    struct mlx5_flow_destination *dest,
		    int num_dest);

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/eswitch.h */
#include <linux/if_ether.h>
#include <linux/if_link.h>
#include <linux/atomic.h>
#include <net/devlink.h>
#include <linux/mlx5/device.h>
#include <linux/mlx5/eswitch.h>

/* klp-ccp: from include/linux/mlx5/vport.h */
#define __MLX5_VPORT_H__

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/eswitch.h */
#include <linux/mlx5/fs.h>

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/lib/mpfs.h */
#include <linux/if_ether.h>
#include <linux/mlx5/device.h>

#define MLX5_L2_ADDR_HASH_SIZE (BIT(BITS_PER_BYTE))

struct l2addr_node {
	struct hlist_node hlist;
	u8                addr[ETH_ALEN];
};

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/en/tc_ct.h */
#include <linux/mlx5/fs.h>
/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/en.h */
#include <linux/if_vlan.h>
#include <linux/etherdevice.h>
#include <linux/timecounter.h>
#include <linux/net_tstamp.h>

#include <linux/mlx5/driver.h>

/* klp-ccp: from include/linux/mlx5/qp.h */
#define MLX5_QP_H

/* klp-ccp: from include/linux/mlx5/cq.h */
#define MLX5_CORE_CQ_H

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/en.h */
#include <linux/mlx5/vport.h>

#include <linux/mlx5/fs.h>

/* klp-ccp: from include/linux/rhashtable.h */
#define _LINUX_RHASHTABLE_H

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/en.h */
#include <net/xdp.h>
#include <linux/dim.h>
#include <linux/bits.h>

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/wq.h */
#include <linux/mlx5/mlx5_ifc.h>
#include <linux/mlx5/cq.h>
#include <linux/mlx5/qp.h>

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/mlx5_core.h */
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/if_link.h>

#include <linux/mlx5/cq.h>
#include <linux/mlx5/fs.h>
#include <linux/mlx5/driver.h>

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/en/mod_hdr.h */
#include <linux/hashtable.h>
#include <linux/mlx5/fs.h>

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/lib/hv.h */
#if IS_ENABLED(CONFIG_PCI_HYPERV_INTERFACE)

#include <linux/mlx5/driver.h>
#endif

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/en/tc_ct.h */
struct mlx5_esw_flow_attr;

struct mlx5_ct_attr {
	u16 zone;
	u16 ct_action;
	struct mlx5_ct_flow *ct_flow;
	struct nf_flowtable *nf_ft;
	u32 ct_labels_id;
};

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/eswitch.h */
#ifdef CONFIG_MLX5_ESWITCH

struct mlx5_eswitch_fdb {
	union {
		struct legacy_fdb {
			struct mlx5_flow_table *fdb;
			struct mlx5_flow_group *addr_grp;
			struct mlx5_flow_group *allmulti_grp;
			struct mlx5_flow_group *promisc_grp;
			struct mlx5_flow_table *vepa_fdb;
			struct mlx5_flow_handle *vepa_uplink_rule;
			struct mlx5_flow_handle *vepa_star_rule;
		} legacy;

		struct offloads_fdb {
			struct mlx5_flow_namespace *ns;
			struct mlx5_flow_table *slow_fdb;
			struct mlx5_flow_group *send_to_vport_grp;
			struct mlx5_flow_group *peer_miss_grp;
			struct mlx5_flow_handle **peer_miss_rules;
			struct mlx5_flow_group *miss_grp;
			struct mlx5_flow_handle *miss_rule_uni;
			struct mlx5_flow_handle *miss_rule_multi;
			int vlan_push_pop_refcount;

			struct mlx5_esw_chains_priv *esw_chains_priv;
			struct {
				DECLARE_HASHTABLE(table, 8);
				/* Protects vports.table */
				struct mutex lock;
			} vports;

		} offloads;
	};
	u32 flags;
};

struct mlx5_esw_offload {
	struct mlx5_flow_table *ft_offloads_restore;
	struct mlx5_flow_group *restore_group;
	struct mlx5_modify_hdr *restore_copy_hdr_id;

	struct mlx5_flow_table *ft_offloads;
	struct mlx5_flow_group *vport_rx_group;
	struct mlx5_eswitch_rep *vport_reps;
	struct list_head peer_flows;
	struct mutex peer_mutex;
	struct mutex encap_tbl_lock; /* protects encap_tbl */
	DECLARE_HASHTABLE(encap_tbl, 8);
	struct mutex decap_tbl_lock; /* protects decap_tbl */
	DECLARE_HASHTABLE(decap_tbl, 8);
	struct mod_hdr_tbl mod_hdr;
	DECLARE_HASHTABLE(termtbl_tbl, 8);
	struct mutex termtbl_mutex; /* protects termtbl hash */
	const struct mlx5_eswitch_rep_ops *rep_ops[NUM_REP_TYPES];
	u8 inline_mode;
	atomic64_t num_flows;
	enum devlink_eswitch_encap_mode encap;
	struct ida vport_metadata_ida;
	unsigned int host_number; /* ECPF supports one external host */
};

struct esw_mc_addr { /* SRIOV only */
	struct l2addr_node     node;
	struct mlx5_flow_handle *uplink_rule; /* Forward to uplink rule */
	u32                    refcnt;
};

struct mlx5_esw_functions {
	struct mlx5_nb		nb;
	u16			num_vfs;
};

struct mlx5_eswitch {
	struct mlx5_core_dev    *dev;
	struct mlx5_nb          nb;
	struct mlx5_eswitch_fdb fdb_table;
	/* legacy data structures */
	struct hlist_head       mc_table[MLX5_L2_ADDR_HASH_SIZE];
	struct esw_mc_addr mc_promisc;
	/* end of legacy */
	struct workqueue_struct *work_queue;
	struct mlx5_vport       *vports;
	u32 flags;
	int                     total_vports;
	int                     enabled_vports;
	/* Synchronize between vport change events
	 * and async SRIOV admin state changes
	 */
	struct mutex            state_lock;

	/* Protects eswitch mode change that occurs via one or more
	 * user commands, i.e. sriov state change, devlink commands.
	 */
	struct mutex mode_lock;

	struct {
		bool            enabled;
		u32             root_tsar_id;
	} qos;

	struct mlx5_esw_offload offloads;
	int                     mode;
	u16                     manager_vport;
	u16                     first_host_vport;
	struct mlx5_esw_functions esw_funcs;
	struct {
		u32             large_group_num;
	}  params;
};

struct mlx5_termtbl_handle;

struct mlx5_flow_handle *
klpp_mlx5_eswitch_add_termtbl_rule(struct mlx5_eswitch *esw,
			      struct mlx5_flow_table *ft,
			      struct mlx5_flow_spec *spec,
			      struct mlx5_esw_flow_attr *attr,
			      struct mlx5_flow_act *flow_act,
			      struct mlx5_flow_destination *dest,
			      int num_dest);

static void
(*klpe_mlx5_eswitch_termtbl_put)(struct mlx5_eswitch *esw,
			 struct mlx5_termtbl_handle *tt);

#define MLX5_MAX_FLOW_FWD_VPORTS 2

enum {
	MLX5_ESW_DEST_ENCAP         = BIT(0),
	MLX5_ESW_DEST_ENCAP_VALID   = BIT(1),
};

struct mlx5_esw_flow_attr {
	struct mlx5_eswitch_rep *in_rep;
	struct mlx5_core_dev	*in_mdev;
	struct mlx5_core_dev    *counter_dev;

	int split_count;
	int out_count;

	int	action;
	__be16	vlan_proto[MLX5_FS_VLAN_DEPTH];
	u16	vlan_vid[MLX5_FS_VLAN_DEPTH];
	u8	vlan_prio[MLX5_FS_VLAN_DEPTH];
	u8	total_vlan;
	struct {
		u32 flags;
		struct mlx5_eswitch_rep *rep;
		struct mlx5_pkt_reformat *pkt_reformat;
		struct mlx5_core_dev *mdev;
		struct mlx5_termtbl_handle *termtbl;
	} dests[MLX5_MAX_FLOW_FWD_VPORTS];
	struct  mlx5_modify_hdr *modify_hdr;
	u8	inner_match_level;
	u8	outer_match_level;
	struct mlx5_fc *counter;
	u32	chain;
	u16	prio;
	u32	dest_chain;
	u32	flags;
	struct mlx5_flow_table *fdb;
	struct mlx5_flow_table *dest_ft;
	struct mlx5_ct_attr ct_attr;
	struct mlx5_pkt_reformat *decap_pkt_reformat;
	struct mlx5e_tc_flow_parse_attr *parse_attr;
};

#define esw_warn(__dev, format, ...)			\
	dev_warn((__dev)->device, "E-Switch: " format, ##__VA_ARGS__)

#else  /* CONFIG_MLX5_ESWITCH */
#error "klp-ccp: non-taken branch"
#endif /* CONFIG_MLX5_ESWITCH */

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/fs_core.h */
#include <linux/refcount.h>
#include <linux/mlx5/fs.h>
#include <linux/rhashtable.h>
#include <linux/llist.h>

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/steering/fs_dr.h */
struct mlx5_fs_dr_action {
	struct mlx5dr_action *dr_action;
};

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/fs_core.h */
struct mlx5_pkt_reformat {
	enum mlx5_flow_namespace_type ns_type;
	int reformat_type; /* from mlx5_ifc */
	union {
		struct mlx5_fs_dr_action action;
		u32 id;
	};
};

/* klp-ccp: from drivers/net/ethernet/mellanox/mlx5/core/eswitch_offloads_termtbl.c */
struct mlx5_termtbl_handle {
	struct hlist_node termtbl_hlist;

	struct mlx5_flow_table *termtbl;
	struct mlx5_flow_act flow_act;
	struct mlx5_flow_destination dest;

	struct mlx5_flow_handle *rule;
	int ref_count;
};

static u32
mlx5_eswitch_termtbl_hash(struct mlx5_flow_act *flow_act,
			  struct mlx5_flow_destination *dest)
{
	u32 hash;

	hash = jhash_1word(flow_act->action, 0);
	hash = jhash((const void *)&flow_act->vlan,
		     sizeof(flow_act->vlan), hash);
	hash = jhash((const void *)&dest->vport.num,
		     sizeof(dest->vport.num), hash);
	hash = jhash((const void *)&dest->vport.vhca_id,
		     sizeof(dest->vport.num), hash);
	if (dest->vport.pkt_reformat)
		hash = jhash(dest->vport.pkt_reformat,
			     sizeof(*dest->vport.pkt_reformat),
			     hash);
	return hash;
}

static int
mlx5_eswitch_termtbl_cmp(struct mlx5_flow_act *flow_act1,
			 struct mlx5_flow_destination *dest1,
			 struct mlx5_flow_act *flow_act2,
			 struct mlx5_flow_destination *dest2)
{
	int ret;

	ret = flow_act1->action != flow_act2->action ||
	      dest1->vport.num != dest2->vport.num ||
	      dest1->vport.vhca_id != dest2->vport.vhca_id ||
	      memcmp(&flow_act1->vlan, &flow_act2->vlan,
		     sizeof(flow_act1->vlan));
	if (ret)
		return ret;

	return dest1->vport.pkt_reformat && dest2->vport.pkt_reformat ?
	       memcmp(dest1->vport.pkt_reformat, dest2->vport.pkt_reformat,
		      sizeof(*dest1->vport.pkt_reformat)) : 0;
}

static int
klpr_mlx5_eswitch_termtbl_create(struct mlx5_core_dev *dev,
			    struct mlx5_termtbl_handle *tt,
			    struct mlx5_flow_act *flow_act)
{
	struct mlx5_flow_table_attr ft_attr = {};
	struct mlx5_flow_namespace *root_ns;
	int err;

	root_ns = (*klpe_mlx5_get_flow_namespace)(dev, MLX5_FLOW_NAMESPACE_FDB);
	if (!root_ns) {
		esw_warn(dev, "Failed to get FDB flow namespace\n");
		return -EOPNOTSUPP;
	}

	/* As this is the terminating action then the termination table is the
	 * same prio as the slow path
	 */
	ft_attr.flags = MLX5_FLOW_TABLE_TERMINATION | MLX5_FLOW_TABLE_UNMANAGED |
			MLX5_FLOW_TABLE_TUNNEL_EN_REFORMAT;
	ft_attr.prio = FDB_TC_OFFLOAD;
	ft_attr.max_fte = 1;
	ft_attr.level = 1;
	ft_attr.autogroup.max_num_groups = 1;
	tt->termtbl = (*klpe_mlx5_create_auto_grouped_flow_table)(root_ns, &ft_attr);
	if (IS_ERR(tt->termtbl)) {
		esw_warn(dev, "Failed to create termination table\n");
		return -EOPNOTSUPP;
	}

	tt->rule = (*klpe_mlx5_add_flow_rules)(tt->termtbl, NULL, flow_act,
				       &tt->dest, 1);
	if (IS_ERR(tt->rule)) {
		esw_warn(dev, "Failed to create termination table rule\n");
		goto add_flow_err;
	}
	return 0;

add_flow_err:
	err = (*klpe_mlx5_destroy_flow_table)(tt->termtbl);
	if (err)
		esw_warn(dev, "Failed to destroy termination table\n");

	return -EOPNOTSUPP;
}

static struct mlx5_termtbl_handle *
klpr_mlx5_eswitch_termtbl_get_create(struct mlx5_eswitch *esw,
				struct mlx5_flow_act *flow_act,
				struct mlx5_flow_destination *dest,
				struct mlx5_esw_flow_attr *attr)
{
	struct mlx5_termtbl_handle *tt;
	bool found = false;
	u32 hash_key;
	int err;

	mutex_lock(&esw->offloads.termtbl_mutex);
	hash_key = mlx5_eswitch_termtbl_hash(flow_act, dest);
	hash_for_each_possible(esw->offloads.termtbl_tbl, tt,
			       termtbl_hlist, hash_key) {
		if (!mlx5_eswitch_termtbl_cmp(&tt->flow_act, &tt->dest,
					      flow_act, dest)) {
			found = true;
			break;
		}
	}
	if (found)
		goto tt_add_ref;

	tt = kzalloc(sizeof(*tt), GFP_KERNEL);
	if (!tt) {
		err = -ENOMEM;
		goto tt_create_err;
	}

	tt->dest.type = MLX5_FLOW_DESTINATION_TYPE_VPORT;
	tt->dest.vport.num = dest->vport.num;
	tt->dest.vport.vhca_id = dest->vport.vhca_id;
	tt->dest.vport.flags = dest->vport.flags;
	memcpy(&tt->flow_act, flow_act, sizeof(*flow_act));

	err = klpr_mlx5_eswitch_termtbl_create(esw->dev, tt, flow_act);
	if (err) {
		esw_warn(esw->dev, "Failed to create termination table\n");
		goto tt_create_err;
	}
	hash_add(esw->offloads.termtbl_tbl, &tt->termtbl_hlist, hash_key);
tt_add_ref:
	tt->ref_count++;
	mutex_unlock(&esw->offloads.termtbl_mutex);
	return tt;
tt_create_err:
	kfree(tt);
	mutex_unlock(&esw->offloads.termtbl_mutex);
	return ERR_PTR(err);
}

static void
(*klpe_mlx5_eswitch_termtbl_actions_move)(struct mlx5_flow_act *src,
				  struct mlx5_flow_act *dst);

struct mlx5_flow_handle *
klpp_mlx5_eswitch_add_termtbl_rule(struct mlx5_eswitch *esw,
			      struct mlx5_flow_table *fdb,
			      struct mlx5_flow_spec *spec,
			      struct mlx5_esw_flow_attr *attr,
			      struct mlx5_flow_act *flow_act,
			      struct mlx5_flow_destination *dest,
			      int num_dest)
{
	struct mlx5_flow_act term_tbl_act = {};
	struct mlx5_flow_handle *rule = NULL;
	bool term_table_created = false;
	int num_vport_dests = 0;
	int i, curr_dest;

	(*klpe_mlx5_eswitch_termtbl_actions_move)(flow_act, &term_tbl_act);
	term_tbl_act.action |= MLX5_FLOW_CONTEXT_ACTION_FWD_DEST;

	for (i = 0; i < num_dest; i++) {
		struct mlx5_termtbl_handle *tt;

		/* only vport destinations can be terminated */
		if (dest[i].type != MLX5_FLOW_DESTINATION_TYPE_VPORT)
			continue;

		if (attr->dests[num_vport_dests].flags & MLX5_ESW_DEST_ENCAP) {
			term_tbl_act.action |= MLX5_FLOW_CONTEXT_ACTION_PACKET_REFORMAT;
			term_tbl_act.pkt_reformat = attr->dests[num_vport_dests].pkt_reformat;
		} else {
			term_tbl_act.action &= ~MLX5_FLOW_CONTEXT_ACTION_PACKET_REFORMAT;
			term_tbl_act.pkt_reformat = NULL;
		}

		/* get the terminating table for the action list */
		tt = klpr_mlx5_eswitch_termtbl_get_create(esw, &term_tbl_act,
						     &dest[i], attr);
		if (IS_ERR(tt)) {
			esw_warn(esw->dev, "Failed to create termination table\n");
			goto revert_changes;
		}
		attr->dests[num_vport_dests].termtbl = tt;
		num_vport_dests++;

		/* link the destination with the termination table */
		dest[i].type = MLX5_FLOW_DESTINATION_TYPE_FLOW_TABLE;
		dest[i].ft = tt->termtbl;
		term_table_created = true;
	}

	/* at least one destination should reference a termination table */
	if (!term_table_created)
		goto revert_changes;

	/* create the FTE */
	flow_act->action &= ~MLX5_FLOW_CONTEXT_ACTION_PACKET_REFORMAT;
	flow_act->pkt_reformat = NULL;
	flow_act->flags |= FLOW_ACT_IGNORE_FLOW_LEVEL;
	rule = (*klpe_mlx5_add_flow_rules)(fdb, spec, flow_act, dest, num_dest);
	if (IS_ERR(rule))
		goto revert_changes;

	goto out;

revert_changes:
	/* revert the changes that were made to the original flow_act
	 * and fall-back to the original rule actions
	 */
	(*klpe_mlx5_eswitch_termtbl_actions_move)(&term_tbl_act, flow_act);

	for (curr_dest = 0; curr_dest < num_vport_dests; curr_dest++) {
		struct mlx5_termtbl_handle *tt = attr->dests[curr_dest].termtbl;

		attr->dests[curr_dest].termtbl = NULL;

		/* search for the destination associated with the
		 * current term table
		 */
		for (i = 0; i < num_dest; i++) {
			if (dest[i].ft != tt->termtbl)
				continue;

			memset(&dest[i], 0, sizeof(dest[i]));
			dest[i].type = MLX5_FLOW_DESTINATION_TYPE_VPORT;
			dest[i].vport.num = tt->dest.vport.num;
			dest[i].vport.vhca_id = tt->dest.vport.vhca_id;
			(*klpe_mlx5_eswitch_termtbl_put)(esw, tt);
			break;
		}
	}
	rule = (*klpe_mlx5_add_flow_rules)(fdb, spec, flow_act, dest, num_dest);
out:
	return rule;
}


#include "livepatch_bsc1233023.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "mlx5_core"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "mlx5_add_flow_rules", (void *)&klpe_mlx5_add_flow_rules,
	  "mlx5_core" },
	{ "mlx5_create_auto_grouped_flow_table",
	  (void *)&klpe_mlx5_create_auto_grouped_flow_table, "mlx5_core" },
	{ "mlx5_destroy_flow_table", (void *)&klpe_mlx5_destroy_flow_table,
	  "mlx5_core" },
	{ "mlx5_eswitch_termtbl_actions_move",
	  (void *)&klpe_mlx5_eswitch_termtbl_actions_move, "mlx5_core" },
	{ "mlx5_eswitch_termtbl_put", (void *)&klpe_mlx5_eswitch_termtbl_put,
	  "mlx5_core" },
	{ "mlx5_get_flow_namespace", (void *)&klpe_mlx5_get_flow_namespace,
	  "mlx5_core" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1233023_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LP_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void livepatch_bsc1233023_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}
