/*
   Test suite for FSRVP server state

   Copyright (C) David Disseldorp 2012-2015

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "includes.h"
#include <unistd.h>
#include <fcntl.h>

#include "librpc/gen_ndr/security.h"
#include "lib/param/param.h"
#include "lib/util/dlinklist.h"
#include "libcli/resolve/resolve.h"
#include "librpc/gen_ndr/ndr_fsrvp.h"
#include "librpc/gen_ndr/ndr_fsrvp_c.h"
#include "source3/rpc_server/fss/srv_fss_private.h"
#include "torture/torture.h"
#include "torture/local/proto.h"
#include "lib/dbwrap/dbwrap.h"
#include "source3/lib/dbwrap/dbwrap_open.h"
#include "source3/include/util_tdb.h"

#define FSS_DB_KEY_VERSION "db_version"
#define FSS_DB_KEY_CONTEXT "context"
#define FSS_DB_KEY_SC_SET_COUNT "sc_set_count"
#define FSS_DB_KEY_PFX_SC_SET "sc_set/"
#define FSS_DB_KEY_PFX_SC "sc/"
#define FSS_DB_KEY_PFX_SMAP "smap/"

#define FSS_DB_FMT_SC_SET "Pddd"
#define FSS_DB_FMT_SC "PPPBd"
#define FSS_DB_FMT_SMAP "dPPPd"

/* old database format version, used for testing upgrades */
#define FSS_DB_VAL_VERSION 0

/*
 * The following db_version=0 fss_state.*_store() source code was obtained
 * from an old srv_fss_state.c where tdb_pack() was used to marshall FSRVP state
 * structures instead of the db_version=1 IDL based marshalling.
 * Using this code, we can test db_version=0->db_version=1 upgrades in the state
 * retrieval code path.
 */
static NTSTATUS fss_state_smap_store(TALLOC_CTX *mem_ctx,
				     struct db_context *db,
				     const char *sc_key_str,
				     struct fss_sc_smap *smap)
{
	NTSTATUS status;
	size_t len;
	TDB_DATA val;
	const char *smap_key_str;

	/* becomes sc_set/@sc_set_id/sc/@sc_id/smap/@sc_share_name */
	smap_key_str = talloc_asprintf(mem_ctx, "%s/%s%s", sc_key_str,
				       FSS_DB_KEY_PFX_SMAP,
				       smap->sc_share_name);
	if (smap_key_str == NULL) {
		return NT_STATUS_NO_MEMORY;
	}

	/*
	 * @smap->sc_share_comment may be null if not exposed.
	 * -1 field was previously used for snum storage.
	 */
	len = tdb_pack(NULL, 0, FSS_DB_FMT_SMAP, -1,
		       smap->share_name, smap->sc_share_name,
		       (smap->sc_share_comment ? smap->sc_share_comment : ""),
		       (int)smap->is_exposed);

	val.dptr = talloc_size(mem_ctx, len);
	if (val.dptr == NULL) {
		return NT_STATUS_NO_MEMORY;
	}
	val.dsize = len;

	tdb_pack(val.dptr, val.dsize, FSS_DB_FMT_SMAP, -1,
		 smap->share_name, smap->sc_share_name,
		 (smap->sc_share_comment ? smap->sc_share_comment : ""),
		 (int)smap->is_exposed);

	status = dbwrap_store(db, string_term_tdb_data(smap_key_str), val, 0);
	if (!NT_STATUS_IS_OK(status)) {
		return status;
	}

	return NT_STATUS_OK;
}

static NTSTATUS fss_state_sc_store(TALLOC_CTX *mem_ctx,
				   struct db_context *db,
				   const char *sc_set_key_str,
				   struct fss_sc *sc)
{
	NTSTATUS status;
	size_t len;
	TDB_DATA val;
	const char *sc_key_str;
	struct fss_sc_smap *smap;

	/* becomes sc_set/@sc_set.id/sc/@sc_id */
	sc_key_str = talloc_asprintf(mem_ctx, "%s/%s%s", sc_set_key_str,
				     FSS_DB_KEY_PFX_SC, sc->id_str);
	if (sc_key_str == NULL) {
		return NT_STATUS_NO_MEMORY;
	}

	/* @sc->sc_path may be null if not committed, store empty str */
	len = tdb_pack(NULL, 0, FSS_DB_FMT_SC, sc->id_str,
		       sc->volume_name,
		       (sc->sc_path ? sc->sc_path : ""),
		       sizeof(sc->create_ts), &sc->create_ts,
		       sc->smaps_count);

	val.dptr = talloc_size(mem_ctx, len);
	if (val.dptr == NULL) {
		return NT_STATUS_NO_MEMORY;
	}
	val.dsize = len;

	tdb_pack(val.dptr, val.dsize, FSS_DB_FMT_SC, sc->id_str,
		 sc->volume_name,
		 (sc->sc_path ? sc->sc_path : ""),
		 sizeof(sc->create_ts), &sc->create_ts,
		 sc->smaps_count);

	status = dbwrap_store(db, string_term_tdb_data(sc_key_str), val, 0);
	if (!NT_STATUS_IS_OK(status)) {
		return status;
	}

	for (smap = sc->smaps; smap; smap = smap->next) {
		status = fss_state_smap_store(mem_ctx, db, sc_key_str, smap);
		if (!NT_STATUS_IS_OK(status)) {
			return status;
		}
	}

	return NT_STATUS_OK;
}

static NTSTATUS fss_state_sc_set_store(TALLOC_CTX *mem_ctx,
				       struct db_context *db,
				       struct fss_sc_set *sc_set)
{
	NTSTATUS status;
	size_t len;
	TDB_DATA val;
	const char *sc_set_key_str;
	struct fss_sc *sc;

	sc_set_key_str = talloc_asprintf(mem_ctx, "%s%s",
					 FSS_DB_KEY_PFX_SC_SET,
					 sc_set->id_str);
	if (sc_set_key_str == NULL) {
		return NT_STATUS_NO_MEMORY;
	}

	len = tdb_pack(NULL, 0, FSS_DB_FMT_SC_SET, sc_set->id_str,
		       sc_set->state, sc_set->context, sc_set->scs_count);

	val.dptr = talloc_size(mem_ctx, len);
	if (val.dptr == NULL) {
		return NT_STATUS_NO_MEMORY;
	}
	val.dsize = len;

	tdb_pack(val.dptr, val.dsize, FSS_DB_FMT_SC_SET, sc_set->id_str,
		 sc_set->state, sc_set->context, sc_set->scs_count);

	status = dbwrap_store(db, string_term_tdb_data(sc_set_key_str), val, 0);
	if (!NT_STATUS_IS_OK(status)) {
		return status;
	}

	for (sc = sc_set->scs; sc; sc = sc->next) {
		status = fss_state_sc_store(mem_ctx, db, sc_set_key_str, sc);
		if (!NT_STATUS_IS_OK(status)) {
			return status;
		}
	}


	return NT_STATUS_OK;
}

/*
 * write out the current fsrvp server state to a TDB using the outdated version
 * zero format. This clears any content currently written to the TDB.
 */
static NTSTATUS fss_state_store_v0(TALLOC_CTX *mem_ctx,
				   struct fss_sc_set *sc_sets,
				   uint32_t sc_sets_count,
				   const char *db_path)
{
	TALLOC_CTX *tmp_ctx;
	struct db_context *db;
	NTSTATUS status;
	int ret;
	struct fss_sc_set *sc_set;

	tmp_ctx = talloc_new(mem_ctx);
	if (tmp_ctx == NULL) {
		return NT_STATUS_NO_MEMORY;
	}

	db = db_open(tmp_ctx, db_path, 0, TDB_DEFAULT,  O_RDWR | O_CREAT,
		     0600, DBWRAP_LOCK_ORDER_1, DBWRAP_FLAG_NONE);
	if (db == NULL) {
		DEBUG(0, ("Failed to open fss state database %s\n", db_path));
		status = NT_STATUS_ACCESS_DENIED;
		goto err_ctx_free;
	}

	ret = dbwrap_wipe(db);
	if (ret != 0) {
		status = NT_STATUS_UNSUCCESSFUL;
		goto err_db_free;
	}

	status = dbwrap_store_int32_bystring(db, FSS_DB_KEY_VERSION,
					     FSS_DB_VAL_VERSION);
	if (!NT_STATUS_IS_OK(status)) {
		goto err_db_free;
	}

	ret = dbwrap_transaction_start(db);
	if (ret != 0) {
		status = NT_STATUS_UNSUCCESSFUL;
		goto err_db_free;
	}

	status = dbwrap_store_int32_bystring(db, FSS_DB_KEY_SC_SET_COUNT,
					     sc_sets_count);
	if (!NT_STATUS_IS_OK(status)) {
		status = NT_STATUS_UNSUCCESSFUL;
		goto err_trans_cancel;
	}

	for (sc_set = sc_sets; sc_set; sc_set = sc_set->next) {
		status = fss_state_sc_set_store(tmp_ctx, db, sc_set);
		if (!NT_STATUS_IS_OK(status)) {
			goto err_trans_cancel;
		}
	}

	ret = dbwrap_transaction_commit(db);
	if (ret != 0) {
		status = NT_STATUS_UNSUCCESSFUL;
		goto err_trans_cancel;
	}

	talloc_free(db);
	talloc_free(tmp_ctx);
	return NT_STATUS_OK;

err_trans_cancel:
	dbwrap_transaction_cancel(db);
err_db_free:
	talloc_free(db);
err_ctx_free:
	talloc_free(tmp_ctx);
	return status;
}

static bool test_fsrvp_state_empty(struct torture_context *tctx)
{
	NTSTATUS status;
	int i;
	struct fss_global fss_global;
	struct stat sbuf;
	char db_dir[] = "fsrvp_torture_XXXXXX";
	char *db_path = talloc_asprintf(NULL, "%s/%s",
					mkdtemp(db_dir), FSS_DB_NAME);
	char *db_path_v0 = talloc_asprintf(NULL, "%s/%s",
					   db_dir, "srv_fss_v0.tdb");

	memset(&fss_global, 0, sizeof(fss_global));
	fss_global.mem_ctx = talloc_new(NULL);

	status = fss_state_store(fss_global.mem_ctx, fss_global.sc_sets,
				 fss_global.sc_sets_count, db_path);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to store empty fss state");

	status = fss_state_store_v0(fss_global.mem_ctx, fss_global.sc_sets,
				    fss_global.sc_sets_count,
				    db_path_v0);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to store empty V0 fss state");

	torture_assert_int_equal(tctx, stat(db_path, &sbuf), 0,
			"failed to stat fss state tdb");
	torture_assert_int_equal(tctx, stat(db_path_v0, &sbuf), 0,
			"failed to stat V0 fss state tdb");
	talloc_free(fss_global.mem_ctx);

	memset(&fss_global, 0, sizeof(fss_global));
	fss_global.mem_ctx = talloc_new(NULL);

	status = fss_state_retrieve(fss_global.mem_ctx, &fss_global.sc_sets,
				    &fss_global.sc_sets_count,
				    db_path);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to retrieve empty fss state");
	torture_assert_int_equal(tctx, fss_global.sc_sets_count, 0,
				 "sc_sets_count set when it should be zero");

	/* read old tdb twice, first triggers upgrade, second verifies */
	for (i = 0; i < 2; i++) {
		talloc_free(fss_global.mem_ctx);
		memset(&fss_global, 0, sizeof(fss_global));
		fss_global.mem_ctx = talloc_new(NULL);
		/* retrieve and force upgrade of V0 database */
		status = fss_state_retrieve(fss_global.mem_ctx, &fss_global.sc_sets,
					    &fss_global.sc_sets_count,
					    db_path_v0);
		torture_assert_ntstatus_ok(tctx, status,
					   "failed to retrieve empty V0 fss state");
		torture_assert_int_equal(tctx, fss_global.sc_sets_count, 0,
					 "V0 sc_sets_count set when it should be zero");
	}

	talloc_free(fss_global.mem_ctx);
	unlink(db_path);
	unlink(db_path_v0);
	rmdir(db_dir);
	talloc_free(db_path);
	talloc_free(db_path_v0);

	return true;
}

static bool test_fsrvp_state_sc_set(struct torture_context *tctx,
				    TALLOC_CTX *mem_ctx,
				    struct fss_sc_set **sc_set_out)
{
	struct fss_sc_set *sc_set;

	sc_set = talloc_zero(mem_ctx, struct fss_sc_set);
	sc_set->id = GUID_random();
	sc_set->id_str = GUID_string(sc_set, &sc_set->id);
	sc_set->state = FSS_SC_COMMITED;
	sc_set->context = FSRVP_CTX_FILE_SHARE_BACKUP;
	*sc_set_out = sc_set;

	return true;
}

static bool test_fsrvp_state_sc(struct torture_context *tctx,
				TALLOC_CTX *mem_ctx,
				struct fss_sc **sc_out)
{
	struct fss_sc *sc;

	sc = talloc_zero(mem_ctx, struct fss_sc);
	sc->id = GUID_random();
	sc->id_str = GUID_string(sc, &sc->id);
	sc->volume_name = talloc_strdup(sc, "/this/is/a/path");
	/* keep snap path NULL, i.e. not yet commited */
	sc->create_ts = time(NULL);
	*sc_out = sc;

	return true;
}

static bool test_fsrvp_state_smap(struct torture_context *tctx,
				TALLOC_CTX *mem_ctx,
				const char *base_share_name,
				const char *sc_share_name,
				struct fss_sc_smap **smap_out)
{
	struct fss_sc_smap *smap;

	smap = talloc_zero(mem_ctx, struct fss_sc_smap);
	smap->share_name = talloc_strdup(mem_ctx, base_share_name);
	smap->sc_share_name = talloc_strdup(mem_ctx, sc_share_name);
	smap->sc_share_comment = talloc_strdup(mem_ctx, "test sc share comment");
	smap->is_exposed = false;
	*smap_out = smap;

	return true;
}

static bool test_fsrvp_state_smap_compare(struct torture_context *tctx,
					  struct fss_sc_smap *smap_1,
					  struct fss_sc_smap *smap_2)
{
	/* already confirmed by caller */
	torture_assert_str_equal(tctx, smap_1->sc_share_name,
				 smap_2->sc_share_name,
				 "smap sc share name strings differ");

	torture_assert_str_equal(tctx, smap_1->share_name,
				 smap_2->share_name,
				 "smap share name strings differ");

	torture_assert_str_equal(tctx, smap_1->sc_share_comment,
				 smap_2->sc_share_comment,
				 "smap sc share comment strings differ");

	torture_assert(tctx, (smap_1->is_exposed == smap_2->is_exposed),
		       "smap exposure settings differ");

	return true;
}

static bool test_fsrvp_state_sc_compare(struct torture_context *tctx,
					struct fss_sc *sc_1,
					struct fss_sc *sc_2)
{
	struct fss_sc_smap *smap_1;
	struct fss_sc_smap *smap_2;
	bool ok;

	/* should have already been confirmed by the caller */
	torture_assert(tctx, GUID_equal(&sc_1->id, &sc_2->id),
		       "sc guids differ");

	torture_assert_str_equal(tctx, sc_1->volume_name, sc_2->volume_name,
				 "sc volume_name strings differ");

	/* may be null, assert_str_eq handles null ptrs safely */
	torture_assert_str_equal(tctx, sc_1->sc_path, sc_2->sc_path,
				 "sc path strings differ");

	torture_assert(tctx, difftime(sc_1->create_ts, sc_2->create_ts) == 0,
		       "sc create timestamps differ");

	torture_assert_int_equal(tctx, sc_1->smaps_count, sc_2->smaps_count,
				 "sc smaps counts differ");

	for (smap_1 = sc_1->smaps; smap_1; smap_1 = smap_1->next) {
		bool matched = false;
		for (smap_2 = sc_2->smaps; smap_2; smap_2 = smap_2->next) {
			if (strcmp(smap_1->sc_share_name,
				   smap_2->sc_share_name) == 0) {
				matched = true;
				ok = test_fsrvp_state_smap_compare(tctx,
								   smap_1,
								   smap_2);
				torture_assert(tctx, ok, "");
				break;
			}
		}
		torture_assert(tctx, matched, "no match for smap");
	}

	return true;
}

static bool test_fsrvp_state_sc_set_compare(struct torture_context *tctx,
					    struct fss_sc_set *sc_set_1,
					    struct fss_sc_set *sc_set_2)
{
	struct fss_sc *sc_1;
	struct fss_sc *sc_2;
	bool ok;

	/* should have already been confirmed by the caller */
	torture_assert(tctx, GUID_equal(&sc_set_1->id, &sc_set_2->id),
		       "sc_set guids differ");

	torture_assert_str_equal(tctx, sc_set_1->id_str, sc_set_2->id_str,
				 "sc_set guid strings differ");

	torture_assert_int_equal(tctx, sc_set_1->state, sc_set_2->state,
				 "sc_set state enums differ");

	torture_assert_int_equal(tctx, sc_set_1->context, sc_set_2->context,
				 "sc_set contexts differ");

	torture_assert_int_equal(tctx, sc_set_1->scs_count, sc_set_2->scs_count,
				 "sc_set sc counts differ");

	for (sc_1 = sc_set_1->scs; sc_1; sc_1 = sc_1->next) {
		bool matched = false;
		for (sc_2 = sc_set_2->scs; sc_2; sc_2 = sc_2->next) {
			if (GUID_equal(&sc_1->id, &sc_2->id)) {
				matched = true;
				ok = test_fsrvp_state_sc_compare(tctx, sc_1,
								       sc_2);
				torture_assert(tctx, ok, "");
				break;
			}
		}
		torture_assert(tctx, matched, "no match for sc");
	}
	return true;
}

static bool test_fsrvp_state_compare(struct torture_context *tctx,
				     struct fss_global *fss_1,
				     struct fss_global *fss_2)
{
	struct fss_sc_set *sc_set_1;
	struct fss_sc_set *sc_set_2;
	bool ok;

	torture_assert_int_equal(tctx, fss_1->sc_sets_count,
				 fss_2->sc_sets_count,
				 "sc_sets_count differ");

	for (sc_set_1 = fss_1->sc_sets; sc_set_1; sc_set_1 = sc_set_1->next) {
		bool matched = false;
		for (sc_set_2 = fss_2->sc_sets;
		     sc_set_2;
		     sc_set_2 = sc_set_2->next) {
			if (GUID_equal(&sc_set_1->id, &sc_set_2->id)) {
				matched = true;
				ok = test_fsrvp_state_sc_set_compare(tctx,
								     sc_set_1,
								     sc_set_2);
				torture_assert(tctx, ok, "");
				break;
			}
		}
		torture_assert(tctx, matched, "no match for sc_set");
	}

	return true;
}

/*
 * test a simple heirarchy of:
 *
 *       |
 *     sc_set
 *       |
 *      sc
 *        \
 *       smap
 */
static bool test_fsrvp_state_single(struct torture_context *tctx)
{
	NTSTATUS status;
	bool ok;
	int i;
	struct fss_global fss_gs;
	struct fss_global fss_gr;
	struct fss_sc_set *sc_set;
	struct fss_sc *sc;
	struct fss_sc_smap *smap;
	char db_dir[] = "fsrvp_torture_XXXXXX";
	char *db_path = talloc_asprintf(NULL, "%s/%s",
					mkdtemp(db_dir), FSS_DB_NAME);
	char *db_path_v0 = talloc_asprintf(NULL, "%s/%s",
					   db_dir, "srv_fss_v0.tdb");

	memset(&fss_gs, 0, sizeof(fss_gs));
	fss_gs.mem_ctx = talloc_new(NULL);

	ok = test_fsrvp_state_sc_set(tctx, fss_gs.mem_ctx, &sc_set);
	torture_assert(tctx, ok, "failed to create sc set");

	/* use parent as mem ctx */
	ok = test_fsrvp_state_sc(tctx, sc_set, &sc);
	torture_assert(tctx, ok, "failed to create sc");

	ok = test_fsrvp_state_smap(tctx, sc, "base_share", "sc_share", &smap);
	torture_assert(tctx, ok, "failed to create smap");

	DLIST_ADD_END(fss_gs.sc_sets, sc_set);
	fss_gs.sc_sets_count++;
	DLIST_ADD_END(sc_set->scs, sc);
	sc_set->scs_count++;
	sc->sc_set = sc_set;
	DLIST_ADD_END(sc->smaps, smap);
	sc->smaps_count++;

	status = fss_state_store(fss_gs.mem_ctx, fss_gs.sc_sets,
				 fss_gs.sc_sets_count, db_path);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to store fss state");

	status = fss_state_store_v0(fss_gs.mem_ctx, fss_gs.sc_sets,
				    fss_gs.sc_sets_count, db_path_v0);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to store V0 fss state");

	memset(&fss_gr, 0, sizeof(fss_gr));
	fss_gr.mem_ctx = talloc_new(NULL);

	status = fss_state_retrieve(fss_gr.mem_ctx, &fss_gr.sc_sets,
				    &fss_gr.sc_sets_count, db_path);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to retrieve fss state");

	ok = test_fsrvp_state_compare(tctx, &fss_gs, &fss_gr);
	torture_assert(tctx, ok,
		       "stored and retrieved state comparison failed");

	/* read old tdb twice, first triggers upgrade, second verifies */
	for (i = 0; i < 2; i++) {
		talloc_free(fss_gr.mem_ctx);
		memset(&fss_gr, 0, sizeof(fss_gr));
		fss_gr.mem_ctx = talloc_new(NULL);
		/* retrieve and force upgrade of V0 database */
		status = fss_state_retrieve(fss_gr.mem_ctx, &fss_gr.sc_sets,
					    &fss_gr.sc_sets_count, db_path_v0);
		torture_assert_ntstatus_ok(tctx, status,
					   "failed to retrieve V0 fss state");

		ok = test_fsrvp_state_compare(tctx, &fss_gs, &fss_gr);
		torture_assert(tctx, ok,
			       "stored and upgraded state comparison failed");
	}

	talloc_free(fss_gs.mem_ctx);
	talloc_free(fss_gr.mem_ctx);
	unlink(db_path);
	unlink(db_path_v0);
	rmdir(db_dir);
	talloc_free(db_path);
	talloc_free(db_path_v0);

	return true;
}

/*
 * test a complex heirarchy of:
 *
 *              /\
 *             /  \
 *     sc_set_a    sc_set_b
 *     /     \
 * sc_aa      sc_ab
 * |          |   \
 * smap_aaa   |    \
 *            |     \
 *       smap_aba   smap_abb
 */
static bool test_fsrvp_state_multi(struct torture_context *tctx)
{
	NTSTATUS status;
	bool ok;
	int i;
	struct fss_global fss_gs;
	struct fss_global fss_gr;
	struct fss_sc_set *sc_set_a;
	struct fss_sc_set *sc_set_b;
	struct fss_sc *sc_aa;
	struct fss_sc *sc_ab;
	struct fss_sc_smap *smap_aaa;
	struct fss_sc_smap *smap_aba;
	struct fss_sc_smap *smap_abb;
	char db_dir[] = "fsrvp_torture_XXXXXX";
	char *db_path = talloc_asprintf(NULL, "%s/%s",
					mkdtemp(db_dir), FSS_DB_NAME);
	char *db_path_v0 = talloc_asprintf(NULL, "%s/%s",
					   db_dir, "srv_fss_v0.tdb");

	memset(&fss_gs, 0, sizeof(fss_gs));
	fss_gs.mem_ctx = talloc_new(NULL);

	ok = test_fsrvp_state_sc_set(tctx, fss_gs.mem_ctx, &sc_set_a);
	torture_assert(tctx, ok, "failed to create sc set");

	ok = test_fsrvp_state_sc_set(tctx, fss_gs.mem_ctx, &sc_set_b);
	torture_assert(tctx, ok, "failed to create sc set");

	/* use parent as mem ctx */
	ok = test_fsrvp_state_sc(tctx, sc_set_a, &sc_aa);
	torture_assert(tctx, ok, "failed to create sc");

	ok = test_fsrvp_state_sc(tctx, sc_set_a, &sc_ab);
	torture_assert(tctx, ok, "failed to create sc");

	ok = test_fsrvp_state_smap(tctx, sc_ab, "share_aa", "sc_share_aaa",
				   &smap_aaa);
	torture_assert(tctx, ok, "failed to create smap");

	ok = test_fsrvp_state_smap(tctx, sc_ab, "share_ab", "sc_share_aba",
				   &smap_aba);
	torture_assert(tctx, ok, "failed to create smap");

	ok = test_fsrvp_state_smap(tctx, sc_ab, "share_ab", "sc_share_abb",
				   &smap_abb);
	torture_assert(tctx, ok, "failed to create smap");

	DLIST_ADD_END(fss_gs.sc_sets, sc_set_a);
	fss_gs.sc_sets_count++;
	DLIST_ADD_END(fss_gs.sc_sets, sc_set_b);
	fss_gs.sc_sets_count++;

	DLIST_ADD_END(sc_set_a->scs, sc_aa);
	sc_set_a->scs_count++;
	sc_aa->sc_set = sc_set_a;
	DLIST_ADD_END(sc_set_a->scs, sc_ab);
	sc_set_a->scs_count++;
	sc_ab->sc_set = sc_set_a;

	DLIST_ADD_END(sc_aa->smaps, smap_aaa);
	sc_aa->smaps_count++;
	DLIST_ADD_END(sc_ab->smaps, smap_aba);
	sc_ab->smaps_count++;
	DLIST_ADD_END(sc_ab->smaps, smap_abb);
	sc_ab->smaps_count++;

	status = fss_state_store(fss_gs.mem_ctx, fss_gs.sc_sets,
				 fss_gs.sc_sets_count, db_path);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to store fss state");

	status = fss_state_store_v0(fss_gs.mem_ctx, fss_gs.sc_sets,
				    fss_gs.sc_sets_count, db_path_v0);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to store V0 fss state");

	memset(&fss_gr, 0, sizeof(fss_gr));
	fss_gr.mem_ctx = talloc_new(NULL);
	status = fss_state_retrieve(fss_gr.mem_ctx, &fss_gr.sc_sets,
				    &fss_gr.sc_sets_count, db_path);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to retrieve fss state");

	ok = test_fsrvp_state_compare(tctx, &fss_gs, &fss_gr);
	torture_assert(tctx, ok,
		       "stored and retrieved state comparison failed");

	/* read old tdb twice, first triggers upgrade, second verifies */
	for (i = 0; i < 2; i++) {
		talloc_free(fss_gr.mem_ctx);
		memset(&fss_gr, 0, sizeof(fss_gr));
		fss_gr.mem_ctx = talloc_new(NULL);
		/* retrieve and force upgrade of V0 database */
		status = fss_state_retrieve(fss_gr.mem_ctx, &fss_gr.sc_sets,
					    &fss_gr.sc_sets_count, db_path_v0);
		torture_assert_ntstatus_ok(tctx, status,
					   "failed to retrieve V0 fss state");

		ok = test_fsrvp_state_compare(tctx, &fss_gs, &fss_gr);
		torture_assert(tctx, ok,
			       "stored and upgraded state comparison failed");
	}

	talloc_free(fss_gs.mem_ctx);
	talloc_free(fss_gr.mem_ctx);
	unlink(db_path);
	unlink(db_path_v0);
	rmdir(db_dir);
	talloc_free(db_path);
	talloc_free(db_path_v0);

	return true;
}

static bool test_fsrvp_state_none(struct torture_context *tctx)
{
	NTSTATUS status;
	struct fss_global fss_global;
	char db_dir[] = "fsrvp_torture_XXXXXX";
	char *db_path = talloc_asprintf(NULL, "%s/%s",
					mkdtemp(db_dir), FSS_DB_NAME);

	memset(&fss_global, 0, sizeof(fss_global));
	fss_global.mem_ctx = talloc_new(NULL);
	fss_global.db_path = db_path;

	status = fss_state_retrieve(fss_global.mem_ctx, &fss_global.sc_sets,
				    &fss_global.sc_sets_count,
				    fss_global.db_path);
	torture_assert_ntstatus_ok(tctx, status,
				   "failed to retrieve fss state");
	torture_assert_int_equal(tctx, fss_global.sc_sets_count, 0,
				 "sc_sets_count set when it should be zero");
	talloc_free(fss_global.mem_ctx);
	unlink(db_path);
	rmdir(db_dir);
	talloc_free(db_path);

	return true;
}

struct torture_suite *torture_local_fsrvp(TALLOC_CTX *mem_ctx)
{
	struct torture_suite *suite = torture_suite_create(mem_ctx,
							   "fsrvp_state");

	/* dbwrap uses talloc_tos(), hence we need a stackframe :( */
	talloc_stackframe();

	torture_suite_add_simple_test(suite,
				      "state_empty",
				      test_fsrvp_state_empty);

	torture_suite_add_simple_test(suite,
				      "state_single",
				      test_fsrvp_state_single);

	torture_suite_add_simple_test(suite,
				      "state_multi",
				      test_fsrvp_state_multi);

	torture_suite_add_simple_test(suite,
				      "state_none",
				      test_fsrvp_state_none);

	return suite;
}
