/*
 * livepatch_bsc1231676
 *
 * Fix for CVE-2024-47674, bsc#1231676
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Vincenzo Mezzela <vincenzo.mezzela@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from mm/memory.c */
#include <linux/kernel_stat.h>
#include <linux/mm.h>


#ifdef __HAVE_ARCH_PFN_MODIFY_ALLOWED
static bool (*klpe_pfn_modify_allowed)(unsigned long pfn, pgprot_t prot);
#define pfn_modify_allowed (*klpe_pfn_modify_allowed)
#else
static inline bool klpr_pfn_modify_allowed(unsigned long pfn, pgprot_t prot)
{
	return true;
}
#define pfn_modify_allowed klpr_pfn_modify_allowed
#endif /* !_HAVE_ARCH_PFN_MODIFY_ALLOWED */


#if defined(CONFIG_S390)


/* In s390 arch, the set_pte_at function included from <asm/pgtable.h>, uses
 * the function ptep_set_pte_at defined in arch/s390/mm/pagetable.c, such
 * function is not exported to modules, hence resolve it through livepatching
 * externalization.
 */

static void (*klpe_ptep_set_pte_at)(struct mm_struct *mm, unsigned long addr, pte_t
			     *ptep, pte_t entry);

static inline void klpr_set_pte_at(struct mm_struct *mm, unsigned long addr,
			      pte_t *ptep, pte_t entry)
{
	if (pte_present(entry))
		pte_val(entry) &= ~_PAGE_UNUSED;
	if (mm_has_pgste(mm))
		(*klpe_ptep_set_pte_at)(mm, addr, ptep, entry);
	else
		*ptep = entry;
}

#define set_pte_at klpr_set_pte_at
#endif

#if defined(CONFIG_PPC64)

 /* The symbols __flush_tlb_pending and ppc64_tlb_batch used by
  * arch_enter_lazy_mmu_mode and arch_leave_lazy_mmu_mode need to be resolved
  * through livepatching externalization for ppc64le */

#include <linux/percpu.h>

static struct ppc64_tlb_batch __percpu (*klpe_ppc64_tlb_batch);

static inline void klpr_arch_enter_lazy_mmu_mode(void)
{
	struct ppc64_tlb_batch *batch;

	if (radix_enabled())
		return;
	batch = this_cpu_ptr(klpe_ppc64_tlb_batch);
	batch->active = 1;
}

static void (*klpe___flush_tlb_pending)(struct ppc64_tlb_batch *batch);
static inline void klpr_arch_leave_lazy_mmu_mode(void)
{
	struct ppc64_tlb_batch *batch;

	if (radix_enabled())
		return;
	batch = this_cpu_ptr(klpe_ppc64_tlb_batch);

	if (batch->index)
		(*klpe___flush_tlb_pending)(batch);
	batch->active = 0;
}

#define arch_enter_lazy_mmu_mode klpr_arch_enter_lazy_mmu_mode
#define arch_leave_lazy_mmu_mode klpr_arch_leave_lazy_mmu_mode

/* pmd_page seems to be implemented as macro for the other archs but not for
 * ppc64le, thus it results in the symbol needing to be externalized. In
 * particular, this is used by pte_lockptr in the macro expanded in
 * klpr_remap_pte_range by klp-ccp.
 */

#if USE_SPLIT_PTE_PTLOCKS
static struct page *(*klpe_pmd_page)(pmd_t pmd);
static inline spinlock_t *klpr_pte_lockptr(struct mm_struct *mm, pmd_t *pmd)
{
	return ptlock_ptr((*klpe_pmd_page)(*pmd));
}

#define pte_lockptr klpr_pte_lockptr
#endif /* USE_SPLIT_PTE_PTLOCKS */

/* Resolve  set_pte_at through externalization */
static void (*klpe_set_pte_at)(struct mm_struct *mm, unsigned long addr, pte_t *ptep,
			pte_t pte);

#define set_pte_at (*klpe_set_pte_at)

#endif /* CONFIG_PPC64 */

/* klp-ccp: from include/linux/mm.h */
#ifdef __PAGETABLE_P4D_FOLDED
static inline int klpr___p4d_alloc(struct mm_struct *mm, pgd_t *pgd,
						unsigned long address)
{
	return 0;
}
#define __p4d_alloc klpr___p4d_alloc
#else
static int (*klpe___p4d_alloc)(struct mm_struct *mm, pgd_t *pgd, unsigned long address);
#define __p4d_alloc (*klpe___p4d_alloc)
#endif

#if defined(__PAGETABLE_PUD_FOLDED) || !defined(CONFIG_MMU)
#error "klp-ccp: non-taken branch"
#else
static int (*klpe___pud_alloc)(struct mm_struct *mm, p4d_t *p4d, unsigned long address);

#endif

#if defined(__PAGETABLE_PMD_FOLDED) || !defined(CONFIG_MMU)
#error "klp-ccp: non-taken branch"
#else
static int (*klpe___pmd_alloc)(struct mm_struct *mm, pud_t *pud, unsigned long address);

#endif

static int (*klpe___pte_alloc)(struct mm_struct *mm, pmd_t *pmd);

#if defined(CONFIG_MMU)

static inline p4d_t *klpr_p4d_alloc(struct mm_struct *mm, pgd_t *pgd,
		unsigned long address)
{
	return (unlikely(pgd_none(*pgd)) && __p4d_alloc(mm, pgd, address)) ?
		NULL : p4d_offset(pgd, address);
}

static inline pud_t *klpr_pud_alloc(struct mm_struct *mm, p4d_t *p4d,
		unsigned long address)
{
	return (unlikely(p4d_none(*p4d)) && (*klpe___pud_alloc)(mm, p4d, address)) ?
		NULL : pud_offset(p4d, address);
}

static inline pmd_t *klpr_pmd_alloc(struct mm_struct *mm, pud_t *pud, unsigned long address)
{
	return (unlikely(pud_none(*pud)) && (*klpe___pmd_alloc)(mm, pud, address))?
		NULL: pmd_offset(pud, address);
}
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif /* CONFIG_MMU */

int klpp_remap_pfn_range_notrack(struct vm_area_struct *vma, unsigned long addr,
		unsigned long pfn, unsigned long size, pgprot_t prot);


/* klp-ccp: from include/linux/sched/mm.h */
#define _LINUX_SCHED_MM_H

/* klp-ccp: from mm/memory.c */
#include <linux/sched/coredump.h>

#include <linux/sched/task.h>
#include <linux/hugetlb.h>

/* klp-ccp: from include/linux/memcontrol.h */
#define _LINUX_MEMCONTROL_H

/* klp-ccp: from include/linux/writeback.h */
#define WRITEBACK_H

/* klp-ccp: from arch/x86/include/asm/io.h */
#define _ASM_X86_IO_H

/* klp-ccp: from include/linux/vmalloc.h */
#define _LINUX_VMALLOC_H

/* klp-ccp: from mm/memory.c */
#include <linux/highmem.h>
#include <linux/pagemap.h>
#include <linux/memremap.h>

/* klp-ccp: from include/linux/rmap.h */
#define _LINUX_RMAP_H

/* klp-ccp: from mm/memory.c */
#include <linux/rmap.h>
#include <linux/export.h>

#include <linux/init.h>

#include <linux/writeback.h>
#include <linux/memcontrol.h>

#include <linux/elf.h>
#include <linux/gfp.h>

#include <linux/string.h>

#include <linux/userfaultfd_k.h>
#include <linux/dax.h>

#include <linux/numa.h>

/* klp-ccp: from include/linux/ptrace.h */
#define _LINUX_PTRACE_H

/* klp-ccp: from mm/memory.c */
#include <linux/ptrace.h>
#include <linux/vmalloc.h>

#include <asm/io.h>

#include <linux/uaccess.h>

#include <asm/tlbflush.h>

/* klp-ccp: from mm/internal.h */
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/tracepoint-defs.h>

static void (*klpe_zap_page_range_single)(struct vm_area_struct *vma, unsigned long address,
		unsigned long size, struct zap_details *details);

/* klp-ccp: not from file */
#undef pte_offset_kernel

/* klp-ccp: from mm/memory.c */
static int klpr_remap_pte_range(struct mm_struct *mm, pmd_t *pmd,
			unsigned long addr, unsigned long end,
			unsigned long pfn, pgprot_t prot)
{
	pte_t *pte, *mapped_pte;
	spinlock_t *ptl;
	int err = 0;

	mapped_pte = pte = ((__builtin_expect(!!(pmd_none(*(pmd))), 0) && (*klpe___pte_alloc)(mm, pmd)) ? ((void *)0) : ({ spinlock_t *__ptl = pte_lockptr(mm, pmd); pte_t *__pte = pte_offset_kernel((pmd), (addr)); *(&ptl) = __ptl; spin_lock(__ptl); __pte; }));
	if (!pte)
		return -ENOMEM;
	arch_enter_lazy_mmu_mode();
	do {
		BUG_ON(!pte_none(*pte));
		if (!pfn_modify_allowed(pfn, prot)) {
			err = -EACCES;
			break;
		}
		set_pte_at(mm, addr, pte, pte_mkspecial(pfn_pte(pfn, prot)));
		pfn++;
	} while (pte++, addr += PAGE_SIZE, addr != end);
	arch_leave_lazy_mmu_mode();
	pte_unmap_unlock(mapped_pte, ptl);
	return err;
}

static inline int klpr_remap_pmd_range(struct mm_struct *mm, pud_t *pud,
			unsigned long addr, unsigned long end,
			unsigned long pfn, pgprot_t prot)
{
	pmd_t *pmd;
	unsigned long next;
	int err;

	pfn -= addr >> PAGE_SHIFT;
	pmd = klpr_pmd_alloc(mm, pud, addr);
	if (!pmd)
		return -ENOMEM;
	VM_BUG_ON(pmd_trans_huge(*pmd));
	do {
		next = pmd_addr_end(addr, end);
		err = klpr_remap_pte_range(mm, pmd, addr, next,
				pfn + (addr >> PAGE_SHIFT), prot);
		if (err)
			return err;
	} while (pmd++, addr = next, addr != end);
	return 0;
}

static inline int klpr_remap_pud_range(struct mm_struct *mm, p4d_t *p4d,
			unsigned long addr, unsigned long end,
			unsigned long pfn, pgprot_t prot)
{
	pud_t *pud;
	unsigned long next;
	int err;

	pfn -= addr >> PAGE_SHIFT;
	pud = klpr_pud_alloc(mm, p4d, addr);
	if (!pud)
		return -ENOMEM;
	do {
		next = pud_addr_end(addr, end);
		err = klpr_remap_pmd_range(mm, pud, addr, next,
				pfn + (addr >> PAGE_SHIFT), prot);
		if (err)
			return err;
	} while (pud++, addr = next, addr != end);
	return 0;
}

static inline int klpr_remap_p4d_range(struct mm_struct *mm, pgd_t *pgd,
			unsigned long addr, unsigned long end,
			unsigned long pfn, pgprot_t prot)
{
	p4d_t *p4d;
	unsigned long next;
	int err;

	pfn -= addr >> PAGE_SHIFT;
	p4d = klpr_p4d_alloc(mm, pgd, addr);
	if (!p4d)
		return -ENOMEM;
	do {
		next = p4d_addr_end(addr, end);
		err = klpr_remap_pud_range(mm, p4d, addr, next,
				pfn + (addr >> PAGE_SHIFT), prot);
		if (err)
			return err;
	} while (p4d++, addr = next, addr != end);
	return 0;
}

static int klpr_remap_pfn_range_internal(struct vm_area_struct *vma, unsigned long addr,
		unsigned long pfn, unsigned long size, pgprot_t prot)
{
	pgd_t *pgd;
	unsigned long next;
	unsigned long end = addr + PAGE_ALIGN(size);
	struct mm_struct *mm = vma->vm_mm;
	int err;

	if (WARN_ON_ONCE(!PAGE_ALIGNED(addr)))
		return -EINVAL;

	/*
	 * Physically remapped pages are special. Tell the
	 * rest of the world about it:
	 *   VM_IO tells people not to look at these pages
	 *	(accesses can have side effects).
	 *   VM_PFNMAP tells the core MM that the base pages are just
	 *	raw PFN mappings, and do not have a "struct page" associated
	 *	with them.
	 *   VM_DONTEXPAND
	 *      Disable vma merging and expanding with mremap().
	 *   VM_DONTDUMP
	 *      Omit vma from core dump, even when VM_IO turned off.
	 *
	 * There's a horrible special case to handle copy-on-write
	 * behaviour that some programs depend on. We mark the "original"
	 * un-COW'ed pages by matching them up with "vma->vm_pgoff".
	 * See vm_normal_page() for details.
	 */
	if (is_cow_mapping(vma->vm_flags)) {
		if (addr != vma->vm_start || end != vma->vm_end)
			return -EINVAL;
		vma->vm_pgoff = pfn;
	}

	vma->vm_flags |= VM_IO | VM_PFNMAP | VM_DONTEXPAND | VM_DONTDUMP;

	BUG_ON(addr >= end);
	pfn -= addr >> PAGE_SHIFT;
	pgd = pgd_offset(mm, addr);
	flush_cache_range(vma, addr, end);
	do {
		next = pgd_addr_end(addr, end);
		err = klpr_remap_p4d_range(mm, pgd, addr, next,
				pfn + (addr >> PAGE_SHIFT), prot);
		if (err)
			return err;
	} while (pgd++, addr = next, addr != end);

	return 0;
}

int klpp_remap_pfn_range_notrack(struct vm_area_struct *vma, unsigned long addr,
		unsigned long pfn, unsigned long size, pgprot_t prot)
{
	int error = klpr_remap_pfn_range_internal(vma, addr, pfn, size, prot);

	if (!error)
		return 0;

	/*
	 * A partial pfn range mapping is dangerous: it does not
	 * maintain page reference counts, and callers may free
	 * pages due to the error. So zap it early.
	 */
	(*klpe_zap_page_range_single)(vma, addr, size, NULL);
	return error;
}


#include "livepatch_bsc1231676.h"

#include <linux/kernel.h>
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
#ifndef __PAGETABLE_P4D_FOLDED
	{ "__p4d_alloc", (void *)&klpe___p4d_alloc },
#endif
	{ "__pmd_alloc", (void *)&klpe___pmd_alloc },
	{ "__pte_alloc", (void *)&klpe___pte_alloc },
	{ "__pud_alloc", (void *)&klpe___pud_alloc },
	{ "zap_page_range_single", (void *)&klpe_zap_page_range_single },
#ifdef __HAVE_ARCH_PFN_MODIFY_ALLOWED
	{ "pfn_modify_allowed", (void *)&klpe_pfn_modify_allowed },
#endif
#if defined(CONFIG_S390)
	{ "ptep_set_pte_at", (void *)&klpe_ptep_set_pte_at },
#endif
#if defined(CONFIG_PPC64)
	{ "__flush_tlb_pending", (void *)&klpe___flush_tlb_pending },
	{ "pmd_page", (void *)&klpe_pmd_page },
	{ "ppc64_tlb_batch", (void *)&klpe_ppc64_tlb_batch },
	{ "set_pte_at", (void *)&klpe_set_pte_at },
#endif
};

int livepatch_bsc1231676_init(void)
{
	return klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

