/*
 * livepatch_bsc1236783
 *
 * Fix for CVE-2024-53104, bsc#1236783
 *
 *  Upstream commit:
 *  ecf2b43018da ("media: uvcvideo: Skip parsing frames of type UVC_VS_UNDEFINED in uvc_parse_format")
 *
 *  SLE12-SP5 commit:
 *  5e374e6f596d3d5704dfd0efeab72dbdc91cc340
 *
 *  SLE15-SP3 commit:
 *  Not affected
 *
 *  SLE15-SP4 and -SP5 commit:
 *  a0c98f3a2d23ab9fb6dc3d6c2c1c252ebdb95a27
 *
 *  SLE15-SP6 commit:
 *  a0907f284ed53dacbcc62e9b83d7b192cec7df9d
 *
 *  SLE MICRO-6-0 commit:
 *  a0907f284ed53dacbcc62e9b83d7b192cec7df9d
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Vincenzo MEZZELA <vincenzo.mezzela@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_USB_VIDEO_CLASS)

#if !IS_MODULE(CONFIG_USB_VIDEO_CLASS)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from drivers/media/usb/uvc/uvc_driver.c */
#include <linux/atomic.h>
#include <linux/kernel.h>

/* klp-ccp: from include/linux/export.h */
#ifndef __ASSEMBLY__
#ifdef MODULE
static struct module (*klpe___this_module);

#else
#error "klp-ccp: non-taken branch"
#endif

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif /* !__ASSEMBLY__ */

/* klp-ccp: from drivers/media/usb/uvc/uvc_driver.c */
#include <linux/list.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/usb.h>

/* klp-ccp: from include/linux/usb.h */
#ifdef __KERNEL__

static struct usb_interface *(*klpe_usb_get_intf)(struct usb_interface *intf);

static struct usb_device *(*klpe_usb_get_dev)(struct usb_device *dev);

#ifdef CONFIG_PM
static void (*klpe_usb_enable_autosuspend)(struct usb_device *udev);

#else
#error "klp-ccp: non-taken branch"
#endif

static int (*klpe_usb_driver_claim_interface)(struct usb_driver *driver,
			struct usb_interface *iface, void *priv);

static void (*klpe_usb_driver_release_interface)(struct usb_driver *driver,
			struct usb_interface *iface);

static struct usb_interface *(*klpe_usb_ifnum_to_if)(const struct usb_device *dev,
		unsigned ifnum);

static int (*klpe_usb_string)(struct usb_device *dev, int index,
	char *buf, size_t size);

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif  /* __KERNEL__ */

/* klp-ccp: from drivers/media/usb/uvc/uvc_driver.c */
#include <linux/videodev2.h>
#include <linux/vmalloc.h>
#include <linux/wait.h>
#include <linux/version.h>
#include <asm/unaligned.h>

#include <media/v4l2-common.h>

/* klp-ccp: from include/media/v4l2-common.h */
static const struct v4l2_format_info *(*klpe_v4l2_format_info)(u32 format);

/* klp-ccp: from drivers/media/usb/uvc/uvc_driver.c */
#include <media/v4l2-ioctl.h>

/* klp-ccp: from drivers/media/usb/uvc/uvcvideo.h */
#include <linux/kernel.h>
#include <linux/poll.h>
#include <linux/usb.h>
#include <linux/usb/video.h>

#include <linux/videodev2.h>
#include <linux/workqueue.h>
#include <media/media-device.h>

/* klp-ccp: from include/media/media-device.h */
#ifdef CONFIG_MEDIA_CONTROLLER

static void (*klpe_media_device_init)(struct media_device *mdev);

static int __must_check (*klpe___media_device_register)(struct media_device *mdev,
					 struct module *owner);

#else
#error "klp-ccp: non-taken branch"
#endif /* CONFIG_MEDIA_CONTROLLER */

/* klp-ccp: from drivers/media/usb/uvc/uvcvideo.h */
#include <media/v4l2-device.h>

/* klp-ccp: from include/media/v4l2-device.h */
static int __must_check (*klpe_v4l2_device_register)(struct device *dev,
				      struct v4l2_device *v4l2_dev);

/* klp-ccp: from drivers/media/usb/uvc/uvcvideo.h */
#include <media/v4l2-fh.h>
#include <media/videobuf2-v4l2.h>

#define UVC_TERM_INPUT			0x0000
#define UVC_TERM_OUTPUT			0x8000

#define UVC_ENTITY_TYPE(entity)		((entity)->type & 0x7fff)

#define UVC_ENTITY_IS_TERM(entity)	(((entity)->type & 0xff00) != 0)
#define UVC_ENTITY_IS_ITERM(entity) \
	(UVC_ENTITY_IS_TERM(entity) && \
	((entity)->type & 0x8000) == UVC_TERM_INPUT)
#define UVC_ENTITY_IS_OTERM(entity) \
	(UVC_ENTITY_IS_TERM(entity) && \
	((entity)->type & 0x8000) == UVC_TERM_OUTPUT)

#define UVC_URBS		5

#define UVC_MAX_PACKETS		32

#define UVC_QUIRK_BUILTIN_ISIGHT	0x00000008

#define UVC_QUIRK_RESTRICT_FRAME_RATE	0x00000200

#define UVC_QUIRK_FORCE_Y8		0x00000800
#define UVC_QUIRK_FORCE_BPP		0x00001000

#define UVC_FMT_FLAG_COMPRESSED		0x00000001
#define UVC_FMT_FLAG_STREAM		0x00000002

struct uvc_format_desc {
	char *name;
	u8 guid[16];
	u32 fcc;
};

#define UVC_ENTITY_FLAG_DEFAULT		(1 << 0)

struct uvc_entity {
	struct list_head list;		/* Entity as part of a UVC device. */
	struct list_head chain;		/* Entity as part of a video device
					 * chain. */
	unsigned int flags;

	u8 id;
	u16 type;
	char name[64];

	/* Media controller-related fields. */
	struct video_device *vdev;
	struct v4l2_subdev subdev;
	unsigned int num_pads;
	unsigned int num_links;
	struct media_pad *pads;

	union {
		struct {
			u16 wObjectiveFocalLengthMin;
			u16 wObjectiveFocalLengthMax;
			u16 wOcularFocalLength;
			u8  bControlSize;
			u8  *bmControls;
		} camera;

		struct {
			u8  bControlSize;
			u8  *bmControls;
			u8  bTransportModeSize;
			u8  *bmTransportModes;
		} media;

		struct {
		} output;

		struct {
			u16 wMaxMultiplier;
			u8  bControlSize;
			u8  *bmControls;
			u8  bmVideoStandards;
		} processing;

		struct {
		} selector;

		struct {
			u8  guidExtensionCode[16];
			u8  bNumControls;
			u8  bControlSize;
			u8  *bmControls;
			u8  *bmControlsType;
		} extension;
	};

	u8 bNrInPins;
	u8 *baSourceID;

	unsigned int ncontrols;
	struct uvc_control *controls;
};

struct uvc_frame {
	u8  bFrameIndex;
	u8  bmCapabilities;
	u16 wWidth;
	u16 wHeight;
	u32 dwMinBitRate;
	u32 dwMaxBitRate;
	u32 dwMaxVideoFrameBufferSize;
	u8  bFrameIntervalType;
	u32 dwDefaultFrameInterval;
	u32 *dwFrameInterval;
};

struct uvc_format {
	u8 type;
	u8 index;
	u8 bpp;
	u8 colorspace;
	u32 fcc;
	u32 flags;

	char name[32];

	unsigned int nframes;
	struct uvc_frame *frame;
};

struct uvc_streaming_header {
	u8 bNumFormats;
	u8 bEndpointAddress;
	u8 bTerminalLink;
	u8 bControlSize;
	u8 *bmaControls;
	/* The following fields are used by input headers only. */
	u8 bmInfo;
	u8 bStillCaptureMethod;
	u8 bTriggerSupport;
	u8 bTriggerUsage;
};

struct uvc_video_queue {
	struct vb2_queue queue;
	struct mutex mutex;			/* Protects queue */

	unsigned int flags;
	unsigned int buf_used;

	spinlock_t irqlock;			/* Protects irqqueue */
	struct list_head irqqueue;
};

struct uvc_video_chain {
	struct uvc_device *dev;
	struct list_head list;

	struct list_head entities;		/* All entities */
	struct uvc_entity *processing;		/* Processing unit */
	struct uvc_entity *selector;		/* Selector unit */

	struct mutex ctrl_mutex;		/* Protects ctrl.info */

	struct v4l2_prio_state prio;		/* V4L2 priority state */
	u32 caps;				/* V4L2 chain-wide caps */
};

struct uvc_stats_frame {
	unsigned int size;		/* Number of bytes captured */
	unsigned int first_data;	/* Index of the first non-empty packet */

	unsigned int nb_packets;	/* Number of packets */
	unsigned int nb_empty;		/* Number of empty packets */
	unsigned int nb_invalid;	/* Number of packets with an invalid header */
	unsigned int nb_errors;		/* Number of packets with the error bit set */

	unsigned int nb_pts;		/* Number of packets with a PTS timestamp */
	unsigned int nb_pts_diffs;	/* Number of PTS differences inside a frame */
	unsigned int last_pts_diff;	/* Index of the last PTS difference */
	bool has_initial_pts;		/* Whether the first non-empty packet has a PTS */
	bool has_early_pts;		/* Whether a PTS is present before the first non-empty packet */
	u32 pts;			/* PTS of the last packet */

	unsigned int nb_scr;		/* Number of packets with a SCR timestamp */
	unsigned int nb_scr_diffs;	/* Number of SCR.STC differences inside a frame */
	u16 scr_sof;			/* SCR.SOF of the last packet */
	u32 scr_stc;			/* SCR.STC of the last packet */
};

struct uvc_stats_stream {
	ktime_t start_ts;		/* Stream start timestamp */
	ktime_t stop_ts;		/* Stream stop timestamp */

	unsigned int nb_frames;		/* Number of frames */

	unsigned int nb_packets;	/* Number of packets */
	unsigned int nb_empty;		/* Number of empty packets */
	unsigned int nb_invalid;	/* Number of packets with an invalid header */
	unsigned int nb_errors;		/* Number of packets with the error bit set */

	unsigned int nb_pts_constant;	/* Number of frames with constant PTS */
	unsigned int nb_pts_early;	/* Number of frames with early PTS */
	unsigned int nb_pts_initial;	/* Number of frames with initial PTS */

	unsigned int nb_scr_count_ok;	/* Number of frames with at least one SCR per non empty packet */
	unsigned int nb_scr_diffs_ok;	/* Number of frames with varying SCR.STC */
	unsigned int scr_sof_count;	/* STC.SOF counter accumulated since stream start */
	unsigned int scr_sof;		/* STC.SOF of the last packet */
	unsigned int min_sof;		/* Minimum STC.SOF value */
	unsigned int max_sof;		/* Maximum STC.SOF value */
};

struct uvc_copy_op {
	struct uvc_buffer *buf;
	void *dst;
	const __u8 *src;
	size_t len;
};

struct uvc_urb {
	struct urb *urb;
	struct uvc_streaming *stream;

	char *buffer;
	dma_addr_t dma;

	unsigned int async_operations;
	struct uvc_copy_op copy_operations[UVC_MAX_PACKETS];
	struct work_struct work;
};

struct uvc_streaming {
	struct list_head list;
	struct uvc_device *dev;
	struct video_device vdev;
	struct uvc_video_chain *chain;
	atomic_t active;

	struct usb_interface *intf;
	int intfnum;
	u16 maxpsize;

	struct uvc_streaming_header header;
	enum v4l2_buf_type type;

	unsigned int nformats;
	struct uvc_format *format;

	struct uvc_streaming_control ctrl;
	struct uvc_format *def_format;
	struct uvc_format *cur_format;
	struct uvc_frame *cur_frame;

	/* Protect access to ctrl, cur_format, cur_frame and hardware video
	 * probe control.
	 */
	struct mutex mutex;

	/* Buffers queue. */
	unsigned int frozen : 1;
	struct uvc_video_queue queue;
	struct workqueue_struct *async_wq;
	void (*decode)(struct uvc_urb *uvc_urb, struct uvc_buffer *buf,
		       struct uvc_buffer *meta_buf);

	struct {
		struct video_device vdev;
		struct uvc_video_queue queue;
		u32 format;
	} meta;

	/* Context data used by the bulk completion handler. */
	struct {
		u8 header[256];
		unsigned int header_size;
		int skip_payload;
		u32 payload_size;
		u32 max_payload_size;
	} bulk;

	struct uvc_urb uvc_urb[UVC_URBS];
	unsigned int urb_size;

	u32 sequence;
	u8 last_fid;

	/* debugfs */
	struct dentry *debugfs_dir;
	struct {
		struct uvc_stats_frame frame;
		struct uvc_stats_stream stream;
	} stats;

	/* Timestamps support. */
	struct uvc_clock {
		struct uvc_clock_sample {
			u32 dev_stc;
			u16 dev_sof;
			u16 host_sof;
			ktime_t host_time;
		} *samples;

		unsigned int head;
		unsigned int count;
		unsigned int size;

		u16 last_sof;
		u16 sof_offset;

		u8 last_scr[6];

		spinlock_t lock;
	} clock;
};

struct uvc_device_info {
	u32	quirks;
	u32	meta_format;
};

struct uvc_device {
	struct usb_device *udev;
	struct usb_interface *intf;
	unsigned long warnings;
	u32 quirks;
	int intfnum;
	char name[32];

	const struct uvc_device_info *info;

	struct mutex lock;		/* Protects users */
	unsigned int users;
	atomic_t nmappings;

#ifdef CONFIG_MEDIA_CONTROLLER
	struct media_device mdev;
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	struct v4l2_device vdev;
	u16 uvc_version;
	u32 clock_frequency;

	struct list_head entities;
	struct list_head chains;

	/* Video Streaming interfaces */
	struct list_head streams;
	struct kref ref;

	/* Status Interrupt Endpoint */
	struct usb_host_endpoint *int_ep;
	struct urb *int_urb;
	u8 *status;
	struct input_dev *input;
	char input_phys[64];

	struct uvc_ctrl_work {
		struct work_struct work;
		struct urb *urb;
		struct uvc_video_chain *chain;
		struct uvc_control *ctrl;
		const void *data;
	} async_ctrl;
};

struct uvc_driver {
	struct usb_driver driver;
};

#define UVC_TRACE_PROBE		(1 << 0)

static unsigned int (*klpe_uvc_trace_param);

#define uvc_printk(level, msg...) \
	printk(level "uvcvideo: " msg)

static struct uvc_driver (*klpe_uvc_driver);

static struct uvc_entity *(*klpe_uvc_entity_by_id)(struct uvc_device *dev, int id);

static const struct v4l2_ioctl_ops (*klpe_uvc_ioctl_ops);
static const struct v4l2_file_operations (*klpe_uvc_fops);

static int (*klpe_uvc_mc_register_entities)(struct uvc_video_chain *chain);

static int (*klpe_uvc_video_init)(struct uvc_streaming *stream);

static int (*klpe_uvc_meta_register)(struct uvc_streaming *stream);

static int (*klpe_uvc_register_video_device)(struct uvc_device *dev,
			      struct uvc_streaming *stream,
			      struct video_device *vdev,
			      struct uvc_video_queue *queue,
			      enum v4l2_buf_type type,
			      const struct v4l2_file_operations *fops,
			      const struct v4l2_ioctl_ops *ioctl_ops);

static int (*klpe_uvc_status_init)(struct uvc_device *dev);

static int (*klpe_uvc_ctrl_init_device)(struct uvc_device *dev);

static struct usb_host_endpoint *(*klpe_uvc_find_endpoint)(struct usb_host_interface *alts,
					    u8 epaddr);

static void (*klpe_uvc_debugfs_init_stream)(struct uvc_streaming *stream);

/* klp-ccp: from drivers/media/usb/uvc/uvc_driver.c */
static unsigned int (*klpe_uvc_quirks_param);

static struct uvc_format_desc (*klpe_uvc_fmts)[35];

static struct uvc_format_desc *klpr_uvc_format_by_guid(const u8 guid[16])
{
	unsigned int len = ARRAY_SIZE((*klpe_uvc_fmts));
	unsigned int i;

	for (i = 0; i < len; ++i) {
		if (memcmp(guid, (*klpe_uvc_fmts)[i].guid, 16) == 0)
			return &(*klpe_uvc_fmts)[i];
	}

	return NULL;
}

static u32 uvc_colorspace(const u8 primaries)
{
	static const u8 colorprimaries[] = {
		0,
		V4L2_COLORSPACE_SRGB,
		V4L2_COLORSPACE_470_SYSTEM_M,
		V4L2_COLORSPACE_470_SYSTEM_BG,
		V4L2_COLORSPACE_SMPTE170M,
		V4L2_COLORSPACE_SMPTE240M,
	};

	if (primaries < ARRAY_SIZE(colorprimaries))
		return colorprimaries[primaries];

	return 0;
}


static struct uvc_entity *uvc_entity_by_reference(struct uvc_device *dev,
	int id, struct uvc_entity *entity)
{
	unsigned int i;

	if (entity == NULL)
		entity = list_entry(&dev->entities, struct uvc_entity, list);

	list_for_each_entry_continue(entity, &dev->entities, list) {
		for (i = 0; i < entity->bNrInPins; ++i)
			if (entity->baSourceID[i] == id)
				return entity;
	}

	return NULL;
}

static struct uvc_streaming *uvc_stream_by_id(struct uvc_device *dev, int id)
{
	struct uvc_streaming *stream;

	list_for_each_entry(stream, &dev->streams, list) {
		if (stream->header.bTerminalLink == id)
			return stream;
	}

	return NULL;
}

static void (*klpe_uvc_stream_delete)(struct uvc_streaming *stream);

static struct uvc_streaming *klpr_uvc_stream_new(struct uvc_device *dev,
					    struct usb_interface *intf)
{
	struct uvc_streaming *stream;

	stream = kzalloc(sizeof(*stream), GFP_KERNEL);
	if (stream == NULL)
		return NULL;

	mutex_init(&stream->mutex);

	stream->dev = dev;
	stream->intf = (*klpe_usb_get_intf)(intf);
	stream->intfnum = intf->cur_altsetting->desc.bInterfaceNumber;

	/* Allocate a stream specific work queue for asynchronous tasks. */
	stream->async_wq = alloc_workqueue("uvcvideo", WQ_UNBOUND | WQ_HIGHPRI,
					   0);
	if (!stream->async_wq) {
		(*klpe_uvc_stream_delete)(stream);
		return NULL;
	}

	return stream;
}

static int klpr_uvc_parse_format(struct uvc_device *dev,
	struct uvc_streaming *streaming, struct uvc_format *format,
	u32 **intervals, unsigned char *buffer, int buflen)
{
	struct usb_interface *intf = streaming->intf;
	struct usb_host_interface *alts = intf->cur_altsetting;
	struct uvc_format_desc *fmtdesc;
	struct uvc_frame *frame;
	const unsigned char *start = buffer;
	unsigned int width_multiplier = 1;
	unsigned int interval;
	unsigned int i, n;
	u8 ftype;

	format->type = buffer[2];
	format->index = buffer[3];

	switch (buffer[2]) {
	case UVC_VS_FORMAT_UNCOMPRESSED:
	case UVC_VS_FORMAT_FRAME_BASED:
		n = buffer[2] == UVC_VS_FORMAT_UNCOMPRESSED ? 27 : 28;
		if (buflen < n) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d FORMAT error\n", dev->udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		/* Find the format descriptor from its GUID. */
		fmtdesc = klpr_uvc_format_by_guid(&buffer[5]);

		if (fmtdesc != NULL) {
			strscpy(format->name, fmtdesc->name,
				sizeof(format->name));
			format->fcc = fmtdesc->fcc;
		} else {
			uvc_printk(KERN_INFO, "Unknown video format %pUl\n",
				&buffer[5]);
			snprintf(format->name, sizeof(format->name), "%pUl\n",
				&buffer[5]);
			format->fcc = 0;
		}

		format->bpp = buffer[21];

		/* Some devices report a format that doesn't match what they
		 * really send.
		 */
		if (dev->quirks & UVC_QUIRK_FORCE_Y8) {
			if (format->fcc == V4L2_PIX_FMT_YUYV) {
				strscpy(format->name, "Greyscale 8-bit (Y8  )",
					sizeof(format->name));
				format->fcc = V4L2_PIX_FMT_GREY;
				format->bpp = 8;
				width_multiplier = 2;
			}
		}

		/* Some devices report bpp that doesn't match the format. */
		if (dev->quirks & UVC_QUIRK_FORCE_BPP) {
			const struct v4l2_format_info *info =
				(*klpe_v4l2_format_info)(format->fcc);

			if (info) {
				unsigned int div = info->hdiv * info->vdiv;

				n = info->bpp[0] * div;
				for (i = 1; i < info->comp_planes; i++)
					n += info->bpp[i];

				format->bpp = DIV_ROUND_UP(8 * n, div);
			}
		}

		if (buffer[2] == UVC_VS_FORMAT_UNCOMPRESSED) {
			ftype = UVC_VS_FRAME_UNCOMPRESSED;
		} else {
			ftype = UVC_VS_FRAME_FRAME_BASED;
			if (buffer[27])
				format->flags = UVC_FMT_FLAG_COMPRESSED;
		}
		break;

	case UVC_VS_FORMAT_MJPEG:
		if (buflen < 11) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d FORMAT error\n", dev->udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		strscpy(format->name, "MJPEG", sizeof(format->name));
		format->fcc = V4L2_PIX_FMT_MJPEG;
		format->flags = UVC_FMT_FLAG_COMPRESSED;
		format->bpp = 0;
		ftype = UVC_VS_FRAME_MJPEG;
		break;

	case UVC_VS_FORMAT_DV:
		if (buflen < 9) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d FORMAT error\n", dev->udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		switch (buffer[8] & 0x7f) {
		case 0:
			strscpy(format->name, "SD-DV", sizeof(format->name));
			break;
		case 1:
			strscpy(format->name, "SDL-DV", sizeof(format->name));
			break;
		case 2:
			strscpy(format->name, "HD-DV", sizeof(format->name));
			break;
		default:
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d: unknown DV format %u\n", dev->udev->devnum, alts->desc.bInterfaceNumber, buffer[8]); } while (0);
			return -EINVAL;
		}

		strlcat(format->name, buffer[8] & (1 << 7) ? " 60Hz" : " 50Hz",
			sizeof(format->name));

		format->fcc = V4L2_PIX_FMT_DV;
		format->flags = UVC_FMT_FLAG_COMPRESSED | UVC_FMT_FLAG_STREAM;
		format->bpp = 0;
		ftype = 0;

		/* Create a dummy frame descriptor. */
		frame = &format->frame[0];
		memset(&format->frame[0], 0, sizeof(format->frame[0]));
		frame->bFrameIntervalType = 1;
		frame->dwDefaultFrameInterval = 1;
		frame->dwFrameInterval = *intervals;
		*(*intervals)++ = 1;
		format->nframes = 1;
		break;

	case UVC_VS_FORMAT_MPEG2TS:
	case UVC_VS_FORMAT_STREAM_BASED:
		/* Not supported yet. */
	default:
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d unsupported format %u\n", dev->udev->devnum, alts->desc.bInterfaceNumber, buffer[2]); } while (0);
		return -EINVAL;
	}

	do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Found format %s.\n", format->name); } while (0);

	buflen -= buffer[0];
	buffer += buffer[0];

	/* Parse the frame descriptors. Only uncompressed, MJPEG and frame
	 * based formats have frame descriptors.
	 */
	while (ftype && buflen > 2 && buffer[1] == USB_DT_CS_INTERFACE &&
	       buffer[2] == ftype) {
		frame = &format->frame[format->nframes];
		if (ftype != UVC_VS_FRAME_FRAME_BASED)
			n = buflen > 25 ? buffer[25] : 0;
		else
			n = buflen > 21 ? buffer[21] : 0;

		n = n ? n : 3;

		if (buflen < 26 + 4*n) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d FRAME error\n", dev->udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		frame->bFrameIndex = buffer[3];
		frame->bmCapabilities = buffer[4];
		frame->wWidth = get_unaligned_le16(&buffer[5])
			      * width_multiplier;
		frame->wHeight = get_unaligned_le16(&buffer[7]);
		frame->dwMinBitRate = get_unaligned_le32(&buffer[9]);
		frame->dwMaxBitRate = get_unaligned_le32(&buffer[13]);
		if (ftype != UVC_VS_FRAME_FRAME_BASED) {
			frame->dwMaxVideoFrameBufferSize =
				get_unaligned_le32(&buffer[17]);
			frame->dwDefaultFrameInterval =
				get_unaligned_le32(&buffer[21]);
			frame->bFrameIntervalType = buffer[25];
		} else {
			frame->dwMaxVideoFrameBufferSize = 0;
			frame->dwDefaultFrameInterval =
				get_unaligned_le32(&buffer[17]);
			frame->bFrameIntervalType = buffer[21];
		}
		frame->dwFrameInterval = *intervals;

		/* Several UVC chipsets screw up dwMaxVideoFrameBufferSize
		 * completely. Observed behaviours range from setting the
		 * value to 1.1x the actual frame size to hardwiring the
		 * 16 low bits to 0. This results in a higher than necessary
		 * memory usage as well as a wrong image size information. For
		 * uncompressed formats this can be fixed by computing the
		 * value from the frame size.
		 */
		if (!(format->flags & UVC_FMT_FLAG_COMPRESSED))
			frame->dwMaxVideoFrameBufferSize = format->bpp
				* frame->wWidth * frame->wHeight / 8;

		/* Some bogus devices report dwMinFrameInterval equal to
		 * dwMaxFrameInterval and have dwFrameIntervalStep set to
		 * zero. Setting all null intervals to 1 fixes the problem and
		 * some other divisions by zero that could happen.
		 */
		for (i = 0; i < n; ++i) {
			interval = get_unaligned_le32(&buffer[26+4*i]);
			*(*intervals)++ = interval ? interval : 1;
		}

		/* Make sure that the default frame interval stays between
		 * the boundaries.
		 */
		n -= frame->bFrameIntervalType ? 1 : 2;
		frame->dwDefaultFrameInterval =
			min(frame->dwFrameInterval[n],
			    max(frame->dwFrameInterval[0],
				frame->dwDefaultFrameInterval));

		if (dev->quirks & UVC_QUIRK_RESTRICT_FRAME_RATE) {
			frame->bFrameIntervalType = 1;
			frame->dwFrameInterval[0] =
				frame->dwDefaultFrameInterval;
		}

		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "- %ux%u (%u.%u fps)\n", frame->wWidth, frame->wHeight, 10000000/frame->dwDefaultFrameInterval, (100000000/frame->dwDefaultFrameInterval)%10); } while (0);

		format->nframes++;
		buflen -= buffer[0];
		buffer += buffer[0];
	}

	if (buflen > 2 && buffer[1] == USB_DT_CS_INTERFACE &&
	    buffer[2] == UVC_VS_STILL_IMAGE_FRAME) {
		buflen -= buffer[0];
		buffer += buffer[0];
	}

	if (buflen > 2 && buffer[1] == USB_DT_CS_INTERFACE &&
	    buffer[2] == UVC_VS_COLORFORMAT) {
		if (buflen < 6) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d COLORFORMAT error\n", dev->udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		format->colorspace = uvc_colorspace(buffer[3]);

		buflen -= buffer[0];
		buffer += buffer[0];
	}

	return buffer - start;
}

static int klpr_uvc_parse_streaming(struct uvc_device *dev,
	struct usb_interface *intf)
{
	struct uvc_streaming *streaming = NULL;
	struct uvc_format *format;
	struct uvc_frame *frame;
	struct usb_host_interface *alts = &intf->altsetting[0];
	unsigned char *_buffer, *buffer = alts->extra;
	int _buflen, buflen = alts->extralen;
	unsigned int nformats = 0, nframes = 0, nintervals = 0;
	unsigned int size, i, n, p;
	u32 *interval;
	u16 psize;
	int ret = -EINVAL;

	if (intf->cur_altsetting->desc.bInterfaceSubClass
		!= UVC_SC_VIDEOSTREAMING) {
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d interface %d isn't a " "video streaming interface\n", dev->udev->devnum, intf->altsetting[0].desc.bInterfaceNumber); } while (0);
		return -EINVAL;
	}

	if ((*klpe_usb_driver_claim_interface)(&(*klpe_uvc_driver).driver, intf, dev)) {
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d interface %d is already " "claimed\n", dev->udev->devnum, intf->altsetting[0].desc.bInterfaceNumber); } while (0);
		return -EINVAL;
	}

	streaming = klpr_uvc_stream_new(dev, intf);
	if (streaming == NULL) {
		(*klpe_usb_driver_release_interface)(&(*klpe_uvc_driver).driver, intf);
		return -ENOMEM;
	}

	/* The Pico iMage webcam has its class-specific interface descriptors
	 * after the endpoint descriptors.
	 */
	if (buflen == 0) {
		for (i = 0; i < alts->desc.bNumEndpoints; ++i) {
			struct usb_host_endpoint *ep = &alts->endpoint[i];

			if (ep->extralen == 0)
				continue;

			if (ep->extralen > 2 &&
			    ep->extra[1] == USB_DT_CS_INTERFACE) {
				do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "trying extra data " "from endpoint %u.\n", i); } while (0);
				buffer = alts->endpoint[i].extra;
				buflen = alts->endpoint[i].extralen;
				break;
			}
		}
	}

	/* Skip the standard interface descriptors. */
	while (buflen > 2 && buffer[1] != USB_DT_CS_INTERFACE) {
		buflen -= buffer[0];
		buffer += buffer[0];
	}

	if (buflen <= 2) {
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "no class-specific streaming " "interface descriptors found.\n"); } while (0);
		goto error;
	}

	/* Parse the header descriptor. */
	switch (buffer[2]) {
	case UVC_VS_OUTPUT_HEADER:
		streaming->type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
		size = 9;
		break;

	case UVC_VS_INPUT_HEADER:
		streaming->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		size = 13;
		break;

	default:
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming interface " "%d HEADER descriptor not found.\n", dev->udev->devnum, alts->desc.bInterfaceNumber); } while (0);
		goto error;
	}

	p = buflen >= 4 ? buffer[3] : 0;
	n = buflen >= size ? buffer[size-1] : 0;

	if (buflen < size + p*n) {
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d HEADER descriptor is invalid.\n", dev->udev->devnum, alts->desc.bInterfaceNumber); } while (0);
		goto error;
	}

	streaming->header.bNumFormats = p;
	streaming->header.bEndpointAddress = buffer[6];
	if (buffer[2] == UVC_VS_INPUT_HEADER) {
		streaming->header.bmInfo = buffer[7];
		streaming->header.bTerminalLink = buffer[8];
		streaming->header.bStillCaptureMethod = buffer[9];
		streaming->header.bTriggerSupport = buffer[10];
		streaming->header.bTriggerUsage = buffer[11];
	} else {
		streaming->header.bTerminalLink = buffer[7];
	}
	streaming->header.bControlSize = n;

	streaming->header.bmaControls = kmemdup(&buffer[size], p * n,
						GFP_KERNEL);
	if (streaming->header.bmaControls == NULL) {
		ret = -ENOMEM;
		goto error;
	}

	buflen -= buffer[0];
	buffer += buffer[0];

	_buffer = buffer;
	_buflen = buflen;

	/* Count the format and frame descriptors. */
	while (_buflen > 2 && _buffer[1] == USB_DT_CS_INTERFACE) {
		switch (_buffer[2]) {
		case UVC_VS_FORMAT_UNCOMPRESSED:
		case UVC_VS_FORMAT_MJPEG:
		case UVC_VS_FORMAT_FRAME_BASED:
			nformats++;
			break;

		case UVC_VS_FORMAT_DV:
			/* DV format has no frame descriptor. We will create a
			 * dummy frame descriptor with a dummy frame interval.
			 */
			nformats++;
			nframes++;
			nintervals++;
			break;

		case UVC_VS_FORMAT_MPEG2TS:
		case UVC_VS_FORMAT_STREAM_BASED:
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming " "interface %d FORMAT %u is not supported.\n", dev->udev->devnum, alts->desc.bInterfaceNumber, _buffer[2]); } while (0);
			break;

		case UVC_VS_FRAME_UNCOMPRESSED:
		case UVC_VS_FRAME_MJPEG:
			nframes++;
			if (_buflen > 25)
				nintervals += _buffer[25] ? _buffer[25] : 3;
			break;

		case UVC_VS_FRAME_FRAME_BASED:
			nframes++;
			if (_buflen > 21)
				nintervals += _buffer[21] ? _buffer[21] : 3;
			break;
		}

		_buflen -= _buffer[0];
		_buffer += _buffer[0];
	}

	if (nformats == 0) {
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming interface " "%d has no supported formats defined.\n", dev->udev->devnum, alts->desc.bInterfaceNumber); } while (0);
		goto error;
	}

	size = nformats * sizeof(*format) + nframes * sizeof(*frame)
	     + nintervals * sizeof(*interval);
	format = kzalloc(size, GFP_KERNEL);
	if (format == NULL) {
		ret = -ENOMEM;
		goto error;
	}

	frame = (struct uvc_frame *)&format[nformats];
	interval = (u32 *)&frame[nframes];

	streaming->format = format;
	streaming->nformats = nformats;

	/* Parse the format descriptors. */
	while (buflen > 2 && buffer[1] == USB_DT_CS_INTERFACE) {
		switch (buffer[2]) {
		case UVC_VS_FORMAT_UNCOMPRESSED:
		case UVC_VS_FORMAT_MJPEG:
		case UVC_VS_FORMAT_DV:
		case UVC_VS_FORMAT_FRAME_BASED:
			format->frame = frame;
			ret = klpr_uvc_parse_format(dev, streaming, format,
				&interval, buffer, buflen);
			if (ret < 0)
				goto error;

			frame += format->nframes;
			format++;

			buflen -= ret;
			buffer += ret;
			continue;

		default:
			break;
		}

		buflen -= buffer[0];
		buffer += buffer[0];
	}

	if (buflen)
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videostreaming interface " "%d has %u bytes of trailing descriptor garbage.\n", dev->udev->devnum, alts->desc.bInterfaceNumber, buflen); } while (0);

	/* Parse the alternate settings to find the maximum bandwidth. */
	for (i = 0; i < intf->num_altsetting; ++i) {
		struct usb_host_endpoint *ep;
		alts = &intf->altsetting[i];
		ep = (*klpe_uvc_find_endpoint)(alts,
				streaming->header.bEndpointAddress);
		if (ep == NULL)
			continue;

		psize = le16_to_cpu(ep->desc.wMaxPacketSize);
		psize = (psize & 0x07ff) * (1 + ((psize >> 11) & 3));
		if (psize > streaming->maxpsize)
			streaming->maxpsize = psize;
	}

	list_add_tail(&streaming->list, &dev->streams);
	return 0;

error:
	(*klpe_usb_driver_release_interface)(&(*klpe_uvc_driver).driver, intf);
	(*klpe_uvc_stream_delete)(streaming);
	return ret;
}

static struct uvc_entity *(*klpe_uvc_alloc_entity)(u16 type, u8 id,
		unsigned int num_pads, unsigned int extra_size);

static int klpr_uvc_parse_vendor_control(struct uvc_device *dev,
	const unsigned char *buffer, int buflen)
{
	struct usb_device *udev = dev->udev;
	struct usb_host_interface *alts = dev->intf->cur_altsetting;
	struct uvc_entity *unit;
	unsigned int n, p;
	int handled = 0;

	switch (le16_to_cpu(dev->udev->descriptor.idVendor)) {
	case 0x046d:		/* Logitech */
		if (buffer[1] != 0x41 || buffer[2] != 0x01)
			break;

		/* Logitech implements several vendor specific functions
		 * through vendor specific extension units (LXU).
		 *
		 * The LXU descriptors are similar to XU descriptors
		 * (see "USB Device Video Class for Video Devices", section
		 * 3.7.2.6 "Extension Unit Descriptor") with the following
		 * differences:
		 *
		 * ----------------------------------------------------------
		 * 0		bLength		1	 Number
		 *	Size of this descriptor, in bytes: 24+p+n*2
		 * ----------------------------------------------------------
		 * 23+p+n	bmControlsType	N	Bitmap
		 *	Individual bits in the set are defined:
		 *	0: Absolute
		 *	1: Relative
		 *
		 *	This bitset is mapped exactly the same as bmControls.
		 * ----------------------------------------------------------
		 * 23+p+n*2	bReserved	1	Boolean
		 * ----------------------------------------------------------
		 * 24+p+n*2	iExtension	1	Index
		 *	Index of a string descriptor that describes this
		 *	extension unit.
		 * ----------------------------------------------------------
		 */
		p = buflen >= 22 ? buffer[21] : 0;
		n = buflen >= 25 + p ? buffer[22+p] : 0;

		if (buflen < 25 + p + 2*n) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d EXTENSION_UNIT error\n", udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			break;
		}

		unit = (*klpe_uvc_alloc_entity)(UVC_VC_EXTENSION_UNIT, buffer[3],
					p + 1, 2*n);
		if (unit == NULL)
			return -ENOMEM;

		memcpy(unit->extension.guidExtensionCode, &buffer[4], 16);
		unit->extension.bNumControls = buffer[20];
		memcpy(unit->baSourceID, &buffer[22], p);
		unit->extension.bControlSize = buffer[22+p];
		unit->extension.bmControls = (u8 *)unit + sizeof(*unit);
		unit->extension.bmControlsType = (u8 *)unit + sizeof(*unit)
					       + n;
		memcpy(unit->extension.bmControls, &buffer[23+p], 2*n);

		if (buffer[24+p+2*n] != 0)
			(*klpe_usb_string)(udev, buffer[24+p+2*n], unit->name,
				   sizeof(unit->name));
		else
			sprintf(unit->name, "Extension %u", buffer[3]);

		list_add_tail(&unit->list, &dev->entities);
		handled = 1;
		break;
	}

	return handled;
}

static int klpr_uvc_parse_standard_control(struct uvc_device *dev,
	const unsigned char *buffer, int buflen)
{
	struct usb_device *udev = dev->udev;
	struct uvc_entity *unit, *term;
	struct usb_interface *intf;
	struct usb_host_interface *alts = dev->intf->cur_altsetting;
	unsigned int i, n, p, len;
	u16 type;

	switch (buffer[2]) {
	case UVC_VC_HEADER:
		n = buflen >= 12 ? buffer[11] : 0;

		if (buflen < 12 + n) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d HEADER error\n", udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		dev->uvc_version = get_unaligned_le16(&buffer[3]);
		dev->clock_frequency = get_unaligned_le32(&buffer[7]);

		/* Parse all USB Video Streaming interfaces. */
		for (i = 0; i < n; ++i) {
			intf = (*klpe_usb_ifnum_to_if)(udev, buffer[12+i]);
			if (intf == NULL) {
				do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d " "interface %d doesn't exists\n", udev->devnum, i); } while (0);
				continue;
			}

			klpr_uvc_parse_streaming(dev, intf);
		}
		break;

	case UVC_VC_INPUT_TERMINAL:
		if (buflen < 8) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d INPUT_TERMINAL error\n", udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		/*
		 * Reject invalid terminal types that would cause issues:
		 *
		 * - The high byte must be non-zero, otherwise it would be
		 *   confused with a unit.
		 *
		 * - Bit 15 must be 0, as we use it internally as a terminal
		 *   direction flag.
		 *
		 * Other unknown types are accepted.
		 */
		type = get_unaligned_le16(&buffer[4]);
		if ((type & 0x7f00) == 0 || (type & 0x8000) != 0) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d INPUT_TERMINAL %d has invalid " "type 0x%04x, skipping\n", udev->devnum, alts->desc.bInterfaceNumber, buffer[3], type); } while (0);
			return 0;
		}

		n = 0;
		p = 0;
		len = 8;

		if (type == UVC_ITT_CAMERA) {
			n = buflen >= 15 ? buffer[14] : 0;
			len = 15;

		} else if (type == UVC_ITT_MEDIA_TRANSPORT_INPUT) {
			n = buflen >= 9 ? buffer[8] : 0;
			p = buflen >= 10 + n ? buffer[9+n] : 0;
			len = 10;
		}

		if (buflen < len + n + p) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d INPUT_TERMINAL error\n", udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		term = (*klpe_uvc_alloc_entity)(type | UVC_TERM_INPUT, buffer[3],
					1, n + p);
		if (term == NULL)
			return -ENOMEM;

		if (UVC_ENTITY_TYPE(term) == UVC_ITT_CAMERA) {
			term->camera.bControlSize = n;
			term->camera.bmControls = (u8 *)term + sizeof(*term);
			term->camera.wObjectiveFocalLengthMin =
				get_unaligned_le16(&buffer[8]);
			term->camera.wObjectiveFocalLengthMax =
				get_unaligned_le16(&buffer[10]);
			term->camera.wOcularFocalLength =
				get_unaligned_le16(&buffer[12]);
			memcpy(term->camera.bmControls, &buffer[15], n);
		} else if (UVC_ENTITY_TYPE(term) ==
			   UVC_ITT_MEDIA_TRANSPORT_INPUT) {
			term->media.bControlSize = n;
			term->media.bmControls = (u8 *)term + sizeof(*term);
			term->media.bTransportModeSize = p;
			term->media.bmTransportModes = (u8 *)term
						     + sizeof(*term) + n;
			memcpy(term->media.bmControls, &buffer[9], n);
			memcpy(term->media.bmTransportModes, &buffer[10+n], p);
		}

		if (buffer[7] != 0)
			(*klpe_usb_string)(udev, buffer[7], term->name,
				   sizeof(term->name));
		else if (UVC_ENTITY_TYPE(term) == UVC_ITT_CAMERA)
			sprintf(term->name, "Camera %u", buffer[3]);
		else if (UVC_ENTITY_TYPE(term) == UVC_ITT_MEDIA_TRANSPORT_INPUT)
			sprintf(term->name, "Media %u", buffer[3]);
		else
			sprintf(term->name, "Input %u", buffer[3]);

		list_add_tail(&term->list, &dev->entities);
		break;

	case UVC_VC_OUTPUT_TERMINAL:
		if (buflen < 9) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d OUTPUT_TERMINAL error\n", udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		/* Make sure the terminal type MSB is not null, otherwise it
		 * could be confused with a unit.
		 */
		type = get_unaligned_le16(&buffer[4]);
		if ((type & 0xff00) == 0) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d OUTPUT_TERMINAL %d has invalid " "type 0x%04x, skipping\n", udev->devnum, alts->desc.bInterfaceNumber, buffer[3], type); } while (0);
			return 0;
		}

		term = (*klpe_uvc_alloc_entity)(type | UVC_TERM_OUTPUT, buffer[3],
					1, 0);
		if (term == NULL)
			return -ENOMEM;

		memcpy(term->baSourceID, &buffer[7], 1);

		if (buffer[8] != 0)
			(*klpe_usb_string)(udev, buffer[8], term->name,
				   sizeof(term->name));
		else
			sprintf(term->name, "Output %u", buffer[3]);

		list_add_tail(&term->list, &dev->entities);
		break;

	case UVC_VC_SELECTOR_UNIT:
		p = buflen >= 5 ? buffer[4] : 0;

		if (buflen < 5 || buflen < 6 + p) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d SELECTOR_UNIT error\n", udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		unit = (*klpe_uvc_alloc_entity)(buffer[2], buffer[3], p + 1, 0);
		if (unit == NULL)
			return -ENOMEM;

		memcpy(unit->baSourceID, &buffer[5], p);

		if (buffer[5+p] != 0)
			(*klpe_usb_string)(udev, buffer[5+p], unit->name,
				   sizeof(unit->name));
		else
			sprintf(unit->name, "Selector %u", buffer[3]);

		list_add_tail(&unit->list, &dev->entities);
		break;

	case UVC_VC_PROCESSING_UNIT:
		n = buflen >= 8 ? buffer[7] : 0;
		p = dev->uvc_version >= 0x0110 ? 10 : 9;

		if (buflen < p + n) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d PROCESSING_UNIT error\n", udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		unit = (*klpe_uvc_alloc_entity)(buffer[2], buffer[3], 2, n);
		if (unit == NULL)
			return -ENOMEM;

		memcpy(unit->baSourceID, &buffer[4], 1);
		unit->processing.wMaxMultiplier =
			get_unaligned_le16(&buffer[5]);
		unit->processing.bControlSize = buffer[7];
		unit->processing.bmControls = (u8 *)unit + sizeof(*unit);
		memcpy(unit->processing.bmControls, &buffer[8], n);
		if (dev->uvc_version >= 0x0110)
			unit->processing.bmVideoStandards = buffer[9+n];

		if (buffer[8+n] != 0)
			(*klpe_usb_string)(udev, buffer[8+n], unit->name,
				   sizeof(unit->name));
		else
			sprintf(unit->name, "Processing %u", buffer[3]);

		list_add_tail(&unit->list, &dev->entities);
		break;

	case UVC_VC_EXTENSION_UNIT:
		p = buflen >= 22 ? buffer[21] : 0;
		n = buflen >= 24 + p ? buffer[22+p] : 0;

		if (buflen < 24 + p + n) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "device %d videocontrol " "interface %d EXTENSION_UNIT error\n", udev->devnum, alts->desc.bInterfaceNumber); } while (0);
			return -EINVAL;
		}

		unit = (*klpe_uvc_alloc_entity)(buffer[2], buffer[3], p + 1, n);
		if (unit == NULL)
			return -ENOMEM;

		memcpy(unit->extension.guidExtensionCode, &buffer[4], 16);
		unit->extension.bNumControls = buffer[20];
		memcpy(unit->baSourceID, &buffer[22], p);
		unit->extension.bControlSize = buffer[22+p];
		unit->extension.bmControls = (u8 *)unit + sizeof(*unit);
		memcpy(unit->extension.bmControls, &buffer[23+p], n);

		if (buffer[23+p+n] != 0)
			(*klpe_usb_string)(udev, buffer[23+p+n], unit->name,
				   sizeof(unit->name));
		else
			sprintf(unit->name, "Extension %u", buffer[3]);

		list_add_tail(&unit->list, &dev->entities);
		break;

	default:
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Found an unknown CS_INTERFACE " "descriptor (%u)\n", buffer[2]); } while (0);
		break;
	}

	return 0;
}

static int klpr_uvc_parse_control(struct uvc_device *dev)
{
	struct usb_host_interface *alts = dev->intf->cur_altsetting;
	unsigned char *buffer = alts->extra;
	int buflen = alts->extralen;
	int ret;

	/* Parse the default alternate setting only, as the UVC specification
	 * defines a single alternate setting, the default alternate setting
	 * zero.
	 */

	while (buflen > 2) {
		if (klpr_uvc_parse_vendor_control(dev, buffer, buflen) ||
		    buffer[1] != USB_DT_CS_INTERFACE)
			goto next_descriptor;

		if ((ret = klpr_uvc_parse_standard_control(dev, buffer, buflen)) < 0)
			return ret;

next_descriptor:
		buflen -= buffer[0];
		buffer += buffer[0];
	}

	/* Check if the optional status endpoint is present. Built-in iSight
	 * webcams have an interrupt endpoint but spit proprietary data that
	 * don't conform to the UVC status endpoint messages. Don't try to
	 * handle the interrupt endpoint for those cameras.
	 */
	if (alts->desc.bNumEndpoints == 1 &&
	    !(dev->quirks & UVC_QUIRK_BUILTIN_ISIGHT)) {
		struct usb_host_endpoint *ep = &alts->endpoint[0];
		struct usb_endpoint_descriptor *desc = &ep->desc;

		if (usb_endpoint_is_int_in(desc) &&
		    le16_to_cpu(desc->wMaxPacketSize) >= 8 &&
		    desc->bInterval != 0) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Found a Status endpoint " "(addr %02x).\n", desc->bEndpointAddress); } while (0);
			dev->int_ep = ep;
		}
	}

	return 0;
}

static int (*klpe_uvc_scan_chain_entity)(struct uvc_video_chain *chain,
	struct uvc_entity *entity);

static int klpr_uvc_scan_chain_forward(struct uvc_video_chain *chain,
	struct uvc_entity *entity, struct uvc_entity *prev)
{
	struct uvc_entity *forward;
	int found;

	/* Forward scan */
	forward = NULL;
	found = 0;

	while (1) {
		forward = uvc_entity_by_reference(chain->dev, entity->id,
			forward);
		if (forward == NULL)
			break;
		if (forward == prev)
			continue;
		if (forward->chain.next || forward->chain.prev) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Found reference to " "entity %d already in chain.\n", forward->id); } while (0);
			return -EINVAL;
		}

		switch (UVC_ENTITY_TYPE(forward)) {
		case UVC_VC_EXTENSION_UNIT:
			if (forward->bNrInPins != 1) {
				do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Extension unit %d " "has more than 1 input pin.\n", entity->id); } while (0);
				return -EINVAL;
			}

			list_add_tail(&forward->chain, &chain->entities);
			if ((*klpe_uvc_trace_param) & UVC_TRACE_PROBE) {
				if (!found)
					printk(KERN_CONT " (->");

				printk(KERN_CONT " XU %d", forward->id);
				found = 1;
			}
			break;

		case UVC_OTT_VENDOR_SPECIFIC:
		case UVC_OTT_DISPLAY:
		case UVC_OTT_MEDIA_TRANSPORT_OUTPUT:
		case UVC_TT_STREAMING:
			if (UVC_ENTITY_IS_ITERM(forward)) {
				do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Unsupported input " "terminal %u.\n", forward->id); } while (0);
				return -EINVAL;
			}

			list_add_tail(&forward->chain, &chain->entities);
			if ((*klpe_uvc_trace_param) & UVC_TRACE_PROBE) {
				if (!found)
					printk(KERN_CONT " (->");

				printk(KERN_CONT " OT %d", forward->id);
				found = 1;
			}
			break;
		}
	}
	if (found)
		printk(KERN_CONT ")");

	return 0;
}

static int klpr_uvc_scan_chain_backward(struct uvc_video_chain *chain,
	struct uvc_entity **_entity)
{
	struct uvc_entity *entity = *_entity;
	struct uvc_entity *term;
	int id = -EINVAL, i;

	switch (UVC_ENTITY_TYPE(entity)) {
	case UVC_VC_EXTENSION_UNIT:
	case UVC_VC_PROCESSING_UNIT:
		id = entity->baSourceID[0];
		break;

	case UVC_VC_SELECTOR_UNIT:
		/* Single-input selector units are ignored. */
		if (entity->bNrInPins == 1) {
			id = entity->baSourceID[0];
			break;
		}

		if ((*klpe_uvc_trace_param) & UVC_TRACE_PROBE)
			printk(KERN_CONT " <- IT");

		chain->selector = entity;
		for (i = 0; i < entity->bNrInPins; ++i) {
			id = entity->baSourceID[i];
			term = (*klpe_uvc_entity_by_id)(chain->dev, id);
			if (term == NULL || !UVC_ENTITY_IS_ITERM(term)) {
				do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Selector unit %d " "input %d isn't connected to an " "input terminal\n", entity->id, i); } while (0);
				return -1;
			}

			if (term->chain.next || term->chain.prev) {
				do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Found reference to " "entity %d already in chain.\n", term->id); } while (0);
				return -EINVAL;
			}

			if ((*klpe_uvc_trace_param) & UVC_TRACE_PROBE)
				printk(KERN_CONT " %d", term->id);

			list_add_tail(&term->chain, &chain->entities);
			klpr_uvc_scan_chain_forward(chain, term, entity);
		}

		if ((*klpe_uvc_trace_param) & UVC_TRACE_PROBE)
			printk(KERN_CONT "\n");

		id = 0;
		break;

	case UVC_ITT_VENDOR_SPECIFIC:
	case UVC_ITT_CAMERA:
	case UVC_ITT_MEDIA_TRANSPORT_INPUT:
	case UVC_OTT_VENDOR_SPECIFIC:
	case UVC_OTT_DISPLAY:
	case UVC_OTT_MEDIA_TRANSPORT_OUTPUT:
	case UVC_TT_STREAMING:
		id = UVC_ENTITY_IS_OTERM(entity) ? entity->baSourceID[0] : 0;
		break;
	}

	if (id <= 0) {
		*_entity = NULL;
		return id;
	}

	entity = (*klpe_uvc_entity_by_id)(chain->dev, id);
	if (entity == NULL) {
		do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Found reference to " "unknown entity %d.\n", id); } while (0);
		return -EINVAL;
	}

	*_entity = entity;
	return 0;
}

static int klpr_uvc_scan_chain(struct uvc_video_chain *chain,
			  struct uvc_entity *term)
{
	struct uvc_entity *entity, *prev;

	do { if ((*klpe_uvc_trace_param) & (1 << 0)) printk("\001" "7" "uvcvideo: " "Scanning UVC chain:"); } while (0);

	entity = term;
	prev = NULL;

	while (entity != NULL) {
		/* Entity must not be part of an existing chain */
		if (entity->chain.next || entity->chain.prev) {
			do { if ((*klpe_uvc_trace_param) & (1 << 1)) printk("\001" "7" "uvcvideo: " "Found reference to " "entity %d already in chain.\n", entity->id); } while (0);
			return -EINVAL;
		}

		/* Process entity */
		if ((*klpe_uvc_scan_chain_entity)(chain, entity) < 0)
			return -EINVAL;

		/* Forward scan */
		if (klpr_uvc_scan_chain_forward(chain, entity, prev) < 0)
			return -EINVAL;

		/* Backward scan */
		prev = entity;
		if (klpr_uvc_scan_chain_backward(chain, &entity) < 0)
			return -EINVAL;
	}

	return 0;
}

static const char *(*klpe_uvc_print_chain)(struct uvc_video_chain *chain);

static struct uvc_video_chain *(*klpe_uvc_alloc_chain)(struct uvc_device *dev);

static int klpr_uvc_scan_fallback(struct uvc_device *dev)
{
	struct uvc_video_chain *chain;
	struct uvc_entity *iterm = NULL;
	struct uvc_entity *oterm = NULL;
	struct uvc_entity *entity;
	struct uvc_entity *prev;

	/*
	 * Start by locating the input and output terminals. We only support
	 * devices with exactly one of each for now.
	 */
	list_for_each_entry(entity, &dev->entities, list) {
		if (UVC_ENTITY_IS_ITERM(entity)) {
			if (iterm)
				return -EINVAL;
			iterm = entity;
		}

		if (UVC_ENTITY_IS_OTERM(entity)) {
			if (oterm)
				return -EINVAL;
			oterm = entity;
		}
	}

	if (iterm == NULL || oterm == NULL)
		return -EINVAL;

	/* Allocate the chain and fill it. */
	chain = (*klpe_uvc_alloc_chain)(dev);
	if (chain == NULL)
		return -ENOMEM;

	if ((*klpe_uvc_scan_chain_entity)(chain, oterm) < 0)
		goto error;

	prev = oterm;

	/*
	 * Add all Processing and Extension Units with two pads. The order
	 * doesn't matter much, use reverse list traversal to connect units in
	 * UVC descriptor order as we build the chain from output to input. This
	 * leads to units appearing in the order meant by the manufacturer for
	 * the cameras known to require this heuristic.
	 */
	list_for_each_entry_reverse(entity, &dev->entities, list) {
		if (entity->type != UVC_VC_PROCESSING_UNIT &&
		    entity->type != UVC_VC_EXTENSION_UNIT)
			continue;

		if (entity->num_pads != 2)
			continue;

		if ((*klpe_uvc_scan_chain_entity)(chain, entity) < 0)
			goto error;

		prev->baSourceID[0] = entity->id;
		prev = entity;
	}

	if ((*klpe_uvc_scan_chain_entity)(chain, iterm) < 0)
		goto error;

	prev->baSourceID[0] = iterm->id;

	list_add_tail(&chain->list, &dev->chains);

	do { if ((*klpe_uvc_trace_param) & (1 << 0)) printk("\001" "7" "uvcvideo: " "Found a video chain by fallback heuristic (%s).\n", (*klpe_uvc_print_chain)(chain)); } while (0);

	return 0;

error:
	kfree(chain);
	return -EINVAL;
}

static int klpr_uvc_scan_device(struct uvc_device *dev)
{
	struct uvc_video_chain *chain;
	struct uvc_entity *term;

	list_for_each_entry(term, &dev->entities, list) {
		if (!UVC_ENTITY_IS_OTERM(term))
			continue;

		/* If the terminal is already included in a chain, skip it.
		 * This can happen for chains that have multiple output
		 * terminals, where all output terminals beside the first one
		 * will be inserted in the chain in forward scans.
		 */
		if (term->chain.next || term->chain.prev)
			continue;

		chain = (*klpe_uvc_alloc_chain)(dev);
		if (chain == NULL)
			return -ENOMEM;

		term->flags |= UVC_ENTITY_FLAG_DEFAULT;

		if (klpr_uvc_scan_chain(chain, term) < 0) {
			kfree(chain);
			continue;
		}

		do { if ((*klpe_uvc_trace_param) & (1 << 0)) printk("\001" "7" "uvcvideo: " "Found a valid video chain (%s).\n", (*klpe_uvc_print_chain)(chain)); } while (0);

		list_add_tail(&chain->list, &dev->chains);
	}

	if (list_empty(&dev->chains))
		klpr_uvc_scan_fallback(dev);

	if (list_empty(&dev->chains)) {
		uvc_printk(KERN_INFO, "No valid video chain found.\n");
		return -1;
	}

	return 0;
}

static void (*klpe_uvc_delete)(struct kref *kref);

static void (*klpe_uvc_unregister_video)(struct uvc_device *dev);


static int klpr_uvc_register_video(struct uvc_device *dev,
		struct uvc_streaming *stream)
{
	int ret;

	/* Initialize the streaming interface with default parameters. */
	ret = (*klpe_uvc_video_init)(stream);
	if (ret < 0) {
		uvc_printk(KERN_ERR, "Failed to initialize the device (%d).\n",
			   ret);
		return ret;
	}

	if (stream->type == V4L2_BUF_TYPE_VIDEO_CAPTURE)
		stream->chain->caps |= V4L2_CAP_VIDEO_CAPTURE
			| V4L2_CAP_META_CAPTURE;
	else
		stream->chain->caps |= V4L2_CAP_VIDEO_OUTPUT;

	(*klpe_uvc_debugfs_init_stream)(stream);

	/* Register the device with V4L. */
	return (*klpe_uvc_register_video_device)(dev, stream, &stream->vdev,
					 &stream->queue, stream->type,
					 &(*klpe_uvc_fops), &(*klpe_uvc_ioctl_ops));
}

static int klpr_uvc_register_terms(struct uvc_device *dev,
	struct uvc_video_chain *chain)
{
	struct uvc_streaming *stream;
	struct uvc_entity *term;
	int ret;

	list_for_each_entry(term, &chain->entities, chain) {
		if (UVC_ENTITY_TYPE(term) != UVC_TT_STREAMING)
			continue;

		stream = uvc_stream_by_id(dev, term->id);
		if (stream == NULL) {
			uvc_printk(KERN_INFO, "No streaming interface found "
				   "for terminal %u.", term->id);
			continue;
		}

		stream->chain = chain;
		ret = klpr_uvc_register_video(dev, stream);
		if (ret < 0)
			return ret;

		/* Register a metadata node, but ignore a possible failure,
		 * complete registration of video nodes anyway.
		 */
		(*klpe_uvc_meta_register)(stream);

		term->vdev = &stream->vdev;
	}

	return 0;
}

static int klpr_uvc_register_chains(struct uvc_device *dev)
{
	struct uvc_video_chain *chain;
	int ret;

	list_for_each_entry(chain, &dev->chains, list) {
		ret = klpr_uvc_register_terms(dev, chain);
		if (ret < 0)
			return ret;

#ifdef CONFIG_MEDIA_CONTROLLER
		ret = (*klpe_uvc_mc_register_entities)(chain);
		if (ret < 0)
			uvc_printk(KERN_INFO,
				   "Failed to register entities (%d).\n", ret);
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	}

	return 0;
}

static const struct uvc_device_info (*klpe_uvc_quirk_none);

int klpp_uvc_probe(struct usb_interface *intf,
		     const struct usb_device_id *id)
{
	struct usb_device *udev = interface_to_usbdev(intf);
	struct uvc_device *dev;
	const struct uvc_device_info *info =
		(const struct uvc_device_info *)id->driver_info;
	int function;
	int ret;

	if (id->idVendor && id->idProduct)
		do { if ((*klpe_uvc_trace_param) & (1 << 0)) printk("\001" "7" "uvcvideo: " "Probing known UVC device %s " "(%04x:%04x)\n", udev->devpath, id->idVendor, id->idProduct); } while (0);
	else
		do { if ((*klpe_uvc_trace_param) & (1 << 0)) printk("\001" "7" "uvcvideo: " "Probing generic UVC device %s\n", udev->devpath); } while (0);

	/* Allocate memory for the device and initialize it. */
	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (dev == NULL)
		return -ENOMEM;

	INIT_LIST_HEAD(&dev->entities);
	INIT_LIST_HEAD(&dev->chains);
	INIT_LIST_HEAD(&dev->streams);
	kref_init(&dev->ref);
	atomic_set(&dev->nmappings, 0);
	mutex_init(&dev->lock);

	dev->udev = (*klpe_usb_get_dev)(udev);
	dev->intf = (*klpe_usb_get_intf)(intf);
	dev->intfnum = intf->cur_altsetting->desc.bInterfaceNumber;
	dev->info = info ? info : &(*klpe_uvc_quirk_none);
	dev->quirks = (*klpe_uvc_quirks_param) == -1
		    ? dev->info->quirks : (*klpe_uvc_quirks_param);

	if (udev->product != NULL)
		strscpy(dev->name, udev->product, sizeof(dev->name));
	else
		snprintf(dev->name, sizeof(dev->name),
			 "UVC Camera (%04x:%04x)",
			 le16_to_cpu(udev->descriptor.idVendor),
			 le16_to_cpu(udev->descriptor.idProduct));

	/*
	 * Add iFunction or iInterface to names when available as additional
	 * distinguishers between interfaces. iFunction is prioritized over
	 * iInterface which matches Windows behavior at the point of writing.
	 */
	if (intf->intf_assoc && intf->intf_assoc->iFunction != 0)
		function = intf->intf_assoc->iFunction;
	else
		function = intf->cur_altsetting->desc.iInterface;
	if (function != 0) {
		size_t len;

		strlcat(dev->name, ": ", sizeof(dev->name));
		len = strlen(dev->name);
		(*klpe_usb_string)(udev, function, dev->name + len,
			   sizeof(dev->name) - len);
	}

#ifdef CONFIG_MEDIA_CONTROLLER
	dev->mdev.dev = &intf->dev;
	strscpy(dev->mdev.model, dev->name, sizeof(dev->mdev.model));
	if (udev->serial)
		strscpy(dev->mdev.serial, udev->serial,
			sizeof(dev->mdev.serial));
	usb_make_path(udev, dev->mdev.bus_info, sizeof(dev->mdev.bus_info));
	dev->mdev.hw_revision = le16_to_cpu(udev->descriptor.bcdDevice);
	(*klpe_media_device_init)(&dev->mdev);

	dev->vdev.mdev = &dev->mdev;
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	if (klpr_uvc_parse_control(dev) < 0) {
		do { if ((*klpe_uvc_trace_param) & (1 << 0)) printk("\001" "7" "uvcvideo: " "Unable to parse UVC " "descriptors.\n"); } while (0);
		goto error;
	}

	uvc_printk(KERN_INFO, "Found UVC %u.%02x device %s (%04x:%04x)\n",
		dev->uvc_version >> 8, dev->uvc_version & 0xff,
		udev->product ? udev->product : "<unnamed>",
		le16_to_cpu(udev->descriptor.idVendor),
		le16_to_cpu(udev->descriptor.idProduct));

	if (dev->quirks != dev->info->quirks) {
		uvc_printk(KERN_INFO, "Forcing device quirks to 0x%x by module "
			"parameter for testing purpose.\n", dev->quirks);
		uvc_printk(KERN_INFO, "Please report required quirks to the "
			"linux-uvc-devel mailing list.\n");
	}

	/* Register the V4L2 device. */
	if ((*klpe_v4l2_device_register)(&intf->dev, &dev->vdev) < 0)
		goto error;

	/* Initialize controls. */
	if ((*klpe_uvc_ctrl_init_device)(dev) < 0)
		goto error;

	/* Scan the device for video chains. */
	if (klpr_uvc_scan_device(dev) < 0)
		goto error;

	/* Register video device nodes. */
	if (klpr_uvc_register_chains(dev) < 0)
		goto error;

#ifdef CONFIG_MEDIA_CONTROLLER
	if ((*klpe___media_device_register)(&dev->mdev, (&(*klpe___this_module))) < 0)
		goto error;
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	usb_set_intfdata(intf, dev);

	/* Initialize the interrupt URB. */
	if ((ret = (*klpe_uvc_status_init)(dev)) < 0) {
		uvc_printk(KERN_INFO, "Unable to initialize the status "
			"endpoint (%d), status interrupt will not be "
			"supported.\n", ret);
	}

	do { if ((*klpe_uvc_trace_param) & (1 << 0)) printk("\001" "7" "uvcvideo: " "UVC device initialized.\n"); } while (0);
	(*klpe_usb_enable_autosuspend)(udev);
	return 0;

error:
	(*klpe_uvc_unregister_video)(dev);
	kref_put(&dev->ref, (*klpe_uvc_delete));
	return -ENODEV;
}

#include "livepatch_bsc1236783.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "uvcvideo"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "__media_device_register", (void *)&klpe___media_device_register,
	  "mc" },
	{ "media_device_init", (void *)&klpe_media_device_init, "mc" },
	{ "__this_module", (void *)&klpe___this_module, "uvcvideo" },
	{ "uvc_alloc_chain", (void *)&klpe_uvc_alloc_chain, "uvcvideo" },
	{ "uvc_alloc_entity", (void *)&klpe_uvc_alloc_entity, "uvcvideo" },
	{ "uvc_ctrl_init_device", (void *)&klpe_uvc_ctrl_init_device,
	  "uvcvideo" },
	{ "uvc_debugfs_init_stream", (void *)&klpe_uvc_debugfs_init_stream,
	  "uvcvideo" },
	{ "uvc_delete", (void *)&klpe_uvc_delete, "uvcvideo" },
	{ "uvc_driver", (void *)&klpe_uvc_driver, "uvcvideo" },
	{ "uvc_entity_by_id", (void *)&klpe_uvc_entity_by_id, "uvcvideo" },
	{ "uvc_find_endpoint", (void *)&klpe_uvc_find_endpoint, "uvcvideo" },
	{ "uvc_fmts", (void *)&klpe_uvc_fmts, "uvcvideo" },
	{ "uvc_fops", (void *)&klpe_uvc_fops, "uvcvideo" },
	{ "uvc_ioctl_ops", (void *)&klpe_uvc_ioctl_ops, "uvcvideo" },
	{ "uvc_mc_register_entities", (void *)&klpe_uvc_mc_register_entities,
	  "uvcvideo" },
	{ "uvc_meta_register", (void *)&klpe_uvc_meta_register, "uvcvideo" },
	{ "uvc_print_chain", (void *)&klpe_uvc_print_chain, "uvcvideo" },
	{ "uvc_quirk_none", (void *)&klpe_uvc_quirk_none, "uvcvideo" },
	{ "uvc_quirks_param", (void *)&klpe_uvc_quirks_param, "uvcvideo" },
	{ "uvc_register_video_device", (void *)&klpe_uvc_register_video_device,
	  "uvcvideo" },
	{ "uvc_scan_chain_entity", (void *)&klpe_uvc_scan_chain_entity,
	  "uvcvideo" },
	{ "uvc_status_init", (void *)&klpe_uvc_status_init, "uvcvideo" },
	{ "uvc_stream_delete", (void *)&klpe_uvc_stream_delete, "uvcvideo" },
	{ "uvc_trace_param", (void *)&klpe_uvc_trace_param, "uvcvideo" },
	{ "uvc_unregister_video", (void *)&klpe_uvc_unregister_video,
	  "uvcvideo" },
	{ "uvc_video_init", (void *)&klpe_uvc_video_init, "uvcvideo" },
	{ "usb_driver_claim_interface",
	  (void *)&klpe_usb_driver_claim_interface, "usbcore" },
	{ "usb_driver_release_interface",
	  (void *)&klpe_usb_driver_release_interface, "usbcore" },
	{ "usb_enable_autosuspend", (void *)&klpe_usb_enable_autosuspend,
	  "usbcore" },
	{ "usb_get_dev", (void *)&klpe_usb_get_dev, "usbcore" },
	{ "usb_get_intf", (void *)&klpe_usb_get_intf, "usbcore" },
	{ "usb_ifnum_to_if", (void *)&klpe_usb_ifnum_to_if, "usbcore" },
	{ "usb_string", (void *)&klpe_usb_string, "usbcore" },
	{ "v4l2_device_register", (void *)&klpe_v4l2_device_register,
	  "videodev" },
	{ "v4l2_format_info", (void *)&klpe_v4l2_format_info, "v4l2_common" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1236783_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LP_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void livepatch_bsc1236783_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}

#endif /* IS_ENABLED(CONFIG_USB_VIDEO_CLASS) */
