/*
 * livepatch_bsc1233294
 *
 * Fix for CVE-2024-50205, bsc#1233294
 *
 *  Upstream commit:
 *  72cafe63b35d ("ALSA: firewire-lib: Avoid division by zero in apply_constraint_to_size()")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  64d7bc388412677cbedffd9844d381bafc7bb98c
 *
 *  SLE15-SP4 and -SP5 commit:
 *  d31c5c9e40b62aeb12500e07b348cb49064ecc71
 *
 *  SLE15-SP6 commit:
 *  29fc531a21c6336cba34e96c28321a28d10aaea2
 *
 *  SLE MICRO-6-0 commit:
 *  29fc531a21c6336cba34e96c28321a28d10aaea2
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Fernando Gonzalez <fernando.gonzalez@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_SND_FIREWIRE_LIB)

#if !IS_MODULE(CONFIG_SND_FIREWIRE_LIB)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from sound/firewire/amdtp-stream.c */
#include <linux/device.h>
#include <linux/err.h>
#include <linux/firewire.h>
#include <linux/firewire-constants.h>
#include <linux/module.h>

#include <sound/pcm.h>

/* klp-ccp: from include/sound/pcm.h */
static int (*klpe_snd_interval_refine)(struct snd_interval *i, const struct snd_interval *v);

/* klp-ccp: from sound/firewire/amdtp-stream.c */
#include <sound/pcm_params.h>

/* klp-ccp: from sound/firewire/amdtp-stream.h */
#include <linux/mutex.h>
#include <linux/sched.h>
#include <sound/asound.h>

/* klp-ccp: from sound/firewire/packets-buffer.h */
#include <linux/dma-mapping.h>

/* klp-ccp: from sound/firewire/amdtp-stream.h */
enum cip_sfc {
	CIP_SFC_32000  = 0,
	CIP_SFC_44100  = 1,
	CIP_SFC_48000  = 2,
	CIP_SFC_88200  = 3,
	CIP_SFC_96000  = 4,
	CIP_SFC_176400 = 5,
	CIP_SFC_192000 = 6,
	CIP_SFC_COUNT
};

/* klp-ccp: from sound/firewire/amdtp-stream.c */
static const unsigned int (*klpe_amdtp_syt_intervals)[CIP_SFC_COUNT];
static const unsigned int (*klpe_amdtp_rate_table)[CIP_SFC_COUNT];

int klpp_apply_constraint_to_size(struct snd_pcm_hw_params *params,
				    struct snd_pcm_hw_rule *rule)
{
	struct snd_interval *s = hw_param_interval(params, rule->var);
	const struct snd_interval *r =
		hw_param_interval_c(params, SNDRV_PCM_HW_PARAM_RATE);
	struct snd_interval t = {0};
	unsigned int step = 0;
	int i;

	for (i = 0; i < CIP_SFC_COUNT; ++i) {
		if (snd_interval_test(r, (*klpe_amdtp_rate_table)[i]))
			step = max(step, (*klpe_amdtp_syt_intervals)[i]);
	}

	if (step == 0)
		return -EINVAL;

	t.min = roundup(s->min, step);
	t.max = rounddown(s->max, step);
	t.integer = 1;

	return (*klpe_snd_interval_refine)(s, &t);
}


#include "livepatch_bsc1233294.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "sound/firewire/snd_firewire_lib"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "amdtp_rate_table", (void *)&klpe_amdtp_rate_table,
	  "snd_firewire_lib" },
	{ "amdtp_syt_intervals", (void *)&klpe_amdtp_syt_intervals,
	  "snd_firewire_lib" },
	{ "snd_interval_refine", (void *)&klpe_snd_interval_refine,
	  "snd_pcm" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1233294_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LP_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void livepatch_bsc1233294_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}

#endif /* IS_ENABLED(CONFIG_SND_FIREWIRE_LIB) */
