/* Contains fix for CVE-2023-5678.  */

#ifndef _DISABLE_CVE_2023_5678

#include <string.h>
#include <stdint.h>

#define OPENSSL_FILE __FILE__
#define OPENSSL_LINE __LINE__
#define STACK_OF(type) struct stack_st_##type
typedef struct bignum_st BIGNUM;
typedef struct bignum_ctx BN_CTX;
typedef struct bn_mont_ctx_st BN_MONT_CTX;
typedef struct bn_gencb_st BN_GENCB;
struct dh_st;
typedef struct dh_st DH;
struct dh_method;
typedef struct dh_method DH_METHOD;
typedef struct engine_st ENGINE;
struct crypto_ex_data_st;
typedef struct crypto_ex_data_st CRYPTO_EX_DATA;
typedef void CRYPTO_RWLOCK;
struct crypto_ex_data_st {
    STACK_OF(void) *sk;
};
#define ERR_PUT_error(a,b,c,d,e)        ERR_put_error(a,b,c,d,e)
#define ERR_LIB_DH              5
#define DHerr(f,r)   ERR_PUT_error(ERR_LIB_DH,(f),(r),OPENSSL_FILE,OPENSSL_LINE)
void ERR_put_error(int lib, int func, int reason, const char *file, int line);
#define BN_ULONG        unsigned long
int BN_is_one(const BIGNUM *a);
BN_CTX *BN_CTX_new(void);
void BN_CTX_free(BN_CTX *c);
void BN_CTX_start(BN_CTX *ctx);
BIGNUM *BN_CTX_get(BN_CTX *ctx);
void BN_CTX_end(BN_CTX *ctx);
int BN_num_bits(const BIGNUM *a);
BIGNUM *BN_copy(BIGNUM *a, const BIGNUM *b);
int BN_sub_word(BIGNUM *a, BN_ULONG w);
int BN_set_word(BIGNUM *a, BN_ULONG w);
int BN_cmp(const BIGNUM *a, const BIGNUM *b);
int BN_mod_exp(BIGNUM *r, const BIGNUM *a, const BIGNUM *p,
               const BIGNUM *m, BN_CTX *ctx);
int BN_ucmp(const BIGNUM *a, const BIGNUM *b);

/** HACK: */
//#define DH_F_DH_CHECK_PUB_KEY                            128

#  define DH_F_DH_CHECK_PUB_KEY_EX                         123
#  define DH_F_DH_CHECK_PUB_KEY                            DH_F_DH_CHECK_PUB_KEY_EX


#define DH_R_MODULUS_TOO_LARGE                           103
#define OPENSSL_DH_CHECK_MAX_MODULUS_BITS  32768
#define DH_CHECK_INVALID_Q_VALUE        0x20
#define DH_MODULUS_TOO_LARGE            0x100
#define DH_CHECK_PUBKEY_TOO_SMALL       0x01
#define DH_CHECK_PUBKEY_TOO_LARGE       0x02
#define DH_CHECK_PUBKEY_INVALID         0x04
int DH_check_pub_key_lp(const DH *dh, const BIGNUM *pub_key, int *codes);
typedef _Atomic int CRYPTO_REF_COUNT;
struct dh_st {
    /*
     * This first argument is used to pick up errors when a DH is passed
     * instead of a EVP_PKEY
     */
    int pad;
    int version;
    BIGNUM *p;
    BIGNUM *g;
    int32_t length;             /* optional */
    BIGNUM *pub_key;            /* g^x % p */
    BIGNUM *priv_key;           /* x */
    int flags;
    BN_MONT_CTX *method_mont_p;
    /* Place holders if we want to do X9.42 DH */
    BIGNUM *q;
    BIGNUM *j;
    unsigned char *seed;
    int seedlen;
    BIGNUM *counter;
    CRYPTO_REF_COUNT references;
    CRYPTO_EX_DATA ex_data;
    const DH_METHOD *meth;
    ENGINE *engine;
    CRYPTO_RWLOCK *lock;
    int nid;
};
struct dh_method {
    char *name;
    /* Methods here */
    int (*generate_key) (DH *dh);
    int (*compute_key) (unsigned char *key, const BIGNUM *pub_key, DH *dh);

    /* Can be null */
    int (*bn_mod_exp) (const DH *dh, BIGNUM *r, const BIGNUM *a,
                       const BIGNUM *p, const BIGNUM *m, BN_CTX *ctx,
                       BN_MONT_CTX *m_ctx);
    int (*init) (DH *dh);
    int (*finish) (DH *dh);
    int flags;
    char *app_data;
    /* If this is non-NULL, it will be used to generate parameters */
    int (*generate_params) (DH *dh, int prime_len, int generator,
                            BN_GENCB *cb);
};
static int dh_check_pub_key_int(const DH *dh, const BIGNUM *q, const BIGNUM *pub_key, int *ret)
{
    int ok = 0;
    BIGNUM *tmp = NULL;
    BN_CTX *ctx = NULL;

    *ret = 0;
    ctx = BN_CTX_new();
    if (ctx == NULL)
        goto err;

    BN_CTX_start(ctx);
    tmp = BN_CTX_get(ctx);
    /* Step(1): Verify pub_key >= 2 */
    if (tmp == NULL || !BN_set_word(tmp, 1))
        goto err;
    if (BN_cmp(pub_key, tmp) <= 0) {
        *ret |= DH_CHECK_PUBKEY_TOO_SMALL;
        goto err;
    }
    /* Step(1): Verify pub_key <=  p-2 */
    if (BN_copy(tmp, dh->p) == NULL || !BN_sub_word(tmp, 1))
        goto err;
    if (BN_cmp(pub_key, tmp) >= 0) {
        *ret |= DH_CHECK_PUBKEY_TOO_LARGE;
        goto err;
    }

    if (q != NULL) {
        ctx = BN_CTX_new();
        if (ctx == NULL)
            goto err;
        BN_CTX_start(ctx);
        tmp = BN_CTX_get(ctx);

        /* Check pub_key^q == 1 mod p */
        if (tmp == NULL || !BN_mod_exp(tmp, pub_key, q, dh->p, ctx))
            goto err;
        if (!BN_is_one(tmp)) {
            *ret |= DH_CHECK_PUBKEY_INVALID;
            goto err;
        }
    }

    ok = 1;
 err:
    if (ctx != NULL) {
        BN_CTX_end(ctx);
        BN_CTX_free(ctx);
    }
    return ok;
}
int DH_check_pub_key_lp(const DH *dh, const BIGNUM *pub_key, int *ret)
{
    /* Don't do any checks at all with an excessively large modulus */
    if (BN_num_bits(dh->p) > OPENSSL_DH_CHECK_MAX_MODULUS_BITS) {
        DHerr(DH_F_DH_CHECK_PUB_KEY, DH_R_MODULUS_TOO_LARGE);
        *ret = DH_MODULUS_TOO_LARGE | DH_CHECK_PUBKEY_INVALID;
        return 0;
    }

    if (dh->q != NULL && BN_ucmp(dh->p, dh->q) < 0) {
        *ret |= DH_CHECK_INVALID_Q_VALUE | DH_CHECK_PUBKEY_INVALID;
        return 1;
    }

    return dh_check_pub_key_int(dh, dh->q, pub_key, ret);
}

#define OPENSSL_FIPS
typedef struct bignum_st BIGNUM;
typedef struct bignum_ctx BN_CTX;
typedef struct bn_mont_ctx_st BN_MONT_CTX;
typedef struct bn_gencb_st BN_GENCB;
struct dh_st;
typedef struct dh_st DH;
struct dh_method;
typedef struct dh_method DH_METHOD;
typedef struct engine_st ENGINE;
struct crypto_ex_data_st;
typedef struct crypto_ex_data_st CRYPTO_EX_DATA;
typedef void CRYPTO_RWLOCK;
int FIPS_mode(void);
#define ERR_PUT_error(a,b,c,d,e)        ERR_put_error(a,b,c,d,e)
#define ERR_LIB_DH              5
#define DHerr(f,r)   ERR_PUT_error(ERR_LIB_DH,(f),(r),OPENSSL_FILE,OPENSSL_LINE)
void ERR_put_error(int lib, int func, int reason, const char *file, int line);
int BN_num_bits(const BIGNUM *a);
#define DH_F_DH_COMPUTE_KEY                              126
#define DH_R_NON_FIPS_METHOD                             127

/** HACK: We avoid having to patch the error functions if we pass an error that
    already exists instead of creating a new one.  */
#define DH_R_CHECK_INVALID_Q_VALUE                       116
#define DH_R_Q_TOO_LARGE                                 DH_R_CHECK_INVALID_Q_VALUE //WAS: 130


#define OPENSSL_DH_MAX_MODULUS_BITS    10000
#define DH_FLAG_FIPS_METHOD                     0x0400
#define DH_FLAG_NON_FIPS_ALLOW                  0x0400
int DH_compute_key_lp(unsigned char *key, const BIGNUM *pub_key, DH *dh);
typedef _Atomic int CRYPTO_REF_COUNT;
int DH_compute_key_lp(unsigned char *key, const BIGNUM *pub_key, DH *dh)
{
    int ret = 0, i;
    volatile size_t npad = 0, mask = 1;

#ifdef OPENSSL_FIPS
    if (FIPS_mode() && !(dh->meth->flags & DH_FLAG_FIPS_METHOD)
        && !(dh->flags & DH_FLAG_NON_FIPS_ALLOW)) {
        DHerr(DH_F_DH_COMPUTE_KEY, DH_R_NON_FIPS_METHOD);
        return 0;
    }
#endif

    if (dh->q != NULL
        && BN_num_bits(dh->q) > OPENSSL_DH_MAX_MODULUS_BITS) {
        DHerr(DH_F_DH_COMPUTE_KEY, DH_R_Q_TOO_LARGE);
        return 0;
    }

    /* compute the key; ret is constant unless compute_key is external */
    if ((ret = dh->meth->compute_key(key, pub_key, dh)) <= 0)
        return ret;

    /* count leading zero bytes, yet still touch all bytes */
    for (i = 0; i < ret; i++) {
        mask &= !key[i];
        npad += mask;
    }

    /* unpad key */
    ret -= npad;
    /* key-dependent memory access, potentially leaking npad / ret */
    memmove(key, key + npad, ret);
    /* key-dependent memory access, potentially leaking npad / ret */
    memset(key + ret, 0, npad);

    return ret;
}

#endif
