"""
Test saving a mini dump.
"""

import os
import lldb
from lldbsuite.test.decorators import *
from lldbsuite.test.lldbtest import *
from lldbsuite.test import lldbutil


class ProcessSaveCoreMinidumpTestCase(TestBase):
    def verify_core_file(
        self,
        core_path,
        expected_pid,
        expected_modules,
        expected_threads,
        stacks_to_sps_map,
    ):
        # To verify, we'll launch with the mini dump
        target = self.dbg.CreateTarget(None)
        process = target.LoadCore(core_path)

        # check if the core is in desired state
        self.assertTrue(process, PROCESS_IS_VALID)
        self.assertTrue(process.GetProcessInfo().IsValid())
        self.assertEqual(process.GetProcessInfo().GetProcessID(), expected_pid)
        self.assertNotEqual(target.GetTriple().find("linux"), -1)
        self.assertTrue(target.GetNumModules(), len(expected_modules))
        self.assertEqual(process.GetNumThreads(), len(expected_threads))

        for module, expected in zip(target.modules, expected_modules):
            self.assertTrue(module.IsValid())
            module_file_name = module.GetFileSpec().GetFilename()
            expected_file_name = expected.GetFileSpec().GetFilename()
            # skip kernel virtual dynamic shared objects
            if "vdso" in expected_file_name:
                continue
            self.assertEqual(module_file_name, expected_file_name)
            self.assertEqual(module.GetUUIDString(), expected.GetUUIDString())

        red_zone = process.GetTarget().GetStackRedZoneSize()
        for thread_idx in range(process.GetNumThreads()):
            thread = process.GetThreadAtIndex(thread_idx)
            self.assertTrue(thread.IsValid())
            thread_id = thread.GetThreadID()
            self.assertIn(thread_id, expected_threads)
            frame = thread.GetFrameAtIndex(0)
            sp_region = lldb.SBMemoryRegionInfo()
            sp = frame.GetSP()
            err = process.GetMemoryRegionInfo(sp, sp_region)
            self.assertTrue(err.Success(), err.GetCString())
            error = lldb.SBError()
            # Ensure thread_id is in the saved map
            self.assertIn(thread_id, stacks_to_sps_map)
            # Ensure the SP is correct
            self.assertEqual(stacks_to_sps_map[thread_id], sp)
            # Try to read at the end of the stack red zone and succeed
            process.ReadMemory(sp - red_zone, 1, error)
            self.assertTrue(error.Success(), error.GetCString())
            # Try to read just past the red zone and fail
            process.ReadMemory(sp - red_zone - 1, 1, error)
            self.assertTrue(error.Fail(), "No failure when reading past the red zone")

        self.dbg.DeleteTarget(target)

    @skipUnlessArch("x86_64")
    @skipUnlessPlatform(["linux"])
    def test_save_linux_mini_dump(self):
        """Test that we can save a Linux mini dump."""

        self.build()
        exe = self.getBuildArtifact("a.out")
        core_stack = self.getBuildArtifact("core.stack.dmp")
        core_dirty = self.getBuildArtifact("core.dirty.dmp")
        core_full = self.getBuildArtifact("core.full.dmp")
        core_sb_stack = self.getBuildArtifact("core_sb.stack.dmp")
        core_sb_dirty = self.getBuildArtifact("core_sb.dirty.dmp")
        core_sb_full = self.getBuildArtifact("core_sb.full.dmp")
        try:
            target = self.dbg.CreateTarget(exe)
            process = target.LaunchSimple(
                None, None, self.get_process_working_directory()
            )
            self.assertState(process.GetState(), lldb.eStateStopped)

            # get neccessary data for the verification phase
            process_info = process.GetProcessInfo()
            expected_pid = process_info.GetProcessID() if process_info.IsValid() else -1
            expected_number_of_modules = target.GetNumModules()
            expected_modules = target.modules
            expected_number_of_threads = process.GetNumThreads()
            expected_threads = []
            stacks_to_sp_map = {}

            for thread_idx in range(process.GetNumThreads()):
                thread = process.GetThreadAtIndex(thread_idx)
                thread_id = thread.GetThreadID()
                expected_threads.append(thread_id)
                stacks_to_sp_map[thread_id] = thread.GetFrameAtIndex(0).GetSP()

            # save core and, kill process and verify corefile existence
            base_command = "process save-core --plugin-name=minidump "
            self.runCmd(base_command + " --style=stack '%s'" % (core_stack))
            self.assertTrue(os.path.isfile(core_stack))
            self.verify_core_file(
                core_stack,
                expected_pid,
                expected_modules,
                expected_threads,
                stacks_to_sp_map,
            )

            self.runCmd(base_command + " --style=modified-memory '%s'" % (core_dirty))
            self.assertTrue(os.path.isfile(core_dirty))
            self.verify_core_file(
                core_dirty,
                expected_pid,
                expected_modules,
                expected_threads,
                stacks_to_sp_map,
            )

            self.runCmd(base_command + " --style=full '%s'" % (core_full))
            self.assertTrue(os.path.isfile(core_full))
            self.verify_core_file(
                core_full,
                expected_pid,
                expected_modules,
                expected_threads,
                stacks_to_sp_map,
            )

            options = lldb.SBSaveCoreOptions()
            core_sb_stack_spec = lldb.SBFileSpec(core_sb_stack)
            options.SetOutputFile(core_sb_stack_spec)
            options.SetPluginName("minidump")
            options.SetStyle(lldb.eSaveCoreStackOnly)
            # validate saving via SBProcess
            error = process.SaveCore(options)
            self.assertTrue(error.Success())
            self.assertTrue(os.path.isfile(core_sb_stack))
            self.verify_core_file(
                core_sb_stack,
                expected_pid,
                expected_modules,
                expected_threads,
                stacks_to_sp_map,
            )

            options = lldb.SBSaveCoreOptions()
            core_sb_dirty_spec = lldb.SBFileSpec(core_sb_dirty)
            options.SetOutputFile(core_sb_dirty_spec)
            options.SetPluginName("minidump")
            options.SetStyle(lldb.eSaveCoreDirtyOnly)
            error = process.SaveCore(options)
            self.assertTrue(error.Success())
            self.assertTrue(os.path.isfile(core_sb_dirty))
            self.verify_core_file(
                core_sb_dirty,
                expected_pid,
                expected_modules,
                expected_threads,
                stacks_to_sp_map,
            )

            # Minidump can now save full core files, but they will be huge and
            # they might cause this test to timeout.
            options = lldb.SBSaveCoreOptions()
            core_sb_full_spec = lldb.SBFileSpec(core_sb_full)
            options.SetOutputFile(core_sb_full_spec)
            options.SetPluginName("minidump")
            options.SetStyle(lldb.eSaveCoreFull)
            error = process.SaveCore(options)
            self.assertTrue(error.Success())
            self.assertTrue(os.path.isfile(core_sb_full))
            self.verify_core_file(
                core_sb_full,
                expected_pid,
                expected_modules,
                expected_threads,
                stacks_to_sp_map,
            )

            self.assertSuccess(process.Kill())
        finally:
            # Clean up the mini dump file.
            self.assertTrue(self.dbg.DeleteTarget(target))
            if os.path.isfile(core_stack):
                os.unlink(core_stack)
            if os.path.isfile(core_dirty):
                os.unlink(core_dirty)
            if os.path.isfile(core_full):
                os.unlink(core_full)
            if os.path.isfile(core_sb_stack):
                os.unlink(core_sb_stack)
            if os.path.isfile(core_sb_dirty):
                os.unlink(core_sb_dirty)
            if os.path.isfile(core_sb_full):
                os.unlink(core_sb_full)
