/**
 * Utility types for sugiyama layout
 *
 * This module should only really matter for those looking to develop custom
 * {@link CoordOperator}s, {@link DecrossOperator}s, or in the rare case
 * advanced {@link SugiNodeSizeAccessor}s.
 *
 * @packageDocumentation
 */
import { Dag, DagLink, DagNode } from "../dag";
/**
 * The NodeDatum used for layered {@link SugiyamaOperator} layouts
 *
 * Nodes in the original graph have a layer and a reference to the original
 * node. "dummy nodes" have a link to the parent and child of the edge their on
 * in the original dag, as well as their actual layer. Given that duplicate
 * edges aren't allowed, this uniquely defines each dummy node.
 */
export declare type SugiData<NodeDatum = unknown, LinkDatum = unknown> = {
    /** layer of the sugi node */
    layer: number;
    /** original node this sugi node wraps */
    node: DagNode<NodeDatum, LinkDatum>;
} | {
    /** layer of the sugi node */
    layer: number;
    /** original link this sugi node is on */
    link: DagLink<NodeDatum, LinkDatum>;
};
/**
 * A {@link DagNode} with {@link SugiData}
 *
 * This is mostly a convenience type.
 */
export declare type SugiNode<NodeDatum = unknown, LinkDatum = unknown> = DagNode<SugiData<NodeDatum, LinkDatum>, undefined>;
/**
 * Create a dag of layered {@link SugiNode}s from a layered dag.
 *
 * Internally the {@link sugiyama} layout converts a dag into {@link SugiNode}s
 * to store metadata about the layout before copying the data back to the
 * original dag. {@link SugiNode}s are normal dag nodes who's data have a
 * special relation to the underlying dag's structure. Each new node contain
 * the wrapped nodes for real nodes, or links to the source and target nodes
 * for dummy nodes.
 *
 * This method returns the nodes in their appropriate layers. After updating
 * metadata on the layered nodes, call {@link unsugify} to copy the data to the
 * underling dag.
 *
 * The only reason to use this is if calling parts of the sugiyama chain
 * independently.
 */
export declare function sugify<N, L>(dag: Dag<N, L>): SugiNode<N, L>[][];
/**
 * Copy layout from a layered sugiyama dag back to it's underling dag.
 *
 * For normal nodes this just sets their x and y equal to the {@link
 * SugiNode}'s x and y. Chains of dummy nodes have their x and y's copied over
 * the link's points attribute.
 *
 * The only reason to call this is if laying out parts of the {@link sugiyama}
 * method independently.
 */
export declare function unsugify(layers: readonly (readonly SugiNode[])[]): void;
/**
 * Verify that x coordinates of a layering are consistent
 *
 * Note, we don't verify that node widths were respected.
 */
export declare function verifyCoordAssignment(layers: readonly (readonly SugiNode[])[], width: number): void;
/**
 * Scale the x and y of a layered dag
 *
 * Note that this only scales the x and y of nodes, not of link points, so this
 * is often called on layered {@link SugiNode}s before calling {@link unsugify}.
 */
export declare function scaleLayers(layers: readonly (readonly DagNode[])[], xscale: number, yscale: number): void;
/** compute the number of crossings in a layered sugi node */
export declare function crossings(layers: readonly (readonly SugiNode[])[]): number;
