import { getParents } from "../../dag/utils";
function createSwapChange(stationary, children) {
    const cache = new Map();
    const inds = new Map(stationary.map((n, i) => [n, i]));
    function swapChange(left, right) {
        var _a;
        const val = (_a = cache.get(left)) === null || _a === void 0 ? void 0 : _a.get(right);
        if (val !== undefined) {
            return val;
        }
        else {
            let delta = 0;
            for (const cl of children(left)) {
                for (const cr of children(right)) {
                    delta += Math.sign(inds.get(cl) - inds.get(cr));
                }
            }
            const cacheLeft = cache.get(left);
            if (cacheLeft === undefined) {
                cache.set(left, new Map([[right, delta]]));
            }
            else {
                cacheLeft.set(right, delta);
            }
            const cacheRight = cache.get(right);
            if (cacheRight === undefined) {
                cache.set(right, new Map([[left, -delta]]));
            }
            else {
                cacheRight.set(left, -delta);
            }
            return delta;
        }
    }
    return swapChange;
}
function adjacentSwap(layer, swapChange) {
    const ranges = [[0, layer.length]];
    let range;
    while ((range = ranges.pop())) {
        const [start, end] = range;
        if (start >= end)
            continue;
        let max = 0;
        let ind = end;
        for (let i = start; i < end - 1; ++i) {
            const diff = swapChange(layer[i], layer[i + 1]);
            if (diff > max) {
                max = diff;
                ind = i;
            }
        }
        if (ind !== end) {
            [layer[ind], layer[ind + 1]] = [layer[ind + 1], layer[ind]];
            ranges.push([start, ind], [ind + 2, end]);
        }
    }
}
function scanSwap(layer, swapChange) {
    const costs = new Array((layer.length * (layer.length - 1)) / 2);
    for (;;) {
        let start = 0;
        for (let ti = 1; ti < layer.length; ++ti) {
            let cum = 0;
            let ind = start;
            for (let fi = ti - 1; fi >= 0; --fi) {
                costs[ind] = cum;
                cum += swapChange(layer[fi], layer[ti]);
                ind -= layer.length - fi - 1;
            }
            start += layer.length - ti;
        }
        let ind = 0;
        let max = 0;
        let fromInd = 0;
        let toInd = 0;
        for (let fi = 0; fi < layer.length - 1; ++fi) {
            let cum = 0;
            for (let ti = fi + 1; ti < layer.length; ++ti) {
                cum += swapChange(layer[fi], layer[ti]);
                const val = costs[ind++] + cum;
                if (val > max) {
                    max = val;
                    fromInd = fi;
                    toInd = ti;
                }
            }
        }
        // no more swaps;
        if (max === 0)
            break;
        // else do swap and try again
        [layer[fromInd], layer[toInd]] = [layer[toInd], layer[fromInd]];
    }
}
function buildOperator({ baseOp, doScan, }) {
    function greedyCall(topLayer, bottomLayer, topDown) {
        baseOp(topLayer, bottomLayer, topDown);
        let layer, swapChange;
        if (topDown) {
            const parents = getParents(topLayer);
            swapChange = createSwapChange(topLayer, (node) => { var _a; return (_a = parents.get(node)) !== null && _a !== void 0 ? _a : []; });
            layer = bottomLayer;
        }
        else {
            swapChange = createSwapChange(bottomLayer, (node) => node.ichildren());
            layer = topLayer;
        }
        if (doScan) {
            scanSwap(layer, swapChange);
        }
        else {
            adjacentSwap(layer, swapChange);
        }
    }
    function base(val) {
        if (val === undefined) {
            return baseOp;
        }
        else {
            return buildOperator({ baseOp: val, doScan });
        }
    }
    greedyCall.base = base;
    function scan(val) {
        if (val === undefined) {
            return doScan;
        }
        else {
            return buildOperator({ baseOp, doScan: val });
        }
    }
    greedyCall.scan = scan;
    return greedyCall;
}
/**
 * Create a default {@link GreedyOperator}, bundled as {@link twolayerGreedy}.
 */
export function greedy(...args) {
    if (args.length) {
        throw new Error(`got arguments to greedy(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({ baseOp: () => undefined, doScan: false });
}
