import { jsxs, jsx } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { forwardRef } from 'react';
import { textUtil, locationUtil } from '@grafana/data';
import { useTheme2 } from '../../themes/ThemeContext.js';
import 'micro-memoize';
import '@emotion/react';
import 'tinycolor2';
import '../../utils/skeleton.js';
import { Icon } from '../Icon/Icon.js';
import { customWeight } from '../Text/utils.js';
import { Link } from './Link.js';

const svgSizes = {
  h1: "xl",
  h2: "xl",
  h3: "lg",
  h4: "lg",
  h5: "md",
  h6: "md",
  body: "md",
  bodySmall: "xs"
};
const TextLink = forwardRef(
  ({ href, color = "link", external = false, inline = true, variant = "body", weight, icon, children, ...rest }, ref) => {
    const validUrl = textUtil.sanitizeUrl(href != null ? href : "");
    const theme = useTheme2();
    const styles = getLinkStyles(theme, inline, variant, weight, color);
    const externalIcon = icon || "external-link-alt";
    if (external) {
      return /* @__PURE__ */ jsxs("a", { href: validUrl, ref, ...rest, target: "_blank", rel: "noreferrer", className: styles, children: [
        children,
        /* @__PURE__ */ jsx(Icon, { size: svgSizes[variant] || "md", name: externalIcon })
      ] });
    }
    const strippedUrl = locationUtil.stripBaseFromUrl(validUrl);
    return /* @__PURE__ */ jsxs(Link, { ref, href: strippedUrl, ...rest, className: styles, children: [
      children,
      icon && /* @__PURE__ */ jsx(Icon, { name: icon, size: svgSizes[variant] || "md" })
    ] });
  }
);
TextLink.displayName = "TextLink";
const getLinkStyles = (theme, inline, variant, weight, color) => {
  return css([
    variant && {
      ...theme.typography[variant]
    },
    weight && {
      fontWeight: customWeight(weight, theme)
    },
    color && {
      color: theme.colors.text[color]
    },
    {
      alignItems: "center",
      gap: "0.25em",
      display: "inline-flex",
      textDecoration: "none",
      "&:hover": {
        textDecoration: "underline",
        color: theme.colors.text.link
      }
    },
    inline && {
      textDecoration: "underline",
      "&:hover": {
        textDecoration: "none"
      }
    }
  ]);
};

export { TextLink, getLinkStyles };
//# sourceMappingURL=TextLink.js.map
