"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var initialize_1 = require("../../initialize");
var testUtils_1 = require("../../testUtils");
var utils_1 = require("../../utils");
describe('api.logs', function () {
    function createAPI(_a) {
        var _b = _a === void 0 ? { dedupe: true } : _a, dedupe = _b.dedupe, logArgsSerializer = _b.logArgsSerializer;
        var transport = new testUtils_1.MockTransport();
        var config = (0, testUtils_1.mockConfig)({
            dedupe: dedupe,
            transports: [transport],
            logArgsSerializer: logArgsSerializer,
        });
        var api = (0, initialize_1.initializeFaro)(config).api;
        return [api, transport];
    }
    describe('pushLog', function () {
        var api;
        var transport;
        beforeEach(function () {
            var _a;
            _a = createAPI(), api = _a[0], transport = _a[1];
        });
        describe('Filtering', function () {
            it('filters the same event', function () {
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(1);
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(1);
            });
            it("doesn't filter events with partially same message", function () {
                api.pushLog(['test', 'another test']);
                expect(transport.items).toHaveLength(1);
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(2);
            });
            it("doesn't filter events with same message and different level", function () {
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(1);
                api.pushLog(['test'], {
                    level: utils_1.LogLevel.INFO,
                });
                expect(transport.items).toHaveLength(2);
            });
            it("doesn't filter events with same message and same level but different context", function () {
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(1);
                api.pushLog(['test'], {
                    context: {
                        a: '1',
                    },
                });
                expect(transport.items).toHaveLength(2);
            });
            it("filters an event and doesn't filter the next different one", function () {
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(1);
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(1);
                api.pushLog(['test'], {
                    level: utils_1.LogLevel.ERROR,
                });
                expect(transport.items).toHaveLength(2);
            });
            it("doesn't filter when dedupe is false", function () {
                var _a;
                _a = createAPI({ dedupe: false }), api = _a[0], transport = _a[1];
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(1);
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(2);
            });
            it("doesn't filter when skipDedupe is true", function () {
                api.pushLog(['test']);
                expect(transport.items).toHaveLength(1);
                api.pushLog(['test'], { skipDedupe: true });
                expect(transport.items).toHaveLength(2);
            });
            it('uses traceId and spanId from custom context', function () {
                var _a;
                var spanContext = {
                    traceId: 'my-trace-id',
                    spanId: 'my-span-id',
                };
                api.pushLog(['test'], { spanContext: spanContext });
                expect(transport.items).toHaveLength(1);
                expect(((_a = transport.items[0]) === null || _a === void 0 ? void 0 : _a.payload).trace).toStrictEqual({
                    trace_id: 'my-trace-id',
                    span_id: 'my-span-id',
                });
            });
        });
        describe('Serializing', function () {
            it('serializes log arguments via String', function () {
                var _a;
                api.pushLog([1, 'test', { a: 1 }]);
                expect(((_a = transport.items[0]) === null || _a === void 0 ? void 0 : _a.payload).message).toBe('1 test [object Object]');
            });
            it('uses custom logArgsSerializer', function () {
                var _a;
                var _b;
                var logArgsSerializer = function (args) { return JSON.stringify(args); };
                _a = createAPI({ dedupe: true, logArgsSerializer: logArgsSerializer }), api = _a[0], transport = _a[1];
                api.pushLog([1, 'test', { a: 1 }]);
                expect(((_b = transport.items[0]) === null || _b === void 0 ? void 0 : _b.payload).message).toBe('[1,"test",{"a":1}]');
            });
        });
        it('Sets the timestamp to the provided custom timestamp', function () {
            var _a;
            api.pushLog(['test'], { timestampOverwriteMs: 123 });
            expect(transport.items).toHaveLength(1);
            expect(((_a = transport.items[0]) === null || _a === void 0 ? void 0 : _a.payload).timestamp).toBe('1970-01-01T00:00:00.123Z');
        });
        it('stringifies all values in the context object', function () {
            var _a;
            api.pushLog(['test'], {
                context: {
                    // @ts-expect-error
                    a: 1,
                    b: 'foo',
                    // @ts-expect-error
                    c: true,
                    // @ts-expect-error
                    d: { e: 'bar' },
                    // @ts-expect-error
                    g: null,
                    // @ts-expect-error
                    h: undefined,
                    // @ts-expect-error
                    i: [1, 2, 3],
                },
            });
            // @ts-expect-error
            expect((_a = transport.items[0]) === null || _a === void 0 ? void 0 : _a.payload.context).toStrictEqual({
                a: '1',
                b: 'foo',
                c: 'true',
                d: '{"e":"bar"}',
                g: 'null',
                h: 'undefined',
                i: '[1,2,3]',
            });
        });
    });
});
//# sourceMappingURL=initialize.test.js.map