import { TransportItemType } from '../../transports';
import { deepEqual, getCurrentTimestamp, isArray, isError, isNull, isObject, stringifyExternalJson, stringifyObjectValues, } from '../../utils';
import { timestampToIsoString } from '../../utils/date';
import { shouldIgnoreEvent } from '../utils';
import { defaultExceptionType } from './const';
let stacktraceParser;
export function initializeExceptionsAPI(_unpatchedConsole, internalLogger, config, metas, transports, tracesApi) {
    var _a;
    internalLogger.debug('Initializing exceptions API');
    let lastPayload = null;
    stacktraceParser = (_a = config.parseStacktrace) !== null && _a !== void 0 ? _a : stacktraceParser;
    const changeStacktraceParser = (newStacktraceParser) => {
        internalLogger.debug('Changing stacktrace parser');
        stacktraceParser = newStacktraceParser !== null && newStacktraceParser !== void 0 ? newStacktraceParser : stacktraceParser;
    };
    const getStacktraceParser = () => stacktraceParser;
    const { ignoreErrors = [] } = config;
    const pushError = (error, { skipDedupe, stackFrames, type, context, spanContext, timestampOverwriteMs } = {}) => {
        if (isErrorIgnored(ignoreErrors, error)) {
            return;
        }
        const item = {
            meta: metas.value,
            payload: {
                type: type || error.name || defaultExceptionType,
                value: error.message,
                timestamp: timestampOverwriteMs ? timestampToIsoString(timestampOverwriteMs) : getCurrentTimestamp(),
                trace: spanContext
                    ? {
                        trace_id: spanContext.traceId,
                        span_id: spanContext.spanId,
                    }
                    : tracesApi.getTraceContext(),
                context: stringifyObjectValues(Object.assign(Object.assign({}, parseCause(error)), (context !== null && context !== void 0 ? context : {}))),
            },
            type: TransportItemType.EXCEPTION,
        };
        stackFrames = stackFrames !== null && stackFrames !== void 0 ? stackFrames : (error.stack ? stacktraceParser === null || stacktraceParser === void 0 ? void 0 : stacktraceParser(error).frames : undefined);
        if (stackFrames === null || stackFrames === void 0 ? void 0 : stackFrames.length) {
            item.payload.stacktrace = {
                frames: stackFrames,
            };
        }
        const testingPayload = {
            type: item.payload.type,
            value: item.payload.value,
            stackTrace: item.payload.stacktrace,
            context: item.payload.context,
        };
        if (!skipDedupe && config.dedupe && !isNull(lastPayload) && deepEqual(testingPayload, lastPayload)) {
            internalLogger.debug('Skipping error push because it is the same as the last one\n', item.payload);
            return;
        }
        lastPayload = testingPayload;
        internalLogger.debug('Pushing exception\n', item);
        transports.execute(item);
    };
    changeStacktraceParser(config.parseStacktrace);
    return {
        changeStacktraceParser,
        getStacktraceParser,
        pushError,
    };
}
function parseCause(error) {
    let cause = error.cause;
    if (isError(cause)) {
        cause = error.cause.toString();
        // typeof operator on null returns "object". This is a well-known quirk in JavaScript and is considered a bug that cannot be fixed due to backward compatibility issues.
        // MDN: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/typeof#typeof_null
    }
    else if (cause !== null && (isObject(error.cause) || isArray(error.cause))) {
        cause = stringifyExternalJson(error.cause);
    }
    else if (cause != null) {
        cause = error.cause.toString();
    }
    return cause == null ? {} : { cause };
}
function isErrorIgnored(ignoreErrors, error) {
    const { message, name, stack } = error;
    return shouldIgnoreEvent(ignoreErrors, message + ' ' + name + ' ' + stack);
}
//# sourceMappingURL=initialize.js.map