/**
 * A string selector
 */
type StringSelector = string;
/**
 * A function selector with one argument
 */
type FunctionSelector = (id: string) => string;
/**
 * A function selector with two arguments
 */
type FunctionSelectorTwoArgs = (arg1: string, arg2: string) => string;
/**
 * A function selector without argument
 */
type CssSelector = () => string;
interface Selectors {
    [key: string]: StringSelector | FunctionSelector | FunctionSelectorTwoArgs | CssSelector | UrlSelector | Selectors;
}
type E2ESelectors<S extends Selectors> = {
    [P in keyof S]: S[P];
};
interface UrlSelector extends Selectors {
    url: string | FunctionSelector;
}
type VersionedFunctionSelector1 = Record<string, FunctionSelector>;
type VersionedFunctionSelector2 = Record<string, FunctionSelectorTwoArgs>;
type VersionedStringSelector = Record<string, StringSelector>;
type VersionedCssSelector = Record<string, CssSelector>;
type VersionedUrlSelector = Record<string, UrlSelector>;
type VersionedSelectors = VersionedFunctionSelector1 | VersionedFunctionSelector2 | VersionedStringSelector | VersionedCssSelector | VersionedUrlSelector;
type VersionedSelectorGroup = {
    [property: string]: VersionedSelectors | VersionedSelectorGroup;
};
type SelectorsOf<T> = {
    [Property in keyof T]: T[Property] extends VersionedFunctionSelector1 ? FunctionSelector : T[Property] extends VersionedFunctionSelector2 ? FunctionSelectorTwoArgs : T[Property] extends VersionedStringSelector ? StringSelector : T[Property] extends VersionedCssSelector ? CssSelector : T[Property] extends VersionedUrlSelector ? UrlSelector : SelectorsOf<T[Property]>;
};

/**
 * Resolves selectors based on the Grafana version
 */
declare function resolveSelectors<T extends VersionedSelectorGroup>(versionedSelectors: T, grafanaVersion?: string): SelectorsOf<T>;

/**
 * Selectors grouped/defined in Components
 */
declare const versionedComponents: {
    RadioButton: {
        container: {
            '10.2.3': string;
        };
    };
    Breadcrumbs: {
        breadcrumb: {
            '9.4.0': (title: string) => string;
        };
    };
    TimePicker: {
        openButton: {
            "8.5.0": string;
        };
        overlayContent: {
            '10.2.3': string;
        };
        fromField: {
            '10.2.3': string;
            "8.5.0": string;
        };
        toField: {
            '10.2.3': string;
            "8.5.0": string;
        };
        applyTimeRange: {
            "8.5.0": string;
        };
        copyTimeRange: {
            '10.4.0': string;
        };
        pasteTimeRange: {
            '10.4.0': string;
        };
        calendar: {
            label: {
                '10.2.3': string;
                "8.5.0": string;
            };
            openButton: {
                '10.2.3': string;
                "8.5.0": string;
            };
            closeButton: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        absoluteTimeRangeTitle: {
            "8.5.0": string;
        };
    };
    DataSourcePermissions: {
        form: {
            '9.5.0': () => string;
        };
        roleType: {
            '9.5.0': string;
        };
        rolePicker: {
            '9.5.0': string;
        };
        permissionLevel: {
            '9.5.0': string;
        };
    };
    DateTimePicker: {
        input: {
            '10.2.3': string;
        };
    };
    DataSource: {
        TestData: {
            QueryTab: {
                scenarioSelectContainer: {
                    "8.5.0": string;
                };
                scenarioSelect: {
                    "8.5.0": string;
                };
                max: {
                    "8.5.0": string;
                };
                min: {
                    "8.5.0": string;
                };
                noise: {
                    "8.5.0": string;
                };
                seriesCount: {
                    "8.5.0": string;
                };
                spread: {
                    "8.5.0": string;
                };
                startValue: {
                    "8.5.0": string;
                };
                drop: {
                    "8.5.0": string;
                };
            };
        };
        DataSourceHttpSettings: {
            urlInput: {
                '10.4.0': string;
                "8.5.0": string;
            };
        };
        Jaeger: {
            traceIDInput: {
                "8.5.0": string;
            };
        };
        Prometheus: {
            configPage: {
                connectionSettings: {
                    "8.5.0": string;
                };
                manageAlerts: {
                    '10.4.0': string;
                };
                scrapeInterval: {
                    '10.4.0': string;
                };
                queryTimeout: {
                    '10.4.0': string;
                };
                defaultEditor: {
                    '10.4.0': string;
                };
                disableMetricLookup: {
                    '10.4.0': string;
                };
                prometheusType: {
                    '10.4.0': string;
                };
                prometheusVersion: {
                    '10.4.0': string;
                };
                cacheLevel: {
                    '10.4.0': string;
                };
                incrementalQuerying: {
                    '10.4.0': string;
                };
                queryOverlapWindow: {
                    '10.4.0': string;
                };
                disableRecordingRules: {
                    '10.4.0': string;
                };
                customQueryParameters: {
                    '10.4.0': string;
                };
                httpMethod: {
                    '10.4.0': string;
                };
                exemplarsAddButton: {
                    '10.3.0': string;
                    "8.5.0": string;
                };
                internalLinkSwitch: {
                    '10.3.0': string;
                    "8.5.0": string;
                };
                codeModeMetricNamesSuggestionLimit: {
                    '11.1.0': string;
                };
            };
            queryEditor: {
                explain: {
                    '10.4.0': string;
                };
                editorToggle: {
                    '10.4.0': string;
                };
                options: {
                    '10.4.0': string;
                };
                legend: {
                    '10.4.0': string;
                };
                format: {
                    '10.4.0': string;
                };
                step: {
                    '10.4.0': string;
                };
                type: {
                    '10.4.0': string;
                };
                exemplars: {
                    '10.4.0': string;
                };
                builder: {
                    metricSelect: {
                        '10.4.0': string;
                    };
                    hints: {
                        '10.4.0': string;
                    };
                    metricsExplorer: {
                        '10.4.0': string;
                    };
                    queryAdvisor: {
                        '10.4.0': string;
                    };
                };
                code: {
                    queryField: {
                        '10.4.0': string;
                    };
                    metricsCountInfo: {
                        '11.1.0': string;
                    };
                    metricsBrowser: {
                        openButton: {
                            '10.4.0': string;
                        };
                        selectMetric: {
                            '10.4.0': string;
                        };
                        seriesLimit: {
                            '10.3.1': string;
                        };
                        metricList: {
                            '10.4.0': string;
                        };
                        labelNamesFilter: {
                            '10.4.0': string;
                        };
                        labelValuesFilter: {
                            '10.4.0': string;
                        };
                        useQuery: {
                            '10.4.0': string;
                        };
                        useAsRateQuery: {
                            '10.4.0': string;
                        };
                        validateSelector: {
                            '10.4.0': string;
                        };
                        clear: {
                            '10.4.0': string;
                        };
                    };
                };
            };
            exemplarMarker: {
                '10.3.0': string;
                "8.5.0": string;
            };
            variableQueryEditor: {
                queryType: {
                    '10.4.0': string;
                };
                labelnames: {
                    metricRegex: {
                        '10.4.0': string;
                    };
                };
                labelValues: {
                    labelSelect: {
                        '10.4.0': string;
                    };
                };
                metricNames: {
                    metricRegex: {
                        '10.4.0': string;
                    };
                };
                varQueryResult: {
                    '10.4.0': string;
                };
                seriesQuery: {
                    '10.4.0': string;
                };
                classicQuery: {
                    '10.4.0': string;
                };
            };
            annotations: {
                minStep: {
                    '10.4.0': string;
                };
                title: {
                    '10.4.0': string;
                };
                tags: {
                    '10.4.0': string;
                };
                text: {
                    '10.4.0': string;
                };
                seriesValueAsTimestamp: {
                    '10.4.0': string;
                };
            };
        };
    };
    Menu: {
        MenuComponent: {
            "8.5.0": (title: string) => string;
        };
        MenuGroup: {
            "8.5.0": (title: string) => string;
        };
        MenuItem: {
            "8.5.0": (title: string) => string;
        };
        SubMenu: {
            container: {
                '10.3.0': string;
                "8.5.0": string;
            };
            icon: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
    };
    Panels: {
        Panel: {
            title: {
                "8.5.0": (title: string) => string;
            };
            content: {
                '11.1.0': string;
            };
            headerItems: {
                '10.2.0': (item: string) => string;
            };
            menuItems: {
                '9.5.0': (item: string) => string;
            };
            menu: {
                '9.5.0': (title: string) => string;
            };
            containerByTitle: {
                "8.5.0": (title: string) => string;
            };
            headerCornerInfo: {
                "8.5.0": (mode: string) => string;
            };
            status: {
                '10.2.0': (status: string) => string;
                "8.5.0": (_: string) => string;
            };
            loadingBar: {
                '10.0.0': () => string;
            };
            HoverWidget: {
                container: {
                    '10.1.0': string;
                    "8.5.0": string;
                };
                dragIcon: {
                    '10.0.0': string;
                };
            };
            PanelDataErrorMessage: {
                '10.4.0': string;
            };
        };
        Visualization: {
            Graph: {
                container: {
                    '9.5.0': string;
                };
                VisualizationTab: {
                    legendSection: {
                        "8.5.0": string;
                    };
                };
                Legend: {
                    legendItemAlias: {
                        "8.5.0": (name: string) => string;
                    };
                    showLegendSwitch: {
                        "8.5.0": string;
                    };
                };
                xAxis: {
                    labels: {
                        "8.5.0": () => string;
                    };
                };
            };
            BarGauge: {
                valueV2: {
                    "8.5.0": string;
                };
            };
            PieChart: {
                svgSlice: {
                    '10.3.0': string;
                };
            };
            Text: {
                container: {
                    "8.5.0": () => string;
                };
            };
            Table: {
                header: {
                    "8.5.0": string;
                };
                footer: {
                    "8.5.0": string;
                };
                body: {
                    '10.2.0': string;
                };
            };
        };
    };
    VizLegend: {
        seriesName: {
            '10.3.0': (name: string) => string;
        };
    };
    Drawer: {
        General: {
            title: {
                "8.5.0": (title: string) => string;
            };
            expand: {
                "8.5.0": string;
            };
            contract: {
                "8.5.0": string;
            };
            close: {
                '10.3.0': string;
                "8.5.0": string;
            };
            rcContentWrapper: {
                '9.4.0': () => string;
            };
            subtitle: {
                '10.4.0': string;
            };
        };
        DashboardSaveDrawer: {
            saveButton: {
                '11.1.0': string;
            };
            saveAsButton: {
                '11.1.0': string;
            };
            saveAsTitleInput: {
                '11.1.0': string;
            };
        };
    };
    PanelEditor: {
        General: {
            content: {
                '11.1.0': string;
                '8.0.0': string;
            };
        };
        OptionsPane: {
            content: {
                '11.1.0': string;
                "8.5.0": string;
            };
            select: {
                "8.5.0": string;
            };
            fieldLabel: {
                "8.5.0": (type: string) => string;
            };
            fieldInput: {
                '11.0.0': (title: string) => string;
            };
        };
        DataPane: {
            content: {
                '11.1.0': string;
                "8.5.0": string;
            };
        };
        applyButton: {
            '9.2.0': string;
            '9.1.0': string;
            '8.0.0': string;
        };
        toggleVizPicker: {
            '10.0.0': string;
            '8.0.0': string;
        };
        toggleVizOptions: {
            '10.1.0': string;
            "8.5.0": string;
        };
        toggleTableView: {
            '11.1.0': string;
            "8.5.0": string;
        };
        showZoomField: {
            '10.2.0': string;
        };
        showAttributionField: {
            '10.2.0': string;
        };
        showScaleField: {
            '10.2.0': string;
        };
        showMeasureField: {
            '10.2.0': string;
        };
        showDebugField: {
            '10.2.0': string;
        };
        measureButton: {
            '9.2.0': string;
        };
    };
    PanelInspector: {
        Data: {
            content: {
                "8.5.0": string;
            };
        };
        Stats: {
            content: {
                "8.5.0": string;
            };
        };
        Json: {
            content: {
                '11.1.0': string;
                "8.5.0": string;
            };
        };
        Query: {
            content: {
                "8.5.0": string;
            };
            refreshButton: {
                "8.5.0": string;
            };
            jsonObjectKeys: {
                "8.5.0": () => string;
            };
        };
    };
    Tab: {
        title: {
            '11.2.0': (title: string) => string;
        };
        active: {
            "8.5.0": () => string;
        };
    };
    RefreshPicker: {
        runButtonV2: {
            "8.5.0": string;
        };
        intervalButtonV2: {
            "8.5.0": string;
        };
    };
    QueryTab: {
        content: {
            "8.5.0": string;
        };
        queryInspectorButton: {
            "8.5.0": string;
        };
        queryHistoryButton: {
            '10.2.0': string;
            "8.5.0": string;
        };
        addQuery: {
            '10.2.0': string;
            "8.5.0": string;
        };
        queryGroupTopSection: {
            '11.2.0': string;
        };
        addExpression: {
            '11.2.0': string;
        };
    };
    QueryHistory: {
        queryText: {
            '9.0.0': string;
        };
    };
    QueryEditorRows: {
        rows: {
            "8.5.0": string;
        };
    };
    QueryEditorRow: {
        actionButton: {
            '10.4.0': (title: string) => string;
        };
        title: {
            "8.5.0": (refId: string) => string;
        };
        container: {
            "8.5.0": (refId: string) => string;
        };
    };
    AlertTab: {
        content: {
            '10.2.3': string;
            "8.5.0": string;
        };
    };
    AlertRules: {
        groupToggle: {
            '11.0.0': string;
        };
        toggle: {
            '11.0.0': string;
        };
        expandedContent: {
            '11.0.0': string;
        };
        previewButton: {
            '11.1.0': string;
        };
        ruleNameField: {
            '11.1.0': string;
        };
        newFolderButton: {
            '11.1.0': string;
        };
        newFolderNameField: {
            '11.1.0': string;
        };
        newFolderNameCreateButton: {
            '11.1.0': string;
        };
        newEvaluationGroupButton: {
            '11.1.0': string;
        };
        newEvaluationGroupName: {
            '11.1.0': string;
        };
        newEvaluationGroupInterval: {
            '11.1.0': string;
        };
        newEvaluationGroupCreate: {
            '11.1.0': string;
        };
    };
    Alert: {
        alertV2: {
            "8.5.0": (severity: string) => string;
        };
    };
    TransformTab: {
        content: {
            '10.1.0': string;
            "8.5.0": string;
        };
        newTransform: {
            '10.1.0': (name: string) => string;
        };
        transformationEditor: {
            '10.1.0': (name: string) => string;
        };
        transformationEditorDebugger: {
            '10.1.0': (name: string) => string;
        };
    };
    Transforms: {
        card: {
            '10.1.0': (name: string) => string;
        };
        disableTransformationButton: {
            '10.4.0': string;
        };
        Reduce: {
            modeLabel: {
                '10.2.3': string;
                "8.5.0": string;
            };
            calculationsLabel: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        SpatialOperations: {
            actionLabel: {
                '9.1.2': string;
            };
            locationLabel: {
                '10.2.0': string;
            };
            location: {
                autoOption: {
                    '9.1.2': string;
                };
                coords: {
                    option: {
                        '9.1.2': string;
                    };
                    latitudeFieldLabel: {
                        '9.1.2': string;
                    };
                    longitudeFieldLabel: {
                        '9.1.2': string;
                    };
                };
                geohash: {
                    option: {
                        '9.1.2': string;
                    };
                    geohashFieldLabel: {
                        '9.1.2': string;
                    };
                };
                lookup: {
                    option: {
                        '9.1.2': string;
                    };
                    lookupFieldLabel: {
                        '9.1.2': string;
                    };
                    gazetteerFieldLabel: {
                        '9.1.2': string;
                    };
                };
            };
        };
        searchInput: {
            '10.2.3': string;
            "8.5.0": string;
        };
        noTransformationsMessage: {
            '10.2.3': string;
        };
        addTransformationButton: {
            '10.1.0': string;
            "8.5.0": string;
        };
        removeAllTransformationsButton: {
            '10.4.0': string;
        };
    };
    NavBar: {
        Configuration: {
            button: {
                '9.5.0': string;
            };
        };
        Toggle: {
            button: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        Reporting: {
            button: {
                '9.5.0': string;
            };
        };
    };
    NavMenu: {
        Menu: {
            '10.2.3': string;
        };
        item: {
            '9.5.0': string;
        };
    };
    NavToolbar: {
        container: {
            '9.4.0': string;
        };
        commandPaletteTrigger: {
            '11.5.0': string;
        };
        shareDashboard: {
            '11.1.0': string;
        };
        markAsFavorite: {
            '11.1.0': string;
        };
        editDashboard: {
            editButton: {
                '11.1.0': string;
            };
            saveButton: {
                '11.1.0': string;
            };
            exitButton: {
                '11.1.0': string;
            };
            settingsButton: {
                '11.1.0': string;
            };
            addRowButton: {
                '11.1.0': string;
            };
            addLibraryPanelButton: {
                '11.1.0': string;
            };
            addVisualizationButton: {
                '11.1.0': string;
            };
            pastePanelButton: {
                '11.1.0': string;
            };
            discardChangesButton: {
                '11.1.0': string;
            };
            discardLibraryPanelButton: {
                '11.1.0': string;
            };
            unlinkLibraryPanelButton: {
                '11.1.0': string;
            };
            saveLibraryPanelButton: {
                '11.1.0': string;
            };
            backToDashboardButton: {
                '11.1.0': string;
            };
        };
    };
    PageToolbar: {
        container: {
            "8.5.0": () => string;
        };
        item: {
            "8.5.0": (tooltip: string) => string;
        };
        itemButton: {
            '9.5.0': (title: string) => string;
        };
    };
    QueryEditorToolbarItem: {
        button: {
            "8.5.0": (title: string) => string;
        };
    };
    BackButton: {
        backArrow: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    OptionsGroup: {
        group: {
            '11.1.0': (title?: string) => string;
            "8.5.0": (title?: string) => string;
        };
        toggle: {
            '11.1.0': (title?: string) => string;
            "8.5.0": (title?: string) => string;
        };
    };
    PluginVisualization: {
        item: {
            "8.5.0": (title: string) => string;
        };
        current: {
            "8.5.0": () => string;
        };
    };
    Select: {
        menu: {
            '11.5.0': string;
            "8.5.0": string;
        };
        option: {
            '11.1.0': string;
            "8.5.0": string;
        };
        toggleAllOptions: {
            '11.3.0': string;
        };
        input: {
            "8.5.0": () => string;
        };
        singleValue: {
            "8.5.0": () => string;
        };
    };
    FieldConfigEditor: {
        content: {
            "8.5.0": string;
        };
    };
    OverridesConfigEditor: {
        content: {
            "8.5.0": string;
        };
    };
    FolderPicker: {
        containerV2: {
            "8.5.0": string;
        };
        input: {
            '10.4.0': string;
        };
    };
    ReadonlyFolderPicker: {
        container: {
            "8.5.0": string;
        };
    };
    DataSourcePicker: {
        container: {
            '10.0.0': string;
            '8.0.0': string;
        };
        inputV2: {
            '10.1.0': string;
            "8.5.0": string;
        };
        dataSourceList: {
            '10.4.0': string;
        };
        advancedModal: {
            dataSourceList: {
                '10.4.0': string;
            };
            builtInDataSourceList: {
                '10.4.0': string;
            };
        };
    };
    TimeZonePicker: {
        containerV2: {
            "8.5.0": string;
        };
        changeTimeSettingsButton: {
            '11.0.0': string;
        };
    };
    WeekStartPicker: {
        containerV2: {
            "8.5.0": string;
        };
        placeholder: {
            "8.5.0": string;
        };
    };
    TraceViewer: {
        spanBar: {
            '9.0.0': string;
        };
    };
    QueryField: {
        container: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    QueryBuilder: {
        queryPatterns: {
            '10.3.0': string;
            "8.5.0": string;
        };
        labelSelect: {
            '10.3.0': string;
            "8.5.0": string;
        };
        inputSelect: {
            '11.1.0': string;
        };
        valueSelect: {
            '10.3.0': string;
            "8.5.0": string;
        };
        matchOperatorSelect: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    ValuePicker: {
        button: {
            '10.3.0': (name: string) => string;
        };
        select: {
            '10.3.0': (name: string) => string;
        };
    };
    Search: {
        sectionV2: {
            "8.5.0": string;
        };
        itemsV2: {
            "8.5.0": string;
        };
        cards: {
            "8.5.0": string;
        };
        collapseFolder: {
            "8.5.0": (sectionId: string) => string;
        };
        expandFolder: {
            "8.5.0": (sectionId: string) => string;
        };
        dashboardItem: {
            "8.5.0": (item: string) => string;
        };
        dashboardCard: {
            "8.5.0": (item: string) => string;
        };
        folderHeader: {
            '9.3.0': (folderName: string) => string;
        };
        folderContent: {
            '9.3.0': (folderName: string) => string;
        };
        dashboardItems: {
            "8.5.0": string;
        };
    };
    DashboardLinks: {
        container: {
            "8.5.0": string;
        };
        dropDown: {
            "8.5.0": string;
        };
        link: {
            "8.5.0": string;
        };
    };
    LoadingIndicator: {
        icon: {
            '10.4.0': string;
            "8.5.0": string;
        };
    };
    CallToActionCard: {
        buttonV2: {
            "8.5.0": (name: string) => string;
        };
    };
    DataLinksContextMenu: {
        singleLink: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    CodeEditor: {
        container: {
            '10.2.3': string;
            "8.5.0": string;
        };
    };
    ReactMonacoEditor: {
        editorLazy: {
            '11.1.0': string;
        };
    };
    DashboardImportPage: {
        textarea: {
            "8.5.0": string;
        };
        submit: {
            "8.5.0": string;
        };
    };
    ImportDashboardForm: {
        name: {
            "8.5.0": string;
        };
        submit: {
            "8.5.0": string;
        };
    };
    PanelAlertTabContent: {
        content: {
            '10.2.3': string;
            "8.5.0": string;
        };
    };
    VisualizationPreview: {
        card: {
            "8.5.0": (name: string) => string;
        };
    };
    ColorSwatch: {
        name: {
            "8.5.0": string;
        };
    };
    DashboardRow: {
        title: {
            "8.5.0": (title: string) => string;
        };
    };
    UserProfile: {
        profileSaveButton: {
            "8.5.0": string;
        };
        preferencesSaveButton: {
            "8.5.0": string;
        };
        orgsTable: {
            "8.5.0": string;
        };
        sessionsTable: {
            "8.5.0": string;
        };
        extensionPointTabs: {
            '10.2.3': string;
        };
        extensionPointTab: {
            '10.2.3': (tabId: string) => string;
        };
    };
    FileUpload: {
        inputField: {
            '9.0.0': string;
        };
        fileNameSpan: {
            '9.0.0': string;
        };
    };
    DebugOverlay: {
        wrapper: {
            '9.2.0': string;
        };
    };
    OrgRolePicker: {
        input: {
            '9.5.0': string;
        };
    };
    AnalyticsToolbarButton: {
        button: {
            '9.5.0': string;
        };
    };
    Variables: {
        variableOption: {
            '9.5.0': string;
        };
        variableLinkWrapper: {
            '11.1.1': string;
        };
    };
    Annotations: {
        annotationsTypeInput: {
            '11.1.0': string;
            "8.5.0": string;
        };
        annotationsChoosePanelInput: {
            '11.1.0': string;
            "8.5.0": string;
        };
        editor: {
            testButton: {
                '11.0.0': string;
            };
            resultContainer: {
                '11.0.0': string;
            };
        };
    };
    Tooltip: {
        container: {
            '10.2.0': string;
        };
    };
    ReturnToPrevious: {
        buttonGroup: {
            '11.0.0': string;
        };
        backButton: {
            '11.0.0': string;
        };
        dismissButton: {
            '11.0.0': string;
        };
    };
    SQLQueryEditor: {
        selectColumn: {
            '11.0.0': string;
        };
        selectColumnInput: {
            '11.0.0': string;
        };
        selectFunctionParameter: {
            '11.0.0': (name: string) => string;
        };
        selectAggregation: {
            '11.0.0': string;
        };
        selectAggregationInput: {
            '11.0.0': string;
        };
        selectAlias: {
            '11.0.0': string;
        };
        selectAliasInput: {
            '11.0.0': string;
        };
        selectInputParameter: {
            '11.0.0': string;
        };
        filterConjunction: {
            '11.0.0': string;
        };
        filterField: {
            '11.0.0': string;
        };
        filterOperator: {
            '11.0.0': string;
        };
        headerTableSelector: {
            '11.0.0': string;
        };
        headerFilterSwitch: {
            '11.0.0': string;
        };
        headerGroupSwitch: {
            '11.0.0': string;
        };
        headerOrderSwitch: {
            '11.0.0': string;
        };
        headerPreviewSwitch: {
            '11.0.0': string;
        };
    };
    EntityNotFound: {
        container: {
            '11.2.0': string;
        };
    };
    Portal: {
        container: {
            '11.5.0': string;
        };
    };
};
type VersionedComponents = typeof versionedComponents;

/**
 * Selectors grouped/defined in Pages
 */
declare const versionedPages: {
    Alerting: {
        AddAlertRule: {
            url: {
                '10.1.0': string;
                "8.5.0": string;
            };
        };
        EditAlertRule: {
            url: {
                "8.5.0": (alertRuleUid: string) => string;
            };
        };
    };
    Login: {
        url: {
            "8.5.0": string;
        };
        username: {
            '10.2.3': string;
            "8.5.0": string;
        };
        password: {
            '10.2.3': string;
            "8.5.0": string;
        };
        submit: {
            '10.2.3': string;
            "8.5.0": string;
        };
        skip: {
            '10.2.3': string;
        };
    };
    PasswordlessLogin: {
        url: {
            "8.5.0": string;
        };
        email: {
            '10.2.3': string;
            "8.5.0": string;
        };
        submit: {
            '10.2.3': string;
            "8.5.0": string;
        };
    };
    Home: {
        url: {
            "8.5.0": string;
        };
    };
    DataSource: {
        name: {
            '10.3.0': string;
            "8.5.0": string;
        };
        delete: {
            "8.5.0": string;
        };
        readOnly: {
            '10.3.0': string;
            "8.5.0": string;
        };
        saveAndTest: {
            '10.0.0': string;
            "8.5.0": string;
        };
        alert: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    DataSources: {
        url: {
            "8.5.0": string;
        };
        dataSources: {
            "8.5.0": (dataSourceName: string) => string;
        };
    };
    EditDataSource: {
        url: {
            '9.5.0': (dataSourceUid: string) => string;
        };
        settings: {
            '9.5.0': string;
        };
    };
    AddDataSource: {
        url: {
            "8.5.0": string;
        };
        dataSourcePluginsV2: {
            '9.3.1': (pluginName: string) => string;
            "8.5.0": (pluginName: string) => string;
        };
    };
    ConfirmModal: {
        delete: {
            '10.0.0': string;
            "8.5.0": string;
        };
    };
    AddDashboard: {
        url: {
            "8.5.0": string;
        };
        itemButton: {
            '9.5.0': (title: string) => string;
        };
        addNewPanel: {
            '11.1.0': string;
            '8.0.0': string;
            "8.5.0": string;
        };
        itemButtonAddViz: {
            "8.5.0": string;
        };
        addNewRow: {
            '11.1.0': string;
            "8.5.0": string;
        };
        addNewPanelLibrary: {
            '11.1.0': string;
            "8.5.0": string;
        };
        Settings: {
            Annotations: {
                List: {
                    url: {
                        "8.5.0": string;
                    };
                };
                Edit: {
                    url: {
                        "8.5.0": (annotationIndex: string) => string;
                    };
                };
            };
            Variables: {
                List: {
                    url: {
                        '11.3.0': string;
                        "8.5.0": string;
                    };
                };
                Edit: {
                    url: {
                        '11.3.0': (editIndex: string) => string;
                        "8.5.0": (editIndex: string) => string;
                    };
                };
            };
        };
    };
    Dashboard: {
        url: {
            "8.5.0": (uid: string) => string;
        };
        DashNav: {
            nav: {
                "8.5.0": string;
            };
            navV2: {
                "8.5.0": string;
            };
            publicDashboardTag: {
                '9.1.0': string;
            };
            shareButton: {
                '10.4.0': string;
            };
            scrollContainer: {
                '11.1.0': string;
            };
            newShareButton: {
                container: {
                    '11.1.0': string;
                };
                shareLink: {
                    '11.1.0': string;
                };
                arrowMenu: {
                    '11.1.0': string;
                };
                menu: {
                    container: {
                        '11.1.0': string;
                    };
                    shareInternally: {
                        '11.1.0': string;
                    };
                    shareExternally: {
                        '11.1.1': string;
                    };
                    shareSnapshot: {
                        '11.2.0': string;
                    };
                    inviteUser: {
                        '11.5.0': string;
                    };
                };
            };
            NewExportButton: {
                container: {
                    '11.2.0': string;
                };
                arrowMenu: {
                    '11.2.0': string;
                };
                Menu: {
                    container: {
                        '11.2.0': string;
                    };
                    exportAsJson: {
                        '11.2.0': string;
                    };
                };
            };
            playlistControls: {
                prev: {
                    '11.0.0': string;
                };
                stop: {
                    '11.0.0': string;
                };
                next: {
                    '11.0.0': string;
                };
            };
        };
        Controls: {
            '11.1.0': string;
        };
        SubMenu: {
            submenu: {
                "8.5.0": string;
            };
            submenuItem: {
                "8.5.0": string;
            };
            submenuItemLabels: {
                "8.5.0": (item: string) => string;
            };
            submenuItemValueDropDownValueLinkTexts: {
                "8.5.0": (item: string) => string;
            };
            submenuItemValueDropDownDropDown: {
                "8.5.0": string;
            };
            submenuItemValueDropDownOptionTexts: {
                "8.5.0": (item: string) => string;
            };
            Annotations: {
                annotationsWrapper: {
                    '10.0.0': string;
                };
                annotationLabel: {
                    '10.0.0': (label: string) => string;
                };
                annotationToggle: {
                    '10.0.0': (label: string) => string;
                };
            };
        };
        Settings: {
            Actions: {
                close: {
                    '9.5.0': string;
                };
            };
            General: {
                deleteDashBoard: {
                    '11.1.0': string;
                };
                sectionItems: {
                    "8.5.0": (item: string) => string;
                };
                saveDashBoard: {
                    "8.5.0": string;
                };
                saveAsDashBoard: {
                    "8.5.0": string;
                };
                title: {
                    '11.2.0': string;
                };
            };
            Annotations: {
                Edit: {
                    urlParams: {
                        "8.5.0": (annotationIndex: string) => string;
                    };
                };
                List: {
                    url: {
                        "8.5.0": (dashboardUid: string) => string;
                    };
                    addAnnotationCTAV2: {
                        "8.5.0": string;
                    };
                    annotations: {
                        '10.4.0': string;
                    };
                };
                Settings: {
                    name: {
                        '11.1.0': string;
                        "8.5.0": string;
                    };
                };
                NewAnnotation: {
                    panelFilterSelect: {
                        '10.0.0': string;
                    };
                    showInLabel: {
                        '11.1.0': string;
                    };
                    previewInDashboard: {
                        '10.0.0': string;
                    };
                    delete: {
                        '10.4.0': string;
                    };
                    apply: {
                        '10.4.0': string;
                    };
                    enable: {
                        '10.4.0': string;
                    };
                    hide: {
                        '10.4.0': string;
                    };
                };
            };
            Variables: {
                List: {
                    url: {
                        '11.3.0': (dashboardUid: string) => string;
                        "8.5.0": (dashboardUid: string) => string;
                    };
                    addVariableCTAV2: {
                        "8.5.0": string;
                    };
                    newButton: {
                        "8.5.0": string;
                    };
                    table: {
                        "8.5.0": string;
                    };
                    tableRowNameFields: {
                        "8.5.0": (variableName: string) => string;
                    };
                    tableRowDefinitionFields: {
                        '10.1.0': (variableName: string) => string;
                    };
                    tableRowArrowUpButtons: {
                        "8.5.0": (variableName: string) => string;
                    };
                    tableRowArrowDownButtons: {
                        "8.5.0": (variableName: string) => string;
                    };
                    tableRowDuplicateButtons: {
                        "8.5.0": (variableName: string) => string;
                    };
                    tableRowRemoveButtons: {
                        "8.5.0": (variableName: string) => string;
                    };
                };
                Edit: {
                    urlParams: {
                        '11.3.0': (editIndex: string) => string;
                        "8.5.0": (editIndex: string) => string;
                    };
                    General: {
                        headerLink: {
                            "8.5.0": string;
                        };
                        modeLabelNew: {
                            "8.5.0": string;
                        };
                        /**
                         * @deprecated
                         */
                        modeLabelEdit: {
                            "8.5.0": string;
                        };
                        generalNameInput: {
                            "8.5.0": string;
                        };
                        generalNameInputV2: {
                            "8.5.0": string;
                        };
                        generalTypeSelect: {
                            "8.5.0": string;
                        };
                        generalTypeSelectV2: {
                            "8.5.0": string;
                        };
                        generalLabelInput: {
                            "8.5.0": string;
                        };
                        generalLabelInputV2: {
                            "8.5.0": string;
                        };
                        generalHideSelect: {
                            "8.5.0": string;
                        };
                        generalHideSelectV2: {
                            "8.5.0": string;
                        };
                        selectionOptionsAllowCustomValueSwitch: {
                            "8.5.0": string;
                        };
                        selectionOptionsMultiSwitch: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        selectionOptionsIncludeAllSwitch: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        selectionOptionsCustomAllInput: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        previewOfValuesOption: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        submitButton: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        applyButton: {
                            '9.3.0': string;
                        };
                    };
                    QueryVariable: {
                        queryOptionsDataSourceSelect: {
                            '10.4.0': string;
                            '10.0.0': string;
                            "8.5.0": string;
                        };
                        queryOptionsRefreshSelect: {
                            "8.5.0": string;
                        };
                        queryOptionsRefreshSelectV2: {
                            "8.5.0": string;
                        };
                        queryOptionsRegExInput: {
                            "8.5.0": string;
                        };
                        queryOptionsRegExInputV2: {
                            "8.5.0": string;
                        };
                        queryOptionsSortSelect: {
                            "8.5.0": string;
                        };
                        queryOptionsSortSelectV2: {
                            "8.5.0": string;
                        };
                        queryOptionsQueryInput: {
                            '10.4.0': string;
                        };
                        valueGroupsTagsEnabledSwitch: {
                            "8.5.0": string;
                        };
                        valueGroupsTagsTagsQueryInput: {
                            "8.5.0": string;
                        };
                        valueGroupsTagsTagsValuesQueryInput: {
                            "8.5.0": string;
                        };
                    };
                    ConstantVariable: {
                        constantOptionsQueryInput: {
                            "8.5.0": string;
                        };
                        constantOptionsQueryInputV2: {
                            "8.5.0": string;
                        };
                    };
                    DatasourceVariable: {
                        datasourceSelect: {
                            "8.5.0": string;
                        };
                    };
                    TextBoxVariable: {
                        textBoxOptionsQueryInput: {
                            "8.5.0": string;
                        };
                        textBoxOptionsQueryInputV2: {
                            "8.5.0": string;
                        };
                    };
                    CustomVariable: {
                        customValueInput: {
                            "8.5.0": string;
                        };
                    };
                    IntervalVariable: {
                        intervalsValueInput: {
                            "8.5.0": string;
                        };
                        autoEnabledCheckbox: {
                            '10.4.0': string;
                        };
                        stepCountIntervalSelect: {
                            '10.4.0': string;
                        };
                        minIntervalInput: {
                            '10.4.0': string;
                        };
                    };
                    GroupByVariable: {
                        dataSourceSelect: {
                            '10.4.0': string;
                        };
                        infoText: {
                            '10.4.0': string;
                        };
                        modeToggle: {
                            '10.4.0': string;
                        };
                    };
                    AdHocFiltersVariable: {
                        datasourceSelect: {
                            '10.4.0': string;
                        };
                        infoText: {
                            '10.4.0': string;
                        };
                        modeToggle: {
                            '11.0.0': string;
                        };
                    };
                };
            };
        };
        Annotations: {
            marker: {
                '10.0.0': string;
            };
        };
        Rows: {
            Repeated: {
                ConfigSection: {
                    warningMessage: {
                        '10.2.0': string;
                    };
                };
            };
        };
    };
    Dashboards: {
        url: {
            "8.5.0": string;
        };
        dashboards: {
            '10.2.0': (title: string) => string;
        };
    };
    SaveDashboardAsModal: {
        newName: {
            '10.2.0': string;
        };
        save: {
            '10.2.0': string;
        };
    };
    SaveDashboardModal: {
        save: {
            '10.2.0': string;
        };
        saveVariables: {
            '10.2.0': string;
        };
        saveTimerange: {
            '10.2.0': string;
        };
        saveRefresh: {
            '11.1.0': string;
        };
    };
    SharePanelModal: {
        linkToRenderedImage: {
            "8.5.0": string;
        };
    };
    ShareDashboardModal: {
        PublicDashboard: {
            WillBePublicCheckbox: {
                '9.1.0': string;
            };
            LimitedDSCheckbox: {
                '9.1.0': string;
            };
            CostIncreaseCheckbox: {
                '9.1.0': string;
            };
            PauseSwitch: {
                '9.5.0': string;
            };
            EnableAnnotationsSwitch: {
                '9.3.0': string;
            };
            CreateButton: {
                '9.5.0': string;
            };
            DeleteButton: {
                '9.3.0': string;
            };
            CopyUrlInput: {
                '9.1.0': string;
            };
            CopyUrlButton: {
                '9.1.0': string;
            };
            SettingsDropdown: {
                '10.1.0': string;
            };
            TemplateVariablesWarningAlert: {
                '9.1.0': string;
            };
            UnsupportedDataSourcesWarningAlert: {
                '9.5.0': string;
            };
            NoUpsertPermissionsWarningAlert: {
                '9.5.0': string;
            };
            EnableTimeRangeSwitch: {
                '9.4.0': string;
            };
            EmailSharingConfiguration: {
                Container: {
                    '9.5.0': string;
                };
                ShareType: {
                    '9.5.0': string;
                };
                EmailSharingInput: {
                    '9.5.0': string;
                };
                EmailSharingInviteButton: {
                    '9.5.0': string;
                };
                EmailSharingList: {
                    '9.5.0': string;
                };
                DeleteEmail: {
                    '9.5.0': string;
                };
                ReshareLink: {
                    '9.5.0': string;
                };
            };
        };
        SnapshotScene: {
            url: {
                '11.1.0': (key: string) => string;
            };
            PublishSnapshot: {
                '11.1.0': string;
            };
            CopyUrlButton: {
                '11.1.0': string;
            };
            CopyUrlInput: {
                '11.1.0': string;
            };
        };
    };
    ShareDashboardDrawer: {
        ShareInternally: {
            container: {
                '11.3.0': string;
            };
            lockTimeRangeSwitch: {
                '11.3.0': string;
            };
            shortenUrlSwitch: {
                '11.3.0': string;
            };
            copyUrlButton: {
                '11.3.0': string;
            };
            SharePanel: {
                preview: {
                    '11.5.0': string;
                };
                widthInput: {
                    '11.5.0': string;
                };
                heightInput: {
                    '11.5.0': string;
                };
                scaleFactorInput: {
                    '11.5.0': string;
                };
                generateImageButton: {
                    '11.5.0': string;
                };
                downloadImageButton: {
                    '11.5.0': string;
                };
            };
        };
        ShareExternally: {
            container: {
                '11.3.0': string;
            };
            publicAlert: {
                '11.3.0': string;
            };
            emailSharingAlert: {
                '11.3.0': string;
            };
            shareTypeSelect: {
                '11.3.0': string;
            };
            Creation: {
                PublicShare: {
                    createButton: {
                        '11.3.0': string;
                    };
                    cancelButton: {
                        '11.3.0': string;
                    };
                };
                EmailShare: {
                    createButton: {
                        '11.3.0': string;
                    };
                    cancelButton: {
                        '11.3.0': string;
                    };
                };
                willBePublicCheckbox: {
                    '11.3.0': string;
                };
            };
            Configuration: {
                enableTimeRangeSwitch: {
                    '11.3.0': string;
                };
                enableAnnotationsSwitch: {
                    '11.3.0': string;
                };
                copyUrlButton: {
                    '11.3.0': string;
                };
                revokeAccessButton: {
                    '11.3.0': string;
                };
                toggleAccessButton: {
                    '11.3.0': string;
                };
            };
        };
        ShareSnapshot: {
            url: {
                '11.3.0': (key: string) => string;
            };
            container: {
                '11.3.0': string;
            };
            publishSnapshot: {
                '11.3.0': string;
            };
            copyUrlButton: {
                '11.3.0': string;
            };
        };
    };
    ExportDashboardDrawer: {
        ExportAsJson: {
            container: {
                '11.3.0': string;
            };
            codeEditor: {
                '11.3.0': string;
            };
            exportExternallyToggle: {
                '11.3.0': string;
            };
            saveToFileButton: {
                '11.3.0': string;
            };
            copyToClipboardButton: {
                '11.3.0': string;
            };
            cancelButton: {
                '11.3.0': string;
            };
        };
    };
    PublicDashboard: {
        page: {
            '9.5.0': string;
        };
        NotAvailable: {
            container: {
                '9.5.0': string;
            };
            title: {
                '9.5.0': string;
            };
            pausedDescription: {
                '9.5.0': string;
            };
        };
        footer: {
            '11.0.0': string;
        };
    };
    PublicDashboardScene: {
        loadingPage: {
            '11.0.0': string;
        };
        page: {
            '11.0.0': string;
        };
        controls: {
            '11.0.0': string;
        };
    };
    RequestViewAccess: {
        form: {
            '9.5.0': string;
        };
        recipientInput: {
            '9.5.0': string;
        };
        submitButton: {
            '9.5.0': string;
        };
    };
    PublicDashboardConfirmAccess: {
        submitButton: {
            '10.2.0': string;
        };
    };
    Explore: {
        url: {
            "8.5.0": string;
        };
        General: {
            container: {
                "8.5.0": string;
            };
            graph: {
                "8.5.0": string;
            };
            table: {
                "8.5.0": string;
            };
            scrollView: {
                '9.0.0': string;
            };
        };
        QueryHistory: {
            container: {
                '11.1.0': string;
            };
        };
    };
    SoloPanel: {
        url: {
            "8.5.0": (page: string) => string;
        };
    };
    PluginsList: {
        page: {
            "8.5.0": string;
        };
        list: {
            "8.5.0": string;
        };
        listItem: {
            "8.5.0": string;
        };
        signatureErrorNotice: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    PluginPage: {
        page: {
            "8.5.0": string;
        };
        signatureInfo: {
            '10.3.0': string;
            "8.5.0": string;
        };
        disabledInfo: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    PlaylistForm: {
        name: {
            "8.5.0": string;
        };
        interval: {
            "8.5.0": string;
        };
        itemDelete: {
            '10.2.0': string;
        };
    };
    BrowseDashboards: {
        table: {
            body: {
                '10.2.0': string;
            };
            row: {
                '10.2.0': (name: string) => string;
            };
            checkbox: {
                '10.0.0': (uid: string) => string;
            };
        };
        NewFolderForm: {
            form: {
                '10.2.0': string;
            };
            nameInput: {
                '10.2.0': string;
            };
            createButton: {
                '10.2.0': string;
            };
        };
    };
    Search: {
        url: {
            '9.3.0': string;
        };
        FolderView: {
            url: {
                '9.3.0': string;
            };
        };
    };
    PublicDashboards: {
        ListItem: {
            linkButton: {
                '9.3.0': string;
            };
            configButton: {
                '9.3.0': string;
            };
            trashcanButton: {
                '9.3.0': string;
            };
            pauseSwitch: {
                '10.1.0': string;
            };
        };
    };
    UserListPage: {
        tabs: {
            allUsers: {
                '10.0.0': string;
            };
            orgUsers: {
                '10.0.0': string;
            };
            anonUserDevices: {
                '10.2.3': string;
            };
            publicDashboardsUsers: {
                '10.0.0': string;
            };
            users: {
                '10.0.0': string;
            };
        };
        org: {
            url: {
                '10.2.0': string;
                '9.5.0': string;
            };
        };
        admin: {
            url: {
                '9.5.0': string;
            };
        };
        publicDashboards: {
            container: {
                '11.1.0': string;
            };
        };
        UserListAdminPage: {
            container: {
                '10.0.0': string;
            };
        };
        UsersListPage: {
            container: {
                '10.0.0': string;
            };
        };
        UserAnonListPage: {
            container: {
                '10.4.0': string;
            };
        };
        UsersListPublicDashboardsPage: {
            container: {
                '10.0.0': string;
            };
            DashboardsListModal: {
                listItem: {
                    '10.0.0': (uid: string) => string;
                };
            };
        };
    };
    ProfilePage: {
        url: {
            '10.2.0': string;
        };
    };
    Plugin: {
        url: {
            "8.5.0": (pluginId: string) => string;
        };
    };
};
type VersionedPages = typeof versionedPages;

declare const Pages: SelectorsOf<{
    Alerting: {
        AddAlertRule: {
            url: {
                '10.1.0': string;
                "8.5.0": string;
            };
        };
        EditAlertRule: {
            url: {
                "8.5.0": (alertRuleUid: string) => string;
            };
        };
    };
    Login: {
        url: {
            "8.5.0": string;
        };
        username: {
            '10.2.3': string;
            "8.5.0": string;
        };
        password: {
            '10.2.3': string;
            "8.5.0": string;
        };
        submit: {
            '10.2.3': string;
            "8.5.0": string;
        };
        skip: {
            '10.2.3': string;
        };
    };
    PasswordlessLogin: {
        url: {
            "8.5.0": string;
        };
        email: {
            '10.2.3': string;
            "8.5.0": string;
        };
        submit: {
            '10.2.3': string;
            "8.5.0": string;
        };
    };
    Home: {
        url: {
            "8.5.0": string;
        };
    };
    DataSource: {
        name: {
            '10.3.0': string;
            "8.5.0": string;
        };
        delete: {
            "8.5.0": string;
        };
        readOnly: {
            '10.3.0': string;
            "8.5.0": string;
        };
        saveAndTest: {
            '10.0.0': string;
            "8.5.0": string;
        };
        alert: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    DataSources: {
        url: {
            "8.5.0": string;
        };
        dataSources: {
            "8.5.0": (dataSourceName: string) => string;
        };
    };
    EditDataSource: {
        url: {
            '9.5.0': (dataSourceUid: string) => string;
        };
        settings: {
            '9.5.0': string;
        };
    };
    AddDataSource: {
        url: {
            "8.5.0": string;
        };
        dataSourcePluginsV2: {
            '9.3.1': (pluginName: string) => string;
            "8.5.0": (pluginName: string) => string;
        };
    };
    ConfirmModal: {
        delete: {
            '10.0.0': string;
            "8.5.0": string;
        };
    };
    AddDashboard: {
        url: {
            "8.5.0": string;
        };
        itemButton: {
            '9.5.0': (title: string) => string;
        };
        addNewPanel: {
            '11.1.0': string;
            '8.0.0': string;
            "8.5.0": string;
        };
        itemButtonAddViz: {
            "8.5.0": string;
        };
        addNewRow: {
            '11.1.0': string;
            "8.5.0": string;
        };
        addNewPanelLibrary: {
            '11.1.0': string;
            "8.5.0": string;
        };
        Settings: {
            Annotations: {
                List: {
                    url: {
                        "8.5.0": string;
                    };
                };
                Edit: {
                    url: {
                        "8.5.0": (annotationIndex: string) => string;
                    };
                };
            };
            Variables: {
                List: {
                    url: {
                        '11.3.0': string;
                        "8.5.0": string;
                    };
                };
                Edit: {
                    url: {
                        '11.3.0': (editIndex: string) => string;
                        "8.5.0": (editIndex: string) => string;
                    };
                };
            };
        };
    };
    Dashboard: {
        url: {
            "8.5.0": (uid: string) => string;
        };
        DashNav: {
            nav: {
                "8.5.0": string;
            };
            navV2: {
                "8.5.0": string;
            };
            publicDashboardTag: {
                '9.1.0': string;
            };
            shareButton: {
                '10.4.0': string;
            };
            scrollContainer: {
                '11.1.0': string;
            };
            newShareButton: {
                container: {
                    '11.1.0': string;
                };
                shareLink: {
                    '11.1.0': string;
                };
                arrowMenu: {
                    '11.1.0': string;
                };
                menu: {
                    container: {
                        '11.1.0': string;
                    };
                    shareInternally: {
                        '11.1.0': string;
                    };
                    shareExternally: {
                        '11.1.1': string;
                    };
                    shareSnapshot: {
                        '11.2.0': string;
                    };
                    inviteUser: {
                        '11.5.0': string;
                    };
                };
            };
            NewExportButton: {
                container: {
                    '11.2.0': string;
                };
                arrowMenu: {
                    '11.2.0': string;
                };
                Menu: {
                    container: {
                        '11.2.0': string;
                    };
                    exportAsJson: {
                        '11.2.0': string;
                    };
                };
            };
            playlistControls: {
                prev: {
                    '11.0.0': string;
                };
                stop: {
                    '11.0.0': string;
                };
                next: {
                    '11.0.0': string;
                };
            };
        };
        Controls: {
            '11.1.0': string;
        };
        SubMenu: {
            submenu: {
                "8.5.0": string;
            };
            submenuItem: {
                "8.5.0": string;
            };
            submenuItemLabels: {
                "8.5.0": (item: string) => string;
            };
            submenuItemValueDropDownValueLinkTexts: {
                "8.5.0": (item: string) => string;
            };
            submenuItemValueDropDownDropDown: {
                "8.5.0": string;
            };
            submenuItemValueDropDownOptionTexts: {
                "8.5.0": (item: string) => string;
            };
            Annotations: {
                annotationsWrapper: {
                    '10.0.0': string;
                };
                annotationLabel: {
                    '10.0.0': (label: string) => string;
                };
                annotationToggle: {
                    '10.0.0': (label: string) => string;
                };
            };
        };
        Settings: {
            Actions: {
                close: {
                    '9.5.0': string;
                };
            };
            General: {
                deleteDashBoard: {
                    '11.1.0': string;
                };
                sectionItems: {
                    "8.5.0": (item: string) => string;
                };
                saveDashBoard: {
                    "8.5.0": string;
                };
                saveAsDashBoard: {
                    "8.5.0": string;
                };
                title: {
                    '11.2.0': string;
                };
            };
            Annotations: {
                Edit: {
                    urlParams: {
                        "8.5.0": (annotationIndex: string) => string;
                    };
                };
                List: {
                    url: {
                        "8.5.0": (dashboardUid: string) => string;
                    };
                    addAnnotationCTAV2: {
                        "8.5.0": string;
                    };
                    annotations: {
                        '10.4.0': string;
                    };
                };
                Settings: {
                    name: {
                        '11.1.0': string;
                        "8.5.0": string;
                    };
                };
                NewAnnotation: {
                    panelFilterSelect: {
                        '10.0.0': string;
                    };
                    showInLabel: {
                        '11.1.0': string;
                    };
                    previewInDashboard: {
                        '10.0.0': string;
                    };
                    delete: {
                        '10.4.0': string;
                    };
                    apply: {
                        '10.4.0': string;
                    };
                    enable: {
                        '10.4.0': string;
                    };
                    hide: {
                        '10.4.0': string;
                    };
                };
            };
            Variables: {
                List: {
                    url: {
                        '11.3.0': (dashboardUid: string) => string;
                        "8.5.0": (dashboardUid: string) => string;
                    };
                    addVariableCTAV2: {
                        "8.5.0": string;
                    };
                    newButton: {
                        "8.5.0": string;
                    };
                    table: {
                        "8.5.0": string;
                    };
                    tableRowNameFields: {
                        "8.5.0": (variableName: string) => string;
                    };
                    tableRowDefinitionFields: {
                        '10.1.0': (variableName: string) => string;
                    };
                    tableRowArrowUpButtons: {
                        "8.5.0": (variableName: string) => string;
                    };
                    tableRowArrowDownButtons: {
                        "8.5.0": (variableName: string) => string;
                    };
                    tableRowDuplicateButtons: {
                        "8.5.0": (variableName: string) => string;
                    };
                    tableRowRemoveButtons: {
                        "8.5.0": (variableName: string) => string;
                    };
                };
                Edit: {
                    urlParams: {
                        '11.3.0': (editIndex: string) => string;
                        "8.5.0": (editIndex: string) => string;
                    };
                    General: {
                        headerLink: {
                            "8.5.0": string;
                        };
                        modeLabelNew: {
                            "8.5.0": string;
                        };
                        modeLabelEdit: {
                            "8.5.0": string;
                        };
                        generalNameInput: {
                            "8.5.0": string;
                        };
                        generalNameInputV2: {
                            "8.5.0": string;
                        };
                        generalTypeSelect: {
                            "8.5.0": string;
                        };
                        generalTypeSelectV2: {
                            "8.5.0": string;
                        };
                        generalLabelInput: {
                            "8.5.0": string;
                        };
                        generalLabelInputV2: {
                            "8.5.0": string;
                        };
                        generalHideSelect: {
                            "8.5.0": string;
                        };
                        generalHideSelectV2: {
                            "8.5.0": string;
                        };
                        selectionOptionsAllowCustomValueSwitch: {
                            "8.5.0": string;
                        };
                        selectionOptionsMultiSwitch: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        selectionOptionsIncludeAllSwitch: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        selectionOptionsCustomAllInput: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        previewOfValuesOption: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        submitButton: {
                            '10.4.0': string;
                            "8.5.0": string;
                        };
                        applyButton: {
                            '9.3.0': string;
                        };
                    };
                    QueryVariable: {
                        queryOptionsDataSourceSelect: {
                            '10.4.0': string;
                            '10.0.0': string;
                            "8.5.0": string;
                        };
                        queryOptionsRefreshSelect: {
                            "8.5.0": string;
                        };
                        queryOptionsRefreshSelectV2: {
                            "8.5.0": string;
                        };
                        queryOptionsRegExInput: {
                            "8.5.0": string;
                        };
                        queryOptionsRegExInputV2: {
                            "8.5.0": string;
                        };
                        queryOptionsSortSelect: {
                            "8.5.0": string;
                        };
                        queryOptionsSortSelectV2: {
                            "8.5.0": string;
                        };
                        queryOptionsQueryInput: {
                            '10.4.0': string;
                        };
                        valueGroupsTagsEnabledSwitch: {
                            "8.5.0": string;
                        };
                        valueGroupsTagsTagsQueryInput: {
                            "8.5.0": string;
                        };
                        valueGroupsTagsTagsValuesQueryInput: {
                            "8.5.0": string;
                        };
                    };
                    ConstantVariable: {
                        constantOptionsQueryInput: {
                            "8.5.0": string;
                        };
                        constantOptionsQueryInputV2: {
                            "8.5.0": string;
                        };
                    };
                    DatasourceVariable: {
                        datasourceSelect: {
                            "8.5.0": string;
                        };
                    };
                    TextBoxVariable: {
                        textBoxOptionsQueryInput: {
                            "8.5.0": string;
                        };
                        textBoxOptionsQueryInputV2: {
                            "8.5.0": string;
                        };
                    };
                    CustomVariable: {
                        customValueInput: {
                            "8.5.0": string;
                        };
                    };
                    IntervalVariable: {
                        intervalsValueInput: {
                            "8.5.0": string;
                        };
                        autoEnabledCheckbox: {
                            '10.4.0': string;
                        };
                        stepCountIntervalSelect: {
                            '10.4.0': string;
                        };
                        minIntervalInput: {
                            '10.4.0': string;
                        };
                    };
                    GroupByVariable: {
                        dataSourceSelect: {
                            '10.4.0': string;
                        };
                        infoText: {
                            '10.4.0': string;
                        };
                        modeToggle: {
                            '10.4.0': string;
                        };
                    };
                    AdHocFiltersVariable: {
                        datasourceSelect: {
                            '10.4.0': string;
                        };
                        infoText: {
                            '10.4.0': string;
                        };
                        modeToggle: {
                            '11.0.0': string;
                        };
                    };
                };
            };
        };
        Annotations: {
            marker: {
                '10.0.0': string;
            };
        };
        Rows: {
            Repeated: {
                ConfigSection: {
                    warningMessage: {
                        '10.2.0': string;
                    };
                };
            };
        };
    };
    Dashboards: {
        url: {
            "8.5.0": string;
        };
        dashboards: {
            '10.2.0': (title: string) => string;
        };
    };
    SaveDashboardAsModal: {
        newName: {
            '10.2.0': string;
        };
        save: {
            '10.2.0': string;
        };
    };
    SaveDashboardModal: {
        save: {
            '10.2.0': string;
        };
        saveVariables: {
            '10.2.0': string;
        };
        saveTimerange: {
            '10.2.0': string;
        };
        saveRefresh: {
            '11.1.0': string;
        };
    };
    SharePanelModal: {
        linkToRenderedImage: {
            "8.5.0": string;
        };
    };
    ShareDashboardModal: {
        PublicDashboard: {
            WillBePublicCheckbox: {
                '9.1.0': string;
            };
            LimitedDSCheckbox: {
                '9.1.0': string;
            };
            CostIncreaseCheckbox: {
                '9.1.0': string;
            };
            PauseSwitch: {
                '9.5.0': string;
            };
            EnableAnnotationsSwitch: {
                '9.3.0': string;
            };
            CreateButton: {
                '9.5.0': string;
            };
            DeleteButton: {
                '9.3.0': string;
            };
            CopyUrlInput: {
                '9.1.0': string;
            };
            CopyUrlButton: {
                '9.1.0': string;
            };
            SettingsDropdown: {
                '10.1.0': string;
            };
            TemplateVariablesWarningAlert: {
                '9.1.0': string;
            };
            UnsupportedDataSourcesWarningAlert: {
                '9.5.0': string;
            };
            NoUpsertPermissionsWarningAlert: {
                '9.5.0': string;
            };
            EnableTimeRangeSwitch: {
                '9.4.0': string;
            };
            EmailSharingConfiguration: {
                Container: {
                    '9.5.0': string;
                };
                ShareType: {
                    '9.5.0': string;
                };
                EmailSharingInput: {
                    '9.5.0': string;
                };
                EmailSharingInviteButton: {
                    '9.5.0': string;
                };
                EmailSharingList: {
                    '9.5.0': string;
                };
                DeleteEmail: {
                    '9.5.0': string;
                };
                ReshareLink: {
                    '9.5.0': string;
                };
            };
        };
        SnapshotScene: {
            url: {
                '11.1.0': (key: string) => string;
            };
            PublishSnapshot: {
                '11.1.0': string;
            };
            CopyUrlButton: {
                '11.1.0': string;
            };
            CopyUrlInput: {
                '11.1.0': string;
            };
        };
    };
    ShareDashboardDrawer: {
        ShareInternally: {
            container: {
                '11.3.0': string;
            };
            lockTimeRangeSwitch: {
                '11.3.0': string;
            };
            shortenUrlSwitch: {
                '11.3.0': string;
            };
            copyUrlButton: {
                '11.3.0': string;
            };
            SharePanel: {
                preview: {
                    '11.5.0': string;
                };
                widthInput: {
                    '11.5.0': string;
                };
                heightInput: {
                    '11.5.0': string;
                };
                scaleFactorInput: {
                    '11.5.0': string;
                };
                generateImageButton: {
                    '11.5.0': string;
                };
                downloadImageButton: {
                    '11.5.0': string;
                };
            };
        };
        ShareExternally: {
            container: {
                '11.3.0': string;
            };
            publicAlert: {
                '11.3.0': string;
            };
            emailSharingAlert: {
                '11.3.0': string;
            };
            shareTypeSelect: {
                '11.3.0': string;
            };
            Creation: {
                PublicShare: {
                    createButton: {
                        '11.3.0': string;
                    };
                    cancelButton: {
                        '11.3.0': string;
                    };
                };
                EmailShare: {
                    createButton: {
                        '11.3.0': string;
                    };
                    cancelButton: {
                        '11.3.0': string;
                    };
                };
                willBePublicCheckbox: {
                    '11.3.0': string;
                };
            };
            Configuration: {
                enableTimeRangeSwitch: {
                    '11.3.0': string;
                };
                enableAnnotationsSwitch: {
                    '11.3.0': string;
                };
                copyUrlButton: {
                    '11.3.0': string;
                };
                revokeAccessButton: {
                    '11.3.0': string;
                };
                toggleAccessButton: {
                    '11.3.0': string;
                };
            };
        };
        ShareSnapshot: {
            url: {
                '11.3.0': (key: string) => string;
            };
            container: {
                '11.3.0': string;
            };
            publishSnapshot: {
                '11.3.0': string;
            };
            copyUrlButton: {
                '11.3.0': string;
            };
        };
    };
    ExportDashboardDrawer: {
        ExportAsJson: {
            container: {
                '11.3.0': string;
            };
            codeEditor: {
                '11.3.0': string;
            };
            exportExternallyToggle: {
                '11.3.0': string;
            };
            saveToFileButton: {
                '11.3.0': string;
            };
            copyToClipboardButton: {
                '11.3.0': string;
            };
            cancelButton: {
                '11.3.0': string;
            };
        };
    };
    PublicDashboard: {
        page: {
            '9.5.0': string;
        };
        NotAvailable: {
            container: {
                '9.5.0': string;
            };
            title: {
                '9.5.0': string;
            };
            pausedDescription: {
                '9.5.0': string;
            };
        };
        footer: {
            '11.0.0': string;
        };
    };
    PublicDashboardScene: {
        loadingPage: {
            '11.0.0': string;
        };
        page: {
            '11.0.0': string;
        };
        controls: {
            '11.0.0': string;
        };
    };
    RequestViewAccess: {
        form: {
            '9.5.0': string;
        };
        recipientInput: {
            '9.5.0': string;
        };
        submitButton: {
            '9.5.0': string;
        };
    };
    PublicDashboardConfirmAccess: {
        submitButton: {
            '10.2.0': string;
        };
    };
    Explore: {
        url: {
            "8.5.0": string;
        };
        General: {
            container: {
                "8.5.0": string;
            };
            graph: {
                "8.5.0": string;
            };
            table: {
                "8.5.0": string;
            };
            scrollView: {
                '9.0.0': string;
            };
        };
        QueryHistory: {
            container: {
                '11.1.0': string;
            };
        };
    };
    SoloPanel: {
        url: {
            "8.5.0": (page: string) => string;
        };
    };
    PluginsList: {
        page: {
            "8.5.0": string;
        };
        list: {
            "8.5.0": string;
        };
        listItem: {
            "8.5.0": string;
        };
        signatureErrorNotice: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    PluginPage: {
        page: {
            "8.5.0": string;
        };
        signatureInfo: {
            '10.3.0': string;
            "8.5.0": string;
        };
        disabledInfo: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    PlaylistForm: {
        name: {
            "8.5.0": string;
        };
        interval: {
            "8.5.0": string;
        };
        itemDelete: {
            '10.2.0': string;
        };
    };
    BrowseDashboards: {
        table: {
            body: {
                '10.2.0': string;
            };
            row: {
                '10.2.0': (name: string) => string;
            };
            checkbox: {
                '10.0.0': (uid: string) => string;
            };
        };
        NewFolderForm: {
            form: {
                '10.2.0': string;
            };
            nameInput: {
                '10.2.0': string;
            };
            createButton: {
                '10.2.0': string;
            };
        };
    };
    Search: {
        url: {
            '9.3.0': string;
        };
        FolderView: {
            url: {
                '9.3.0': string;
            };
        };
    };
    PublicDashboards: {
        ListItem: {
            linkButton: {
                '9.3.0': string;
            };
            configButton: {
                '9.3.0': string;
            };
            trashcanButton: {
                '9.3.0': string;
            };
            pauseSwitch: {
                '10.1.0': string;
            };
        };
    };
    UserListPage: {
        tabs: {
            allUsers: {
                '10.0.0': string;
            };
            orgUsers: {
                '10.0.0': string;
            };
            anonUserDevices: {
                '10.2.3': string;
            };
            publicDashboardsUsers: {
                '10.0.0': string;
            };
            users: {
                '10.0.0': string;
            };
        };
        org: {
            url: {
                '10.2.0': string;
                '9.5.0': string;
            };
        };
        admin: {
            url: {
                '9.5.0': string;
            };
        };
        publicDashboards: {
            container: {
                '11.1.0': string;
            };
        };
        UserListAdminPage: {
            container: {
                '10.0.0': string;
            };
        };
        UsersListPage: {
            container: {
                '10.0.0': string;
            };
        };
        UserAnonListPage: {
            container: {
                '10.4.0': string;
            };
        };
        UsersListPublicDashboardsPage: {
            container: {
                '10.0.0': string;
            };
            DashboardsListModal: {
                listItem: {
                    '10.0.0': (uid: string) => string;
                };
            };
        };
    };
    ProfilePage: {
        url: {
            '10.2.0': string;
        };
    };
    Plugin: {
        url: {
            "8.5.0": (pluginId: string) => string;
        };
    };
}>;
declare const Components: SelectorsOf<{
    RadioButton: {
        container: {
            '10.2.3': string;
        };
    };
    Breadcrumbs: {
        breadcrumb: {
            '9.4.0': (title: string) => string;
        };
    };
    TimePicker: {
        openButton: {
            "8.5.0": string;
        };
        overlayContent: {
            '10.2.3': string;
        };
        fromField: {
            '10.2.3': string;
            "8.5.0": string;
        };
        toField: {
            '10.2.3': string;
            "8.5.0": string;
        };
        applyTimeRange: {
            "8.5.0": string;
        };
        copyTimeRange: {
            '10.4.0': string;
        };
        pasteTimeRange: {
            '10.4.0': string;
        };
        calendar: {
            label: {
                '10.2.3': string;
                "8.5.0": string;
            };
            openButton: {
                '10.2.3': string;
                "8.5.0": string;
            };
            closeButton: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        absoluteTimeRangeTitle: {
            "8.5.0": string;
        };
    };
    DataSourcePermissions: {
        form: {
            '9.5.0': () => string;
        };
        roleType: {
            '9.5.0': string;
        };
        rolePicker: {
            '9.5.0': string;
        };
        permissionLevel: {
            '9.5.0': string;
        };
    };
    DateTimePicker: {
        input: {
            '10.2.3': string;
        };
    };
    DataSource: {
        TestData: {
            QueryTab: {
                scenarioSelectContainer: {
                    "8.5.0": string;
                };
                scenarioSelect: {
                    "8.5.0": string;
                };
                max: {
                    "8.5.0": string;
                };
                min: {
                    "8.5.0": string;
                };
                noise: {
                    "8.5.0": string;
                };
                seriesCount: {
                    "8.5.0": string;
                };
                spread: {
                    "8.5.0": string;
                };
                startValue: {
                    "8.5.0": string;
                };
                drop: {
                    "8.5.0": string;
                };
            };
        };
        DataSourceHttpSettings: {
            urlInput: {
                '10.4.0': string;
                "8.5.0": string;
            };
        };
        Jaeger: {
            traceIDInput: {
                "8.5.0": string;
            };
        };
        Prometheus: {
            configPage: {
                connectionSettings: {
                    "8.5.0": string;
                };
                manageAlerts: {
                    '10.4.0': string;
                };
                scrapeInterval: {
                    '10.4.0': string;
                };
                queryTimeout: {
                    '10.4.0': string;
                };
                defaultEditor: {
                    '10.4.0': string;
                };
                disableMetricLookup: {
                    '10.4.0': string;
                };
                prometheusType: {
                    '10.4.0': string;
                };
                prometheusVersion: {
                    '10.4.0': string;
                };
                cacheLevel: {
                    '10.4.0': string;
                };
                incrementalQuerying: {
                    '10.4.0': string;
                };
                queryOverlapWindow: {
                    '10.4.0': string;
                };
                disableRecordingRules: {
                    '10.4.0': string;
                };
                customQueryParameters: {
                    '10.4.0': string;
                };
                httpMethod: {
                    '10.4.0': string;
                };
                exemplarsAddButton: {
                    '10.3.0': string;
                    "8.5.0": string;
                };
                internalLinkSwitch: {
                    '10.3.0': string;
                    "8.5.0": string;
                };
                codeModeMetricNamesSuggestionLimit: {
                    '11.1.0': string;
                };
            };
            queryEditor: {
                explain: {
                    '10.4.0': string;
                };
                editorToggle: {
                    '10.4.0': string;
                };
                options: {
                    '10.4.0': string;
                };
                legend: {
                    '10.4.0': string;
                };
                format: {
                    '10.4.0': string;
                };
                step: {
                    '10.4.0': string;
                };
                type: {
                    '10.4.0': string;
                };
                exemplars: {
                    '10.4.0': string;
                };
                builder: {
                    metricSelect: {
                        '10.4.0': string;
                    };
                    hints: {
                        '10.4.0': string;
                    };
                    metricsExplorer: {
                        '10.4.0': string;
                    };
                    queryAdvisor: {
                        '10.4.0': string;
                    };
                };
                code: {
                    queryField: {
                        '10.4.0': string;
                    };
                    metricsCountInfo: {
                        '11.1.0': string;
                    };
                    metricsBrowser: {
                        openButton: {
                            '10.4.0': string;
                        };
                        selectMetric: {
                            '10.4.0': string;
                        };
                        seriesLimit: {
                            '10.3.1': string;
                        };
                        metricList: {
                            '10.4.0': string;
                        };
                        labelNamesFilter: {
                            '10.4.0': string;
                        };
                        labelValuesFilter: {
                            '10.4.0': string;
                        };
                        useQuery: {
                            '10.4.0': string;
                        };
                        useAsRateQuery: {
                            '10.4.0': string;
                        };
                        validateSelector: {
                            '10.4.0': string;
                        };
                        clear: {
                            '10.4.0': string;
                        };
                    };
                };
            };
            exemplarMarker: {
                '10.3.0': string;
                "8.5.0": string;
            };
            variableQueryEditor: {
                queryType: {
                    '10.4.0': string;
                };
                labelnames: {
                    metricRegex: {
                        '10.4.0': string;
                    };
                };
                labelValues: {
                    labelSelect: {
                        '10.4.0': string;
                    };
                };
                metricNames: {
                    metricRegex: {
                        '10.4.0': string;
                    };
                };
                varQueryResult: {
                    '10.4.0': string;
                };
                seriesQuery: {
                    '10.4.0': string;
                };
                classicQuery: {
                    '10.4.0': string;
                };
            };
            annotations: {
                minStep: {
                    '10.4.0': string;
                };
                title: {
                    '10.4.0': string;
                };
                tags: {
                    '10.4.0': string;
                };
                text: {
                    '10.4.0': string;
                };
                seriesValueAsTimestamp: {
                    '10.4.0': string;
                };
            };
        };
    };
    Menu: {
        MenuComponent: {
            "8.5.0": (title: string) => string;
        };
        MenuGroup: {
            "8.5.0": (title: string) => string;
        };
        MenuItem: {
            "8.5.0": (title: string) => string;
        };
        SubMenu: {
            container: {
                '10.3.0': string;
                "8.5.0": string;
            };
            icon: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
    };
    Panels: {
        Panel: {
            title: {
                "8.5.0": (title: string) => string;
            };
            content: {
                '11.1.0': string;
            };
            headerItems: {
                '10.2.0': (item: string) => string;
            };
            menuItems: {
                '9.5.0': (item: string) => string;
            };
            menu: {
                '9.5.0': (title: string) => string;
            };
            containerByTitle: {
                "8.5.0": (title: string) => string;
            };
            headerCornerInfo: {
                "8.5.0": (mode: string) => string;
            };
            status: {
                '10.2.0': (status: string) => string;
                "8.5.0": (_: string) => string;
            };
            loadingBar: {
                '10.0.0': () => string;
            };
            HoverWidget: {
                container: {
                    '10.1.0': string;
                    "8.5.0": string;
                };
                dragIcon: {
                    '10.0.0': string;
                };
            };
            PanelDataErrorMessage: {
                '10.4.0': string;
            };
        };
        Visualization: {
            Graph: {
                container: {
                    '9.5.0': string;
                };
                VisualizationTab: {
                    legendSection: {
                        "8.5.0": string;
                    };
                };
                Legend: {
                    legendItemAlias: {
                        "8.5.0": (name: string) => string;
                    };
                    showLegendSwitch: {
                        "8.5.0": string;
                    };
                };
                xAxis: {
                    labels: {
                        "8.5.0": () => string;
                    };
                };
            };
            BarGauge: {
                valueV2: {
                    "8.5.0": string;
                };
            };
            PieChart: {
                svgSlice: {
                    '10.3.0': string;
                };
            };
            Text: {
                container: {
                    "8.5.0": () => string;
                };
            };
            Table: {
                header: {
                    "8.5.0": string;
                };
                footer: {
                    "8.5.0": string;
                };
                body: {
                    '10.2.0': string;
                };
            };
        };
    };
    VizLegend: {
        seriesName: {
            '10.3.0': (name: string) => string;
        };
    };
    Drawer: {
        General: {
            title: {
                "8.5.0": (title: string) => string;
            };
            expand: {
                "8.5.0": string;
            };
            contract: {
                "8.5.0": string;
            };
            close: {
                '10.3.0': string;
                "8.5.0": string;
            };
            rcContentWrapper: {
                '9.4.0': () => string;
            };
            subtitle: {
                '10.4.0': string;
            };
        };
        DashboardSaveDrawer: {
            saveButton: {
                '11.1.0': string;
            };
            saveAsButton: {
                '11.1.0': string;
            };
            saveAsTitleInput: {
                '11.1.0': string;
            };
        };
    };
    PanelEditor: {
        General: {
            content: {
                '11.1.0': string;
                '8.0.0': string;
            };
        };
        OptionsPane: {
            content: {
                '11.1.0': string;
                "8.5.0": string;
            };
            select: {
                "8.5.0": string;
            };
            fieldLabel: {
                "8.5.0": (type: string) => string;
            };
            fieldInput: {
                '11.0.0': (title: string) => string;
            };
        };
        DataPane: {
            content: {
                '11.1.0': string;
                "8.5.0": string;
            };
        };
        applyButton: {
            '9.2.0': string;
            '9.1.0': string;
            '8.0.0': string;
        };
        toggleVizPicker: {
            '10.0.0': string;
            '8.0.0': string;
        };
        toggleVizOptions: {
            '10.1.0': string;
            "8.5.0": string;
        };
        toggleTableView: {
            '11.1.0': string;
            "8.5.0": string;
        };
        showZoomField: {
            '10.2.0': string;
        };
        showAttributionField: {
            '10.2.0': string;
        };
        showScaleField: {
            '10.2.0': string;
        };
        showMeasureField: {
            '10.2.0': string;
        };
        showDebugField: {
            '10.2.0': string;
        };
        measureButton: {
            '9.2.0': string;
        };
    };
    PanelInspector: {
        Data: {
            content: {
                "8.5.0": string;
            };
        };
        Stats: {
            content: {
                "8.5.0": string;
            };
        };
        Json: {
            content: {
                '11.1.0': string;
                "8.5.0": string;
            };
        };
        Query: {
            content: {
                "8.5.0": string;
            };
            refreshButton: {
                "8.5.0": string;
            };
            jsonObjectKeys: {
                "8.5.0": () => string;
            };
        };
    };
    Tab: {
        title: {
            '11.2.0': (title: string) => string;
        };
        active: {
            "8.5.0": () => string;
        };
    };
    RefreshPicker: {
        runButtonV2: {
            "8.5.0": string;
        };
        intervalButtonV2: {
            "8.5.0": string;
        };
    };
    QueryTab: {
        content: {
            "8.5.0": string;
        };
        queryInspectorButton: {
            "8.5.0": string;
        };
        queryHistoryButton: {
            '10.2.0': string;
            "8.5.0": string;
        };
        addQuery: {
            '10.2.0': string;
            "8.5.0": string;
        };
        queryGroupTopSection: {
            '11.2.0': string;
        };
        addExpression: {
            '11.2.0': string;
        };
    };
    QueryHistory: {
        queryText: {
            '9.0.0': string;
        };
    };
    QueryEditorRows: {
        rows: {
            "8.5.0": string;
        };
    };
    QueryEditorRow: {
        actionButton: {
            '10.4.0': (title: string) => string;
        };
        title: {
            "8.5.0": (refId: string) => string;
        };
        container: {
            "8.5.0": (refId: string) => string;
        };
    };
    AlertTab: {
        content: {
            '10.2.3': string;
            "8.5.0": string;
        };
    };
    AlertRules: {
        groupToggle: {
            '11.0.0': string;
        };
        toggle: {
            '11.0.0': string;
        };
        expandedContent: {
            '11.0.0': string;
        };
        previewButton: {
            '11.1.0': string;
        };
        ruleNameField: {
            '11.1.0': string;
        };
        newFolderButton: {
            '11.1.0': string;
        };
        newFolderNameField: {
            '11.1.0': string;
        };
        newFolderNameCreateButton: {
            '11.1.0': string;
        };
        newEvaluationGroupButton: {
            '11.1.0': string;
        };
        newEvaluationGroupName: {
            '11.1.0': string;
        };
        newEvaluationGroupInterval: {
            '11.1.0': string;
        };
        newEvaluationGroupCreate: {
            '11.1.0': string;
        };
    };
    Alert: {
        alertV2: {
            "8.5.0": (severity: string) => string;
        };
    };
    TransformTab: {
        content: {
            '10.1.0': string;
            "8.5.0": string;
        };
        newTransform: {
            '10.1.0': (name: string) => string;
        };
        transformationEditor: {
            '10.1.0': (name: string) => string;
        };
        transformationEditorDebugger: {
            '10.1.0': (name: string) => string;
        };
    };
    Transforms: {
        card: {
            '10.1.0': (name: string) => string;
        };
        disableTransformationButton: {
            '10.4.0': string;
        };
        Reduce: {
            modeLabel: {
                '10.2.3': string;
                "8.5.0": string;
            };
            calculationsLabel: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        SpatialOperations: {
            actionLabel: {
                '9.1.2': string;
            };
            locationLabel: {
                '10.2.0': string;
            };
            location: {
                autoOption: {
                    '9.1.2': string;
                };
                coords: {
                    option: {
                        '9.1.2': string;
                    };
                    latitudeFieldLabel: {
                        '9.1.2': string;
                    };
                    longitudeFieldLabel: {
                        '9.1.2': string;
                    };
                };
                geohash: {
                    option: {
                        '9.1.2': string;
                    };
                    geohashFieldLabel: {
                        '9.1.2': string;
                    };
                };
                lookup: {
                    option: {
                        '9.1.2': string;
                    };
                    lookupFieldLabel: {
                        '9.1.2': string;
                    };
                    gazetteerFieldLabel: {
                        '9.1.2': string;
                    };
                };
            };
        };
        searchInput: {
            '10.2.3': string;
            "8.5.0": string;
        };
        noTransformationsMessage: {
            '10.2.3': string;
        };
        addTransformationButton: {
            '10.1.0': string;
            "8.5.0": string;
        };
        removeAllTransformationsButton: {
            '10.4.0': string;
        };
    };
    NavBar: {
        Configuration: {
            button: {
                '9.5.0': string;
            };
        };
        Toggle: {
            button: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        Reporting: {
            button: {
                '9.5.0': string;
            };
        };
    };
    NavMenu: {
        Menu: {
            '10.2.3': string;
        };
        item: {
            '9.5.0': string;
        };
    };
    NavToolbar: {
        container: {
            '9.4.0': string;
        };
        commandPaletteTrigger: {
            '11.5.0': string;
        };
        shareDashboard: {
            '11.1.0': string;
        };
        markAsFavorite: {
            '11.1.0': string;
        };
        editDashboard: {
            editButton: {
                '11.1.0': string;
            };
            saveButton: {
                '11.1.0': string;
            };
            exitButton: {
                '11.1.0': string;
            };
            settingsButton: {
                '11.1.0': string;
            };
            addRowButton: {
                '11.1.0': string;
            };
            addLibraryPanelButton: {
                '11.1.0': string;
            };
            addVisualizationButton: {
                '11.1.0': string;
            };
            pastePanelButton: {
                '11.1.0': string;
            };
            discardChangesButton: {
                '11.1.0': string;
            };
            discardLibraryPanelButton: {
                '11.1.0': string;
            };
            unlinkLibraryPanelButton: {
                '11.1.0': string;
            };
            saveLibraryPanelButton: {
                '11.1.0': string;
            };
            backToDashboardButton: {
                '11.1.0': string;
            };
        };
    };
    PageToolbar: {
        container: {
            "8.5.0": () => string;
        };
        item: {
            "8.5.0": (tooltip: string) => string;
        };
        itemButton: {
            '9.5.0': (title: string) => string;
        };
    };
    QueryEditorToolbarItem: {
        button: {
            "8.5.0": (title: string) => string;
        };
    };
    BackButton: {
        backArrow: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    OptionsGroup: {
        group: {
            '11.1.0': (title?: string) => string;
            "8.5.0": (title?: string) => string;
        };
        toggle: {
            '11.1.0': (title?: string) => string;
            "8.5.0": (title?: string) => string;
        };
    };
    PluginVisualization: {
        item: {
            "8.5.0": (title: string) => string;
        };
        current: {
            "8.5.0": () => string;
        };
    };
    Select: {
        menu: {
            '11.5.0': string;
            "8.5.0": string;
        };
        option: {
            '11.1.0': string;
            "8.5.0": string;
        };
        toggleAllOptions: {
            '11.3.0': string;
        };
        input: {
            "8.5.0": () => string;
        };
        singleValue: {
            "8.5.0": () => string;
        };
    };
    FieldConfigEditor: {
        content: {
            "8.5.0": string;
        };
    };
    OverridesConfigEditor: {
        content: {
            "8.5.0": string;
        };
    };
    FolderPicker: {
        containerV2: {
            "8.5.0": string;
        };
        input: {
            '10.4.0': string;
        };
    };
    ReadonlyFolderPicker: {
        container: {
            "8.5.0": string;
        };
    };
    DataSourcePicker: {
        container: {
            '10.0.0': string;
            '8.0.0': string;
        };
        inputV2: {
            '10.1.0': string;
            "8.5.0": string;
        };
        dataSourceList: {
            '10.4.0': string;
        };
        advancedModal: {
            dataSourceList: {
                '10.4.0': string;
            };
            builtInDataSourceList: {
                '10.4.0': string;
            };
        };
    };
    TimeZonePicker: {
        containerV2: {
            "8.5.0": string;
        };
        changeTimeSettingsButton: {
            '11.0.0': string;
        };
    };
    WeekStartPicker: {
        containerV2: {
            "8.5.0": string;
        };
        placeholder: {
            "8.5.0": string;
        };
    };
    TraceViewer: {
        spanBar: {
            '9.0.0': string;
        };
    };
    QueryField: {
        container: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    QueryBuilder: {
        queryPatterns: {
            '10.3.0': string;
            "8.5.0": string;
        };
        labelSelect: {
            '10.3.0': string;
            "8.5.0": string;
        };
        inputSelect: {
            '11.1.0': string;
        };
        valueSelect: {
            '10.3.0': string;
            "8.5.0": string;
        };
        matchOperatorSelect: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    ValuePicker: {
        button: {
            '10.3.0': (name: string) => string;
        };
        select: {
            '10.3.0': (name: string) => string;
        };
    };
    Search: {
        sectionV2: {
            "8.5.0": string;
        };
        itemsV2: {
            "8.5.0": string;
        };
        cards: {
            "8.5.0": string;
        };
        collapseFolder: {
            "8.5.0": (sectionId: string) => string;
        };
        expandFolder: {
            "8.5.0": (sectionId: string) => string;
        };
        dashboardItem: {
            "8.5.0": (item: string) => string;
        };
        dashboardCard: {
            "8.5.0": (item: string) => string;
        };
        folderHeader: {
            '9.3.0': (folderName: string) => string;
        };
        folderContent: {
            '9.3.0': (folderName: string) => string;
        };
        dashboardItems: {
            "8.5.0": string;
        };
    };
    DashboardLinks: {
        container: {
            "8.5.0": string;
        };
        dropDown: {
            "8.5.0": string;
        };
        link: {
            "8.5.0": string;
        };
    };
    LoadingIndicator: {
        icon: {
            '10.4.0': string;
            "8.5.0": string;
        };
    };
    CallToActionCard: {
        buttonV2: {
            "8.5.0": (name: string) => string;
        };
    };
    DataLinksContextMenu: {
        singleLink: {
            '10.3.0': string;
            "8.5.0": string;
        };
    };
    CodeEditor: {
        container: {
            '10.2.3': string;
            "8.5.0": string;
        };
    };
    ReactMonacoEditor: {
        editorLazy: {
            '11.1.0': string;
        };
    };
    DashboardImportPage: {
        textarea: {
            "8.5.0": string;
        };
        submit: {
            "8.5.0": string;
        };
    };
    ImportDashboardForm: {
        name: {
            "8.5.0": string;
        };
        submit: {
            "8.5.0": string;
        };
    };
    PanelAlertTabContent: {
        content: {
            '10.2.3': string;
            "8.5.0": string;
        };
    };
    VisualizationPreview: {
        card: {
            "8.5.0": (name: string) => string;
        };
    };
    ColorSwatch: {
        name: {
            "8.5.0": string;
        };
    };
    DashboardRow: {
        title: {
            "8.5.0": (title: string) => string;
        };
    };
    UserProfile: {
        profileSaveButton: {
            "8.5.0": string;
        };
        preferencesSaveButton: {
            "8.5.0": string;
        };
        orgsTable: {
            "8.5.0": string;
        };
        sessionsTable: {
            "8.5.0": string;
        };
        extensionPointTabs: {
            '10.2.3': string;
        };
        extensionPointTab: {
            '10.2.3': (tabId: string) => string;
        };
    };
    FileUpload: {
        inputField: {
            '9.0.0': string;
        };
        fileNameSpan: {
            '9.0.0': string;
        };
    };
    DebugOverlay: {
        wrapper: {
            '9.2.0': string;
        };
    };
    OrgRolePicker: {
        input: {
            '9.5.0': string;
        };
    };
    AnalyticsToolbarButton: {
        button: {
            '9.5.0': string;
        };
    };
    Variables: {
        variableOption: {
            '9.5.0': string;
        };
        variableLinkWrapper: {
            '11.1.1': string;
        };
    };
    Annotations: {
        annotationsTypeInput: {
            '11.1.0': string;
            "8.5.0": string;
        };
        annotationsChoosePanelInput: {
            '11.1.0': string;
            "8.5.0": string;
        };
        editor: {
            testButton: {
                '11.0.0': string;
            };
            resultContainer: {
                '11.0.0': string;
            };
        };
    };
    Tooltip: {
        container: {
            '10.2.0': string;
        };
    };
    ReturnToPrevious: {
        buttonGroup: {
            '11.0.0': string;
        };
        backButton: {
            '11.0.0': string;
        };
        dismissButton: {
            '11.0.0': string;
        };
    };
    SQLQueryEditor: {
        selectColumn: {
            '11.0.0': string;
        };
        selectColumnInput: {
            '11.0.0': string;
        };
        selectFunctionParameter: {
            '11.0.0': (name: string) => string;
        };
        selectAggregation: {
            '11.0.0': string;
        };
        selectAggregationInput: {
            '11.0.0': string;
        };
        selectAlias: {
            '11.0.0': string;
        };
        selectAliasInput: {
            '11.0.0': string;
        };
        selectInputParameter: {
            '11.0.0': string;
        };
        filterConjunction: {
            '11.0.0': string;
        };
        filterField: {
            '11.0.0': string;
        };
        filterOperator: {
            '11.0.0': string;
        };
        headerTableSelector: {
            '11.0.0': string;
        };
        headerFilterSwitch: {
            '11.0.0': string;
        };
        headerGroupSwitch: {
            '11.0.0': string;
        };
        headerOrderSwitch: {
            '11.0.0': string;
        };
        headerPreviewSwitch: {
            '11.0.0': string;
        };
    };
    EntityNotFound: {
        container: {
            '11.2.0': string;
        };
    };
    Portal: {
        container: {
            '11.5.0': string;
        };
    };
}>;
declare const selectors: {
    pages: SelectorsOf<{
        Alerting: {
            AddAlertRule: {
                url: {
                    '10.1.0': string;
                    "8.5.0": string;
                };
            };
            EditAlertRule: {
                url: {
                    "8.5.0": (alertRuleUid: string) => string;
                };
            };
        };
        Login: {
            url: {
                "8.5.0": string;
            };
            username: {
                '10.2.3': string;
                "8.5.0": string;
            };
            password: {
                '10.2.3': string;
                "8.5.0": string;
            };
            submit: {
                '10.2.3': string;
                "8.5.0": string;
            };
            skip: {
                '10.2.3': string;
            };
        };
        PasswordlessLogin: {
            url: {
                "8.5.0": string;
            };
            email: {
                '10.2.3': string;
                "8.5.0": string;
            };
            submit: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        Home: {
            url: {
                "8.5.0": string;
            };
        };
        DataSource: {
            name: {
                '10.3.0': string;
                "8.5.0": string;
            };
            delete: {
                "8.5.0": string;
            };
            readOnly: {
                '10.3.0': string;
                "8.5.0": string;
            };
            saveAndTest: {
                '10.0.0': string;
                "8.5.0": string;
            };
            alert: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
        DataSources: {
            url: {
                "8.5.0": string;
            };
            dataSources: {
                "8.5.0": (dataSourceName: string) => string;
            };
        };
        EditDataSource: {
            url: {
                '9.5.0': (dataSourceUid: string) => string;
            };
            settings: {
                '9.5.0': string;
            };
        };
        AddDataSource: {
            url: {
                "8.5.0": string;
            };
            dataSourcePluginsV2: {
                '9.3.1': (pluginName: string) => string;
                "8.5.0": (pluginName: string) => string;
            };
        };
        ConfirmModal: {
            delete: {
                '10.0.0': string;
                "8.5.0": string;
            };
        };
        AddDashboard: {
            url: {
                "8.5.0": string;
            };
            itemButton: {
                '9.5.0': (title: string) => string;
            };
            addNewPanel: {
                '11.1.0': string;
                '8.0.0': string;
                "8.5.0": string;
            };
            itemButtonAddViz: {
                "8.5.0": string;
            };
            addNewRow: {
                '11.1.0': string;
                "8.5.0": string;
            };
            addNewPanelLibrary: {
                '11.1.0': string;
                "8.5.0": string;
            };
            Settings: {
                Annotations: {
                    List: {
                        url: {
                            "8.5.0": string;
                        };
                    };
                    Edit: {
                        url: {
                            "8.5.0": (annotationIndex: string) => string;
                        };
                    };
                };
                Variables: {
                    List: {
                        url: {
                            '11.3.0': string;
                            "8.5.0": string;
                        };
                    };
                    Edit: {
                        url: {
                            '11.3.0': (editIndex: string) => string;
                            "8.5.0": (editIndex: string) => string;
                        };
                    };
                };
            };
        };
        Dashboard: {
            url: {
                "8.5.0": (uid: string) => string;
            };
            DashNav: {
                nav: {
                    "8.5.0": string;
                };
                navV2: {
                    "8.5.0": string;
                };
                publicDashboardTag: {
                    '9.1.0': string;
                };
                shareButton: {
                    '10.4.0': string;
                };
                scrollContainer: {
                    '11.1.0': string;
                };
                newShareButton: {
                    container: {
                        '11.1.0': string;
                    };
                    shareLink: {
                        '11.1.0': string;
                    };
                    arrowMenu: {
                        '11.1.0': string;
                    };
                    menu: {
                        container: {
                            '11.1.0': string;
                        };
                        shareInternally: {
                            '11.1.0': string;
                        };
                        shareExternally: {
                            '11.1.1': string;
                        };
                        shareSnapshot: {
                            '11.2.0': string;
                        };
                        inviteUser: {
                            '11.5.0': string;
                        };
                    };
                };
                NewExportButton: {
                    container: {
                        '11.2.0': string;
                    };
                    arrowMenu: {
                        '11.2.0': string;
                    };
                    Menu: {
                        container: {
                            '11.2.0': string;
                        };
                        exportAsJson: {
                            '11.2.0': string;
                        };
                    };
                };
                playlistControls: {
                    prev: {
                        '11.0.0': string;
                    };
                    stop: {
                        '11.0.0': string;
                    };
                    next: {
                        '11.0.0': string;
                    };
                };
            };
            Controls: {
                '11.1.0': string;
            };
            SubMenu: {
                submenu: {
                    "8.5.0": string;
                };
                submenuItem: {
                    "8.5.0": string;
                };
                submenuItemLabels: {
                    "8.5.0": (item: string) => string;
                };
                submenuItemValueDropDownValueLinkTexts: {
                    "8.5.0": (item: string) => string;
                };
                submenuItemValueDropDownDropDown: {
                    "8.5.0": string;
                };
                submenuItemValueDropDownOptionTexts: {
                    "8.5.0": (item: string) => string;
                };
                Annotations: {
                    annotationsWrapper: {
                        '10.0.0': string;
                    };
                    annotationLabel: {
                        '10.0.0': (label: string) => string;
                    };
                    annotationToggle: {
                        '10.0.0': (label: string) => string;
                    };
                };
            };
            Settings: {
                Actions: {
                    close: {
                        '9.5.0': string;
                    };
                };
                General: {
                    deleteDashBoard: {
                        '11.1.0': string;
                    };
                    sectionItems: {
                        "8.5.0": (item: string) => string;
                    };
                    saveDashBoard: {
                        "8.5.0": string;
                    };
                    saveAsDashBoard: {
                        "8.5.0": string;
                    };
                    title: {
                        '11.2.0': string;
                    };
                };
                Annotations: {
                    Edit: {
                        urlParams: {
                            "8.5.0": (annotationIndex: string) => string;
                        };
                    };
                    List: {
                        url: {
                            "8.5.0": (dashboardUid: string) => string;
                        };
                        addAnnotationCTAV2: {
                            "8.5.0": string;
                        };
                        annotations: {
                            '10.4.0': string;
                        };
                    };
                    Settings: {
                        name: {
                            '11.1.0': string;
                            "8.5.0": string;
                        };
                    };
                    NewAnnotation: {
                        panelFilterSelect: {
                            '10.0.0': string;
                        };
                        showInLabel: {
                            '11.1.0': string;
                        };
                        previewInDashboard: {
                            '10.0.0': string;
                        };
                        delete: {
                            '10.4.0': string;
                        };
                        apply: {
                            '10.4.0': string;
                        };
                        enable: {
                            '10.4.0': string;
                        };
                        hide: {
                            '10.4.0': string;
                        };
                    };
                };
                Variables: {
                    List: {
                        url: {
                            '11.3.0': (dashboardUid: string) => string;
                            "8.5.0": (dashboardUid: string) => string;
                        };
                        addVariableCTAV2: {
                            "8.5.0": string;
                        };
                        newButton: {
                            "8.5.0": string;
                        };
                        table: {
                            "8.5.0": string;
                        };
                        tableRowNameFields: {
                            "8.5.0": (variableName: string) => string;
                        };
                        tableRowDefinitionFields: {
                            '10.1.0': (variableName: string) => string;
                        };
                        tableRowArrowUpButtons: {
                            "8.5.0": (variableName: string) => string;
                        };
                        tableRowArrowDownButtons: {
                            "8.5.0": (variableName: string) => string;
                        };
                        tableRowDuplicateButtons: {
                            "8.5.0": (variableName: string) => string;
                        };
                        tableRowRemoveButtons: {
                            "8.5.0": (variableName: string) => string;
                        };
                    };
                    Edit: {
                        urlParams: {
                            '11.3.0': (editIndex: string) => string;
                            "8.5.0": (editIndex: string) => string;
                        };
                        General: {
                            headerLink: {
                                "8.5.0": string;
                            };
                            modeLabelNew: {
                                "8.5.0": string;
                            };
                            modeLabelEdit: {
                                "8.5.0": string;
                            };
                            generalNameInput: {
                                "8.5.0": string;
                            };
                            generalNameInputV2: {
                                "8.5.0": string;
                            };
                            generalTypeSelect: {
                                "8.5.0": string;
                            };
                            generalTypeSelectV2: {
                                "8.5.0": string;
                            };
                            generalLabelInput: {
                                "8.5.0": string;
                            };
                            generalLabelInputV2: {
                                "8.5.0": string;
                            };
                            generalHideSelect: {
                                "8.5.0": string;
                            };
                            generalHideSelectV2: {
                                "8.5.0": string;
                            };
                            selectionOptionsAllowCustomValueSwitch: {
                                "8.5.0": string;
                            };
                            selectionOptionsMultiSwitch: {
                                '10.4.0': string;
                                "8.5.0": string;
                            };
                            selectionOptionsIncludeAllSwitch: {
                                '10.4.0': string;
                                "8.5.0": string;
                            };
                            selectionOptionsCustomAllInput: {
                                '10.4.0': string;
                                "8.5.0": string;
                            };
                            previewOfValuesOption: {
                                '10.4.0': string;
                                "8.5.0": string;
                            };
                            submitButton: {
                                '10.4.0': string;
                                "8.5.0": string;
                            };
                            applyButton: {
                                '9.3.0': string;
                            };
                        };
                        QueryVariable: {
                            queryOptionsDataSourceSelect: {
                                '10.4.0': string;
                                '10.0.0': string;
                                "8.5.0": string;
                            };
                            queryOptionsRefreshSelect: {
                                "8.5.0": string;
                            };
                            queryOptionsRefreshSelectV2: {
                                "8.5.0": string;
                            };
                            queryOptionsRegExInput: {
                                "8.5.0": string;
                            };
                            queryOptionsRegExInputV2: {
                                "8.5.0": string;
                            };
                            queryOptionsSortSelect: {
                                "8.5.0": string;
                            };
                            queryOptionsSortSelectV2: {
                                "8.5.0": string;
                            };
                            queryOptionsQueryInput: {
                                '10.4.0': string;
                            };
                            valueGroupsTagsEnabledSwitch: {
                                "8.5.0": string;
                            };
                            valueGroupsTagsTagsQueryInput: {
                                "8.5.0": string;
                            };
                            valueGroupsTagsTagsValuesQueryInput: {
                                "8.5.0": string;
                            };
                        };
                        ConstantVariable: {
                            constantOptionsQueryInput: {
                                "8.5.0": string;
                            };
                            constantOptionsQueryInputV2: {
                                "8.5.0": string;
                            };
                        };
                        DatasourceVariable: {
                            datasourceSelect: {
                                "8.5.0": string;
                            };
                        };
                        TextBoxVariable: {
                            textBoxOptionsQueryInput: {
                                "8.5.0": string;
                            };
                            textBoxOptionsQueryInputV2: {
                                "8.5.0": string;
                            };
                        };
                        CustomVariable: {
                            customValueInput: {
                                "8.5.0": string;
                            };
                        };
                        IntervalVariable: {
                            intervalsValueInput: {
                                "8.5.0": string;
                            };
                            autoEnabledCheckbox: {
                                '10.4.0': string;
                            };
                            stepCountIntervalSelect: {
                                '10.4.0': string;
                            };
                            minIntervalInput: {
                                '10.4.0': string;
                            };
                        };
                        GroupByVariable: {
                            dataSourceSelect: {
                                '10.4.0': string;
                            };
                            infoText: {
                                '10.4.0': string;
                            };
                            modeToggle: {
                                '10.4.0': string;
                            };
                        };
                        AdHocFiltersVariable: {
                            datasourceSelect: {
                                '10.4.0': string;
                            };
                            infoText: {
                                '10.4.0': string;
                            };
                            modeToggle: {
                                '11.0.0': string;
                            };
                        };
                    };
                };
            };
            Annotations: {
                marker: {
                    '10.0.0': string;
                };
            };
            Rows: {
                Repeated: {
                    ConfigSection: {
                        warningMessage: {
                            '10.2.0': string;
                        };
                    };
                };
            };
        };
        Dashboards: {
            url: {
                "8.5.0": string;
            };
            dashboards: {
                '10.2.0': (title: string) => string;
            };
        };
        SaveDashboardAsModal: {
            newName: {
                '10.2.0': string;
            };
            save: {
                '10.2.0': string;
            };
        };
        SaveDashboardModal: {
            save: {
                '10.2.0': string;
            };
            saveVariables: {
                '10.2.0': string;
            };
            saveTimerange: {
                '10.2.0': string;
            };
            saveRefresh: {
                '11.1.0': string;
            };
        };
        SharePanelModal: {
            linkToRenderedImage: {
                "8.5.0": string;
            };
        };
        ShareDashboardModal: {
            PublicDashboard: {
                WillBePublicCheckbox: {
                    '9.1.0': string;
                };
                LimitedDSCheckbox: {
                    '9.1.0': string;
                };
                CostIncreaseCheckbox: {
                    '9.1.0': string;
                };
                PauseSwitch: {
                    '9.5.0': string;
                };
                EnableAnnotationsSwitch: {
                    '9.3.0': string;
                };
                CreateButton: {
                    '9.5.0': string;
                };
                DeleteButton: {
                    '9.3.0': string;
                };
                CopyUrlInput: {
                    '9.1.0': string;
                };
                CopyUrlButton: {
                    '9.1.0': string;
                };
                SettingsDropdown: {
                    '10.1.0': string;
                };
                TemplateVariablesWarningAlert: {
                    '9.1.0': string;
                };
                UnsupportedDataSourcesWarningAlert: {
                    '9.5.0': string;
                };
                NoUpsertPermissionsWarningAlert: {
                    '9.5.0': string;
                };
                EnableTimeRangeSwitch: {
                    '9.4.0': string;
                };
                EmailSharingConfiguration: {
                    Container: {
                        '9.5.0': string;
                    };
                    ShareType: {
                        '9.5.0': string;
                    };
                    EmailSharingInput: {
                        '9.5.0': string;
                    };
                    EmailSharingInviteButton: {
                        '9.5.0': string;
                    };
                    EmailSharingList: {
                        '9.5.0': string;
                    };
                    DeleteEmail: {
                        '9.5.0': string;
                    };
                    ReshareLink: {
                        '9.5.0': string;
                    };
                };
            };
            SnapshotScene: {
                url: {
                    '11.1.0': (key: string) => string;
                };
                PublishSnapshot: {
                    '11.1.0': string;
                };
                CopyUrlButton: {
                    '11.1.0': string;
                };
                CopyUrlInput: {
                    '11.1.0': string;
                };
            };
        };
        ShareDashboardDrawer: {
            ShareInternally: {
                container: {
                    '11.3.0': string;
                };
                lockTimeRangeSwitch: {
                    '11.3.0': string;
                };
                shortenUrlSwitch: {
                    '11.3.0': string;
                };
                copyUrlButton: {
                    '11.3.0': string;
                };
                SharePanel: {
                    preview: {
                        '11.5.0': string;
                    };
                    widthInput: {
                        '11.5.0': string;
                    };
                    heightInput: {
                        '11.5.0': string;
                    };
                    scaleFactorInput: {
                        '11.5.0': string;
                    };
                    generateImageButton: {
                        '11.5.0': string;
                    };
                    downloadImageButton: {
                        '11.5.0': string;
                    };
                };
            };
            ShareExternally: {
                container: {
                    '11.3.0': string;
                };
                publicAlert: {
                    '11.3.0': string;
                };
                emailSharingAlert: {
                    '11.3.0': string;
                };
                shareTypeSelect: {
                    '11.3.0': string;
                };
                Creation: {
                    PublicShare: {
                        createButton: {
                            '11.3.0': string;
                        };
                        cancelButton: {
                            '11.3.0': string;
                        };
                    };
                    EmailShare: {
                        createButton: {
                            '11.3.0': string;
                        };
                        cancelButton: {
                            '11.3.0': string;
                        };
                    };
                    willBePublicCheckbox: {
                        '11.3.0': string;
                    };
                };
                Configuration: {
                    enableTimeRangeSwitch: {
                        '11.3.0': string;
                    };
                    enableAnnotationsSwitch: {
                        '11.3.0': string;
                    };
                    copyUrlButton: {
                        '11.3.0': string;
                    };
                    revokeAccessButton: {
                        '11.3.0': string;
                    };
                    toggleAccessButton: {
                        '11.3.0': string;
                    };
                };
            };
            ShareSnapshot: {
                url: {
                    '11.3.0': (key: string) => string;
                };
                container: {
                    '11.3.0': string;
                };
                publishSnapshot: {
                    '11.3.0': string;
                };
                copyUrlButton: {
                    '11.3.0': string;
                };
            };
        };
        ExportDashboardDrawer: {
            ExportAsJson: {
                container: {
                    '11.3.0': string;
                };
                codeEditor: {
                    '11.3.0': string;
                };
                exportExternallyToggle: {
                    '11.3.0': string;
                };
                saveToFileButton: {
                    '11.3.0': string;
                };
                copyToClipboardButton: {
                    '11.3.0': string;
                };
                cancelButton: {
                    '11.3.0': string;
                };
            };
        };
        PublicDashboard: {
            page: {
                '9.5.0': string;
            };
            NotAvailable: {
                container: {
                    '9.5.0': string;
                };
                title: {
                    '9.5.0': string;
                };
                pausedDescription: {
                    '9.5.0': string;
                };
            };
            footer: {
                '11.0.0': string;
            };
        };
        PublicDashboardScene: {
            loadingPage: {
                '11.0.0': string;
            };
            page: {
                '11.0.0': string;
            };
            controls: {
                '11.0.0': string;
            };
        };
        RequestViewAccess: {
            form: {
                '9.5.0': string;
            };
            recipientInput: {
                '9.5.0': string;
            };
            submitButton: {
                '9.5.0': string;
            };
        };
        PublicDashboardConfirmAccess: {
            submitButton: {
                '10.2.0': string;
            };
        };
        Explore: {
            url: {
                "8.5.0": string;
            };
            General: {
                container: {
                    "8.5.0": string;
                };
                graph: {
                    "8.5.0": string;
                };
                table: {
                    "8.5.0": string;
                };
                scrollView: {
                    '9.0.0': string;
                };
            };
            QueryHistory: {
                container: {
                    '11.1.0': string;
                };
            };
        };
        SoloPanel: {
            url: {
                "8.5.0": (page: string) => string;
            };
        };
        PluginsList: {
            page: {
                "8.5.0": string;
            };
            list: {
                "8.5.0": string;
            };
            listItem: {
                "8.5.0": string;
            };
            signatureErrorNotice: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
        PluginPage: {
            page: {
                "8.5.0": string;
            };
            signatureInfo: {
                '10.3.0': string;
                "8.5.0": string;
            };
            disabledInfo: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
        PlaylistForm: {
            name: {
                "8.5.0": string;
            };
            interval: {
                "8.5.0": string;
            };
            itemDelete: {
                '10.2.0': string;
            };
        };
        BrowseDashboards: {
            table: {
                body: {
                    '10.2.0': string;
                };
                row: {
                    '10.2.0': (name: string) => string;
                };
                checkbox: {
                    '10.0.0': (uid: string) => string;
                };
            };
            NewFolderForm: {
                form: {
                    '10.2.0': string;
                };
                nameInput: {
                    '10.2.0': string;
                };
                createButton: {
                    '10.2.0': string;
                };
            };
        };
        Search: {
            url: {
                '9.3.0': string;
            };
            FolderView: {
                url: {
                    '9.3.0': string;
                };
            };
        };
        PublicDashboards: {
            ListItem: {
                linkButton: {
                    '9.3.0': string;
                };
                configButton: {
                    '9.3.0': string;
                };
                trashcanButton: {
                    '9.3.0': string;
                };
                pauseSwitch: {
                    '10.1.0': string;
                };
            };
        };
        UserListPage: {
            tabs: {
                allUsers: {
                    '10.0.0': string;
                };
                orgUsers: {
                    '10.0.0': string;
                };
                anonUserDevices: {
                    '10.2.3': string;
                };
                publicDashboardsUsers: {
                    '10.0.0': string;
                };
                users: {
                    '10.0.0': string;
                };
            };
            org: {
                url: {
                    '10.2.0': string;
                    '9.5.0': string;
                };
            };
            admin: {
                url: {
                    '9.5.0': string;
                };
            };
            publicDashboards: {
                container: {
                    '11.1.0': string;
                };
            };
            UserListAdminPage: {
                container: {
                    '10.0.0': string;
                };
            };
            UsersListPage: {
                container: {
                    '10.0.0': string;
                };
            };
            UserAnonListPage: {
                container: {
                    '10.4.0': string;
                };
            };
            UsersListPublicDashboardsPage: {
                container: {
                    '10.0.0': string;
                };
                DashboardsListModal: {
                    listItem: {
                        '10.0.0': (uid: string) => string;
                    };
                };
            };
        };
        ProfilePage: {
            url: {
                '10.2.0': string;
            };
        };
        Plugin: {
            url: {
                "8.5.0": (pluginId: string) => string;
            };
        };
    }>;
    components: SelectorsOf<{
        RadioButton: {
            container: {
                '10.2.3': string;
            };
        };
        Breadcrumbs: {
            breadcrumb: {
                '9.4.0': (title: string) => string;
            };
        };
        TimePicker: {
            openButton: {
                "8.5.0": string;
            };
            overlayContent: {
                '10.2.3': string;
            };
            fromField: {
                '10.2.3': string;
                "8.5.0": string;
            };
            toField: {
                '10.2.3': string;
                "8.5.0": string;
            };
            applyTimeRange: {
                "8.5.0": string;
            };
            copyTimeRange: {
                '10.4.0': string;
            };
            pasteTimeRange: {
                '10.4.0': string;
            };
            calendar: {
                label: {
                    '10.2.3': string;
                    "8.5.0": string;
                };
                openButton: {
                    '10.2.3': string;
                    "8.5.0": string;
                };
                closeButton: {
                    '10.2.3': string;
                    "8.5.0": string;
                };
            };
            absoluteTimeRangeTitle: {
                "8.5.0": string;
            };
        };
        DataSourcePermissions: {
            form: {
                '9.5.0': () => string;
            };
            roleType: {
                '9.5.0': string;
            };
            rolePicker: {
                '9.5.0': string;
            };
            permissionLevel: {
                '9.5.0': string;
            };
        };
        DateTimePicker: {
            input: {
                '10.2.3': string;
            };
        };
        DataSource: {
            TestData: {
                QueryTab: {
                    scenarioSelectContainer: {
                        "8.5.0": string;
                    };
                    scenarioSelect: {
                        "8.5.0": string;
                    };
                    max: {
                        "8.5.0": string;
                    };
                    min: {
                        "8.5.0": string;
                    };
                    noise: {
                        "8.5.0": string;
                    };
                    seriesCount: {
                        "8.5.0": string;
                    };
                    spread: {
                        "8.5.0": string;
                    };
                    startValue: {
                        "8.5.0": string;
                    };
                    drop: {
                        "8.5.0": string;
                    };
                };
            };
            DataSourceHttpSettings: {
                urlInput: {
                    '10.4.0': string;
                    "8.5.0": string;
                };
            };
            Jaeger: {
                traceIDInput: {
                    "8.5.0": string;
                };
            };
            Prometheus: {
                configPage: {
                    connectionSettings: {
                        "8.5.0": string;
                    };
                    manageAlerts: {
                        '10.4.0': string;
                    };
                    scrapeInterval: {
                        '10.4.0': string;
                    };
                    queryTimeout: {
                        '10.4.0': string;
                    };
                    defaultEditor: {
                        '10.4.0': string;
                    };
                    disableMetricLookup: {
                        '10.4.0': string;
                    };
                    prometheusType: {
                        '10.4.0': string;
                    };
                    prometheusVersion: {
                        '10.4.0': string;
                    };
                    cacheLevel: {
                        '10.4.0': string;
                    };
                    incrementalQuerying: {
                        '10.4.0': string;
                    };
                    queryOverlapWindow: {
                        '10.4.0': string;
                    };
                    disableRecordingRules: {
                        '10.4.0': string;
                    };
                    customQueryParameters: {
                        '10.4.0': string;
                    };
                    httpMethod: {
                        '10.4.0': string;
                    };
                    exemplarsAddButton: {
                        '10.3.0': string;
                        "8.5.0": string;
                    };
                    internalLinkSwitch: {
                        '10.3.0': string;
                        "8.5.0": string;
                    };
                    codeModeMetricNamesSuggestionLimit: {
                        '11.1.0': string;
                    };
                };
                queryEditor: {
                    explain: {
                        '10.4.0': string;
                    };
                    editorToggle: {
                        '10.4.0': string;
                    };
                    options: {
                        '10.4.0': string;
                    };
                    legend: {
                        '10.4.0': string;
                    };
                    format: {
                        '10.4.0': string;
                    };
                    step: {
                        '10.4.0': string;
                    };
                    type: {
                        '10.4.0': string;
                    };
                    exemplars: {
                        '10.4.0': string;
                    };
                    builder: {
                        metricSelect: {
                            '10.4.0': string;
                        };
                        hints: {
                            '10.4.0': string;
                        };
                        metricsExplorer: {
                            '10.4.0': string;
                        };
                        queryAdvisor: {
                            '10.4.0': string;
                        };
                    };
                    code: {
                        queryField: {
                            '10.4.0': string;
                        };
                        metricsCountInfo: {
                            '11.1.0': string;
                        };
                        metricsBrowser: {
                            openButton: {
                                '10.4.0': string;
                            };
                            selectMetric: {
                                '10.4.0': string;
                            };
                            seriesLimit: {
                                '10.3.1': string;
                            };
                            metricList: {
                                '10.4.0': string;
                            };
                            labelNamesFilter: {
                                '10.4.0': string;
                            };
                            labelValuesFilter: {
                                '10.4.0': string;
                            };
                            useQuery: {
                                '10.4.0': string;
                            };
                            useAsRateQuery: {
                                '10.4.0': string;
                            };
                            validateSelector: {
                                '10.4.0': string;
                            };
                            clear: {
                                '10.4.0': string;
                            };
                        };
                    };
                };
                exemplarMarker: {
                    '10.3.0': string;
                    "8.5.0": string;
                };
                variableQueryEditor: {
                    queryType: {
                        '10.4.0': string;
                    };
                    labelnames: {
                        metricRegex: {
                            '10.4.0': string;
                        };
                    };
                    labelValues: {
                        labelSelect: {
                            '10.4.0': string;
                        };
                    };
                    metricNames: {
                        metricRegex: {
                            '10.4.0': string;
                        };
                    };
                    varQueryResult: {
                        '10.4.0': string;
                    };
                    seriesQuery: {
                        '10.4.0': string;
                    };
                    classicQuery: {
                        '10.4.0': string;
                    };
                };
                annotations: {
                    minStep: {
                        '10.4.0': string;
                    };
                    title: {
                        '10.4.0': string;
                    };
                    tags: {
                        '10.4.0': string;
                    };
                    text: {
                        '10.4.0': string;
                    };
                    seriesValueAsTimestamp: {
                        '10.4.0': string;
                    };
                };
            };
        };
        Menu: {
            MenuComponent: {
                "8.5.0": (title: string) => string;
            };
            MenuGroup: {
                "8.5.0": (title: string) => string;
            };
            MenuItem: {
                "8.5.0": (title: string) => string;
            };
            SubMenu: {
                container: {
                    '10.3.0': string;
                    "8.5.0": string;
                };
                icon: {
                    '10.3.0': string;
                    "8.5.0": string;
                };
            };
        };
        Panels: {
            Panel: {
                title: {
                    "8.5.0": (title: string) => string;
                };
                content: {
                    '11.1.0': string;
                };
                headerItems: {
                    '10.2.0': (item: string) => string;
                };
                menuItems: {
                    '9.5.0': (item: string) => string;
                };
                menu: {
                    '9.5.0': (title: string) => string;
                };
                containerByTitle: {
                    "8.5.0": (title: string) => string;
                };
                headerCornerInfo: {
                    "8.5.0": (mode: string) => string;
                };
                status: {
                    '10.2.0': (status: string) => string;
                    "8.5.0": (_: string) => string;
                };
                loadingBar: {
                    '10.0.0': () => string;
                };
                HoverWidget: {
                    container: {
                        '10.1.0': string;
                        "8.5.0": string;
                    };
                    dragIcon: {
                        '10.0.0': string;
                    };
                };
                PanelDataErrorMessage: {
                    '10.4.0': string;
                };
            };
            Visualization: {
                Graph: {
                    container: {
                        '9.5.0': string;
                    };
                    VisualizationTab: {
                        legendSection: {
                            "8.5.0": string;
                        };
                    };
                    Legend: {
                        legendItemAlias: {
                            "8.5.0": (name: string) => string;
                        };
                        showLegendSwitch: {
                            "8.5.0": string;
                        };
                    };
                    xAxis: {
                        labels: {
                            "8.5.0": () => string;
                        };
                    };
                };
                BarGauge: {
                    valueV2: {
                        "8.5.0": string;
                    };
                };
                PieChart: {
                    svgSlice: {
                        '10.3.0': string;
                    };
                };
                Text: {
                    container: {
                        "8.5.0": () => string;
                    };
                };
                Table: {
                    header: {
                        "8.5.0": string;
                    };
                    footer: {
                        "8.5.0": string;
                    };
                    body: {
                        '10.2.0': string;
                    };
                };
            };
        };
        VizLegend: {
            seriesName: {
                '10.3.0': (name: string) => string;
            };
        };
        Drawer: {
            General: {
                title: {
                    "8.5.0": (title: string) => string;
                };
                expand: {
                    "8.5.0": string;
                };
                contract: {
                    "8.5.0": string;
                };
                close: {
                    '10.3.0': string;
                    "8.5.0": string;
                };
                rcContentWrapper: {
                    '9.4.0': () => string;
                };
                subtitle: {
                    '10.4.0': string;
                };
            };
            DashboardSaveDrawer: {
                saveButton: {
                    '11.1.0': string;
                };
                saveAsButton: {
                    '11.1.0': string;
                };
                saveAsTitleInput: {
                    '11.1.0': string;
                };
            };
        };
        PanelEditor: {
            General: {
                content: {
                    '11.1.0': string;
                    '8.0.0': string;
                };
            };
            OptionsPane: {
                content: {
                    '11.1.0': string;
                    "8.5.0": string;
                };
                select: {
                    "8.5.0": string;
                };
                fieldLabel: {
                    "8.5.0": (type: string) => string;
                };
                fieldInput: {
                    '11.0.0': (title: string) => string;
                };
            };
            DataPane: {
                content: {
                    '11.1.0': string;
                    "8.5.0": string;
                };
            };
            applyButton: {
                '9.2.0': string;
                '9.1.0': string;
                '8.0.0': string;
            };
            toggleVizPicker: {
                '10.0.0': string;
                '8.0.0': string;
            };
            toggleVizOptions: {
                '10.1.0': string;
                "8.5.0": string;
            };
            toggleTableView: {
                '11.1.0': string;
                "8.5.0": string;
            };
            showZoomField: {
                '10.2.0': string;
            };
            showAttributionField: {
                '10.2.0': string;
            };
            showScaleField: {
                '10.2.0': string;
            };
            showMeasureField: {
                '10.2.0': string;
            };
            showDebugField: {
                '10.2.0': string;
            };
            measureButton: {
                '9.2.0': string;
            };
        };
        PanelInspector: {
            Data: {
                content: {
                    "8.5.0": string;
                };
            };
            Stats: {
                content: {
                    "8.5.0": string;
                };
            };
            Json: {
                content: {
                    '11.1.0': string;
                    "8.5.0": string;
                };
            };
            Query: {
                content: {
                    "8.5.0": string;
                };
                refreshButton: {
                    "8.5.0": string;
                };
                jsonObjectKeys: {
                    "8.5.0": () => string;
                };
            };
        };
        Tab: {
            title: {
                '11.2.0': (title: string) => string;
            };
            active: {
                "8.5.0": () => string;
            };
        };
        RefreshPicker: {
            runButtonV2: {
                "8.5.0": string;
            };
            intervalButtonV2: {
                "8.5.0": string;
            };
        };
        QueryTab: {
            content: {
                "8.5.0": string;
            };
            queryInspectorButton: {
                "8.5.0": string;
            };
            queryHistoryButton: {
                '10.2.0': string;
                "8.5.0": string;
            };
            addQuery: {
                '10.2.0': string;
                "8.5.0": string;
            };
            queryGroupTopSection: {
                '11.2.0': string;
            };
            addExpression: {
                '11.2.0': string;
            };
        };
        QueryHistory: {
            queryText: {
                '9.0.0': string;
            };
        };
        QueryEditorRows: {
            rows: {
                "8.5.0": string;
            };
        };
        QueryEditorRow: {
            actionButton: {
                '10.4.0': (title: string) => string;
            };
            title: {
                "8.5.0": (refId: string) => string;
            };
            container: {
                "8.5.0": (refId: string) => string;
            };
        };
        AlertTab: {
            content: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        AlertRules: {
            groupToggle: {
                '11.0.0': string;
            };
            toggle: {
                '11.0.0': string;
            };
            expandedContent: {
                '11.0.0': string;
            };
            previewButton: {
                '11.1.0': string;
            };
            ruleNameField: {
                '11.1.0': string;
            };
            newFolderButton: {
                '11.1.0': string;
            };
            newFolderNameField: {
                '11.1.0': string;
            };
            newFolderNameCreateButton: {
                '11.1.0': string;
            };
            newEvaluationGroupButton: {
                '11.1.0': string;
            };
            newEvaluationGroupName: {
                '11.1.0': string;
            };
            newEvaluationGroupInterval: {
                '11.1.0': string;
            };
            newEvaluationGroupCreate: {
                '11.1.0': string;
            };
        };
        Alert: {
            alertV2: {
                "8.5.0": (severity: string) => string;
            };
        };
        TransformTab: {
            content: {
                '10.1.0': string;
                "8.5.0": string;
            };
            newTransform: {
                '10.1.0': (name: string) => string;
            };
            transformationEditor: {
                '10.1.0': (name: string) => string;
            };
            transformationEditorDebugger: {
                '10.1.0': (name: string) => string;
            };
        };
        Transforms: {
            card: {
                '10.1.0': (name: string) => string;
            };
            disableTransformationButton: {
                '10.4.0': string;
            };
            Reduce: {
                modeLabel: {
                    '10.2.3': string;
                    "8.5.0": string;
                };
                calculationsLabel: {
                    '10.2.3': string;
                    "8.5.0": string;
                };
            };
            SpatialOperations: {
                actionLabel: {
                    '9.1.2': string;
                };
                locationLabel: {
                    '10.2.0': string;
                };
                location: {
                    autoOption: {
                        '9.1.2': string;
                    };
                    coords: {
                        option: {
                            '9.1.2': string;
                        };
                        latitudeFieldLabel: {
                            '9.1.2': string;
                        };
                        longitudeFieldLabel: {
                            '9.1.2': string;
                        };
                    };
                    geohash: {
                        option: {
                            '9.1.2': string;
                        };
                        geohashFieldLabel: {
                            '9.1.2': string;
                        };
                    };
                    lookup: {
                        option: {
                            '9.1.2': string;
                        };
                        lookupFieldLabel: {
                            '9.1.2': string;
                        };
                        gazetteerFieldLabel: {
                            '9.1.2': string;
                        };
                    };
                };
            };
            searchInput: {
                '10.2.3': string;
                "8.5.0": string;
            };
            noTransformationsMessage: {
                '10.2.3': string;
            };
            addTransformationButton: {
                '10.1.0': string;
                "8.5.0": string;
            };
            removeAllTransformationsButton: {
                '10.4.0': string;
            };
        };
        NavBar: {
            Configuration: {
                button: {
                    '9.5.0': string;
                };
            };
            Toggle: {
                button: {
                    '10.2.3': string;
                    "8.5.0": string;
                };
            };
            Reporting: {
                button: {
                    '9.5.0': string;
                };
            };
        };
        NavMenu: {
            Menu: {
                '10.2.3': string;
            };
            item: {
                '9.5.0': string;
            };
        };
        NavToolbar: {
            container: {
                '9.4.0': string;
            };
            commandPaletteTrigger: {
                '11.5.0': string;
            };
            shareDashboard: {
                '11.1.0': string;
            };
            markAsFavorite: {
                '11.1.0': string;
            };
            editDashboard: {
                editButton: {
                    '11.1.0': string;
                };
                saveButton: {
                    '11.1.0': string;
                };
                exitButton: {
                    '11.1.0': string;
                };
                settingsButton: {
                    '11.1.0': string;
                };
                addRowButton: {
                    '11.1.0': string;
                };
                addLibraryPanelButton: {
                    '11.1.0': string;
                };
                addVisualizationButton: {
                    '11.1.0': string;
                };
                pastePanelButton: {
                    '11.1.0': string;
                };
                discardChangesButton: {
                    '11.1.0': string;
                };
                discardLibraryPanelButton: {
                    '11.1.0': string;
                };
                unlinkLibraryPanelButton: {
                    '11.1.0': string;
                };
                saveLibraryPanelButton: {
                    '11.1.0': string;
                };
                backToDashboardButton: {
                    '11.1.0': string;
                };
            };
        };
        PageToolbar: {
            container: {
                "8.5.0": () => string;
            };
            item: {
                "8.5.0": (tooltip: string) => string;
            };
            itemButton: {
                '9.5.0': (title: string) => string;
            };
        };
        QueryEditorToolbarItem: {
            button: {
                "8.5.0": (title: string) => string;
            };
        };
        BackButton: {
            backArrow: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
        OptionsGroup: {
            group: {
                '11.1.0': (title?: string) => string;
                "8.5.0": (title?: string) => string;
            };
            toggle: {
                '11.1.0': (title?: string) => string;
                "8.5.0": (title?: string) => string;
            };
        };
        PluginVisualization: {
            item: {
                "8.5.0": (title: string) => string;
            };
            current: {
                "8.5.0": () => string;
            };
        };
        Select: {
            menu: {
                '11.5.0': string;
                "8.5.0": string;
            };
            option: {
                '11.1.0': string;
                "8.5.0": string;
            };
            toggleAllOptions: {
                '11.3.0': string;
            };
            input: {
                "8.5.0": () => string;
            };
            singleValue: {
                "8.5.0": () => string;
            };
        };
        FieldConfigEditor: {
            content: {
                "8.5.0": string;
            };
        };
        OverridesConfigEditor: {
            content: {
                "8.5.0": string;
            };
        };
        FolderPicker: {
            containerV2: {
                "8.5.0": string;
            };
            input: {
                '10.4.0': string;
            };
        };
        ReadonlyFolderPicker: {
            container: {
                "8.5.0": string;
            };
        };
        DataSourcePicker: {
            container: {
                '10.0.0': string;
                '8.0.0': string;
            };
            inputV2: {
                '10.1.0': string;
                "8.5.0": string;
            };
            dataSourceList: {
                '10.4.0': string;
            };
            advancedModal: {
                dataSourceList: {
                    '10.4.0': string;
                };
                builtInDataSourceList: {
                    '10.4.0': string;
                };
            };
        };
        TimeZonePicker: {
            containerV2: {
                "8.5.0": string;
            };
            changeTimeSettingsButton: {
                '11.0.0': string;
            };
        };
        WeekStartPicker: {
            containerV2: {
                "8.5.0": string;
            };
            placeholder: {
                "8.5.0": string;
            };
        };
        TraceViewer: {
            spanBar: {
                '9.0.0': string;
            };
        };
        QueryField: {
            container: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
        QueryBuilder: {
            queryPatterns: {
                '10.3.0': string;
                "8.5.0": string;
            };
            labelSelect: {
                '10.3.0': string;
                "8.5.0": string;
            };
            inputSelect: {
                '11.1.0': string;
            };
            valueSelect: {
                '10.3.0': string;
                "8.5.0": string;
            };
            matchOperatorSelect: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
        ValuePicker: {
            button: {
                '10.3.0': (name: string) => string;
            };
            select: {
                '10.3.0': (name: string) => string;
            };
        };
        Search: {
            sectionV2: {
                "8.5.0": string;
            };
            itemsV2: {
                "8.5.0": string;
            };
            cards: {
                "8.5.0": string;
            };
            collapseFolder: {
                "8.5.0": (sectionId: string) => string;
            };
            expandFolder: {
                "8.5.0": (sectionId: string) => string;
            };
            dashboardItem: {
                "8.5.0": (item: string) => string;
            };
            dashboardCard: {
                "8.5.0": (item: string) => string;
            };
            folderHeader: {
                '9.3.0': (folderName: string) => string;
            };
            folderContent: {
                '9.3.0': (folderName: string) => string;
            };
            dashboardItems: {
                "8.5.0": string;
            };
        };
        DashboardLinks: {
            container: {
                "8.5.0": string;
            };
            dropDown: {
                "8.5.0": string;
            };
            link: {
                "8.5.0": string;
            };
        };
        LoadingIndicator: {
            icon: {
                '10.4.0': string;
                "8.5.0": string;
            };
        };
        CallToActionCard: {
            buttonV2: {
                "8.5.0": (name: string) => string;
            };
        };
        DataLinksContextMenu: {
            singleLink: {
                '10.3.0': string;
                "8.5.0": string;
            };
        };
        CodeEditor: {
            container: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        ReactMonacoEditor: {
            editorLazy: {
                '11.1.0': string;
            };
        };
        DashboardImportPage: {
            textarea: {
                "8.5.0": string;
            };
            submit: {
                "8.5.0": string;
            };
        };
        ImportDashboardForm: {
            name: {
                "8.5.0": string;
            };
            submit: {
                "8.5.0": string;
            };
        };
        PanelAlertTabContent: {
            content: {
                '10.2.3': string;
                "8.5.0": string;
            };
        };
        VisualizationPreview: {
            card: {
                "8.5.0": (name: string) => string;
            };
        };
        ColorSwatch: {
            name: {
                "8.5.0": string;
            };
        };
        DashboardRow: {
            title: {
                "8.5.0": (title: string) => string;
            };
        };
        UserProfile: {
            profileSaveButton: {
                "8.5.0": string;
            };
            preferencesSaveButton: {
                "8.5.0": string;
            };
            orgsTable: {
                "8.5.0": string;
            };
            sessionsTable: {
                "8.5.0": string;
            };
            extensionPointTabs: {
                '10.2.3': string;
            };
            extensionPointTab: {
                '10.2.3': (tabId: string) => string;
            };
        };
        FileUpload: {
            inputField: {
                '9.0.0': string;
            };
            fileNameSpan: {
                '9.0.0': string;
            };
        };
        DebugOverlay: {
            wrapper: {
                '9.2.0': string;
            };
        };
        OrgRolePicker: {
            input: {
                '9.5.0': string;
            };
        };
        AnalyticsToolbarButton: {
            button: {
                '9.5.0': string;
            };
        };
        Variables: {
            variableOption: {
                '9.5.0': string;
            };
            variableLinkWrapper: {
                '11.1.1': string;
            };
        };
        Annotations: {
            annotationsTypeInput: {
                '11.1.0': string;
                "8.5.0": string;
            };
            annotationsChoosePanelInput: {
                '11.1.0': string;
                "8.5.0": string;
            };
            editor: {
                testButton: {
                    '11.0.0': string;
                };
                resultContainer: {
                    '11.0.0': string;
                };
            };
        };
        Tooltip: {
            container: {
                '10.2.0': string;
            };
        };
        ReturnToPrevious: {
            buttonGroup: {
                '11.0.0': string;
            };
            backButton: {
                '11.0.0': string;
            };
            dismissButton: {
                '11.0.0': string;
            };
        };
        SQLQueryEditor: {
            selectColumn: {
                '11.0.0': string;
            };
            selectColumnInput: {
                '11.0.0': string;
            };
            selectFunctionParameter: {
                '11.0.0': (name: string) => string;
            };
            selectAggregation: {
                '11.0.0': string;
            };
            selectAggregationInput: {
                '11.0.0': string;
            };
            selectAlias: {
                '11.0.0': string;
            };
            selectAliasInput: {
                '11.0.0': string;
            };
            selectInputParameter: {
                '11.0.0': string;
            };
            filterConjunction: {
                '11.0.0': string;
            };
            filterField: {
                '11.0.0': string;
            };
            filterOperator: {
                '11.0.0': string;
            };
            headerTableSelector: {
                '11.0.0': string;
            };
            headerFilterSwitch: {
                '11.0.0': string;
            };
            headerGroupSwitch: {
                '11.0.0': string;
            };
            headerOrderSwitch: {
                '11.0.0': string;
            };
            headerPreviewSwitch: {
                '11.0.0': string;
            };
        };
        EntityNotFound: {
            container: {
                '11.2.0': string;
            };
        };
        Portal: {
            container: {
                '11.5.0': string;
            };
        };
    }>;
};

export { Components, type CssSelector, type E2ESelectors, type FunctionSelector, type FunctionSelectorTwoArgs, Pages, type Selectors, type SelectorsOf, type StringSelector, type UrlSelector, type VersionedComponents, type VersionedCssSelector, type VersionedFunctionSelector1, type VersionedFunctionSelector2, type VersionedPages, type VersionedSelectorGroup, type VersionedSelectors, type VersionedStringSelector, type VersionedUrlSelector, resolveSelectors, selectors, versionedComponents, versionedPages };
