// Copyright 2021 Contributors to the Parsec project.
// SPDX-License-Identifier: Apache-2.0
use crate::{
    handles::KeyHandle,
    structures::{Digest, HashcheckTicket, Signature, SignatureScheme, VerifiedTicket},
    tss2_esys::{Esys_Sign, Esys_VerifySignature},
    Context, Result, ReturnCode,
};
use log::error;
use std::convert::{TryFrom, TryInto};
use std::ptr::null_mut;

impl Context {
    /// Verify if a signature was generated by signing a given digest with a key in the TPM.
    pub fn verify_signature(
        &mut self,
        key_handle: KeyHandle,
        digest: Digest,
        signature: Signature,
    ) -> Result<VerifiedTicket> {
        let mut validation_ptr = null_mut();
        ReturnCode::ensure_success(
            unsafe {
                Esys_VerifySignature(
                    self.mut_context(),
                    key_handle.into(),
                    self.optional_session_1(),
                    self.optional_session_2(),
                    self.optional_session_3(),
                    &digest.into(),
                    &signature.try_into()?,
                    &mut validation_ptr,
                )
            },
            |ret| {
                error!("Error when verifying signature: {:#010X}", ret);
            },
        )?;
        VerifiedTicket::try_from(Context::ffi_data_to_owned(validation_ptr))
    }

    /// Sign a digest with a key present in the TPM and return the signature.
    ///
    /// # Details
    /// For signatures using a restricted key, a hashcheck must be provided. For unrestricted keys, this may be None.
    ///
    /// # Parameters
    /// `key_handle` - Handle to the key be used for signing.
    /// `digest`     - The digest that is going to be signed.
    /// `scheme`     - The scheme to use if the scheme for the key referenced by the key handle is null.
    /// `validation` - An optional [HashcheckTicket] that proof that the digest was created by the TPM.
    ///                N.B. None will be treated as a "Null ticket".
    /// # Example
    ///
    /// ```rust
    /// # use tss_esapi::{Context, TctiNameConf,
    /// #    interface_types::{
    /// #        algorithm::{HashingAlgorithm, RsaSchemeAlgorithm},
    /// #        key_bits::RsaKeyBits,
    /// #        resource_handles::Hierarchy,
    /// #    },
    /// #    structures::{RsaScheme, RsaExponent},
    /// #    utils::create_unrestricted_signing_rsa_public
    /// # };
    /// use tss_esapi::structures::SignatureScheme;
    /// # let mut context =
    /// #     Context::new(
    /// #         TctiNameConf::from_environment_variable().expect("Failed to get TCTI"),
    /// #     ).expect("Failed to create Context");
    /// # let signing_key_pub = create_unrestricted_signing_rsa_public(
    /// #         RsaScheme::create(RsaSchemeAlgorithm::RsaSsa, Some(HashingAlgorithm::Sha256))
    /// #         .expect("Failed to create RSA scheme"),
    /// #     RsaKeyBits::Rsa2048,
    /// #     RsaExponent::default(),
    /// # )
    /// # .expect("Failed to create an unrestricted signing rsa public structure");
    /// # let unrestricted_signing_key_handle = context
    /// #     .execute_with_nullauth_session(|ctx| {
    /// #         ctx.create_primary(Hierarchy::Owner, signing_key_pub, None, None, None, None)
    /// #     })
    /// #     .unwrap()
    /// #     .key_handle;
    /// # let digest = context.get_random(32).unwrap();
    /// let signature = context.execute_with_nullauth_session(|ctx| {
    ///     ctx.sign(
    ///         unrestricted_signing_key_handle,
    ///         digest,
    ///         SignatureScheme::Null,
    ///         None,
    ///     )
    /// })
    /// .expect("Failed to sign digest");
    /// ```
    pub fn sign(
        &mut self,
        key_handle: KeyHandle,
        digest: Digest,
        scheme: SignatureScheme,
        validation: impl Into<Option<HashcheckTicket>>,
    ) -> Result<Signature> {
        let mut signature_ptr = null_mut();
        let validation_ticket = validation.into().unwrap_or_default().try_into()?;
        ReturnCode::ensure_success(
            unsafe {
                Esys_Sign(
                    self.mut_context(),
                    key_handle.into(),
                    self.required_session_1()?,
                    self.optional_session_2(),
                    self.optional_session_3(),
                    &digest.into(),
                    &scheme.into(),
                    &validation_ticket,
                    &mut signature_ptr,
                )
            },
            |ret| {
                error!("Error when signing: {:#010X}", ret);
            },
        )?;
        Signature::try_from(Context::ffi_data_to_owned(signature_ptr))
    }
}
