/*
 * bsc1229275_fs_fscache_volume
 *
 * Fix for CVE-2024-41057, bsc#1229275
 *
 *  Copyright (c) 2024 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from fs/fscache/volume.c */
#include <linux/export.h>
#include <linux/slab.h>

#include <linux/slab.h>
#include <linux/fscache-cache.h>

/* klp-ccp: from include/linux/fscache-cache.h */
static struct rw_semaphore (*klpe_fscache_addremove_sem);

static void (*klpe_fscache_end_volume_access)(struct fscache_volume *volume,
				      struct fscache_cookie *cookie,
				      enum fscache_access_trace why);

/* klp-ccp: from fs/fscache/internal.h */
#include <trace/events/fscache.h>

#include "klp_trace.h"

/* klp-ccp: from include/trace/events/fscache.h */
KLPR_TRACE_EVENT(fscache_volume,
                 TP_PROTO(unsigned int volume_debug_id, int usage,
                          enum fscache_volume_trace where),
                 TP_ARGS(volume_debug_id, usage, where)
);

/* klp-ccp: from fs/fscache/internal.h */
#include <linux/sched.h>
#include <linux/seq_file.h>

static void (*klpe_fscache_put_cache)(struct fscache_cache *cache, enum fscache_cache_trace where);

static atomic_t (*klpe_fscache_n_volumes);

/* klp-ccp: from include/linux/compiler_types.h */
#define inline inline __gnu_inline __inline_maybe_unused notrace

/* klp-ccp: from fs/fscache/internal.h */
static inline void fscache_stat_d(atomic_t *stat)
{
	atomic_dec(stat);
}

/* klp-ccp: from fs/fscache/volume.c */
#define fscache_volume_hash_shift 10
static struct hlist_bl_head (*klpe_fscache_volume_hash)[1 << fscache_volume_hash_shift];

static void klpr_fscache_see_volume(struct fscache_volume *volume,
			       enum fscache_volume_trace where)
{
	int ref = refcount_read(&volume->ref);

	klpr_trace_fscache_volume(volume->debug_id, ref, where);
}

static void (*klpe___fscache_begin_volume_access)(struct fscache_volume *volume,
					  struct fscache_cookie *cookie,
					  enum fscache_access_trace why);

static bool fscache_volume_same(const struct fscache_volume *a,
				const struct fscache_volume *b)
{
	size_t klen;

	if (a->key_hash	!= b->key_hash ||
	    a->cache	!= b->cache ||
	    a->key[0]	!= b->key[0])
		return false;

	klen = round_up(a->key[0] + 1, sizeof(__le32));
	return memcmp(a->key, b->key, klen) == 0;
}

static void klpr_fscache_wake_pending_volume(struct fscache_volume *volume,
					struct hlist_bl_head *h)
{
	struct fscache_volume *cursor;
	struct hlist_bl_node *p;

	hlist_bl_for_each_entry(cursor, p, h, hash_link) {
		if (fscache_volume_same(cursor, volume)) {
			klpr_fscache_see_volume(cursor, fscache_volume_see_hash_wake);
			clear_and_wake_up_bit(FSCACHE_VOLUME_ACQUIRE_PENDING,
					      &cursor->flags);
			return;
		}
	}
}

static void klpr_fscache_unhash_volume(struct fscache_volume *volume)
{
	struct hlist_bl_head *h;
	unsigned int bucket;

	bucket = volume->key_hash & (ARRAY_SIZE((*klpe_fscache_volume_hash)) - 1);
	h = &(*klpe_fscache_volume_hash)[bucket];

	hlist_bl_lock(h);
	hlist_bl_del(&volume->hash_link);
	if (test_bit(FSCACHE_VOLUME_COLLIDED_WITH, &volume->flags))
		klpr_fscache_wake_pending_volume(volume, h);
	hlist_bl_unlock(h);
}

static void klpr_fscache_free_volume(struct fscache_volume *volume)
{
	struct fscache_cache *cache = volume->cache;

	if (volume->cache_priv) {
		(*klpe___fscache_begin_volume_access)(volume, NULL,
					      fscache_access_relinquish_volume);
		if (volume->cache_priv)
			cache->ops->free_volume(volume);
		(*klpe_fscache_end_volume_access)(volume, NULL,
					  fscache_access_relinquish_volume_end);
	}

	down_write(&(*klpe_fscache_addremove_sem));
	list_del_init(&volume->proc_link);
	atomic_dec(&volume->cache->n_volumes);
	up_write(&(*klpe_fscache_addremove_sem));

	if (!hlist_bl_unhashed(&volume->hash_link))
		klpr_fscache_unhash_volume(volume);

	klpr_trace_fscache_volume(volume->debug_id, 0, fscache_volume_free);
	kfree(volume->key);
	kfree(volume);
	fscache_stat_d(&(*klpe_fscache_n_volumes));
	(*klpe_fscache_put_cache)(cache, fscache_cache_put_volume);
}

void klpp_fscache_put_volume(struct fscache_volume *volume)
{
	if (volume) {
		bool zero;
		int ref;

		zero = __refcount_dec_and_test(&volume->ref, &ref);
		if (zero)
			klpr_fscache_free_volume(volume);
	}
}


#include "livepatch_bsc1229275.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "fscache"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "__fscache_begin_volume_access",
	  (void *)&klpe___fscache_begin_volume_access, "fscache" },
	{ "__tracepoint_fscache_volume",
	  (void *)&klpe___tracepoint_fscache_volume, "fscache" },
	{ "__traceiter_fscache_volume",
	  (void *)&klpe___traceiter_fscache_volume, "fscache" },
	{ "fscache_addremove_sem", (void *)&klpe_fscache_addremove_sem,
	  "fscache" },
	{ "fscache_end_volume_access", (void *)&klpe_fscache_end_volume_access,
	  "fscache" },
	{ "fscache_n_volumes", (void *)&klpe_fscache_n_volumes, "fscache" },
	{ "fscache_put_cache", (void *)&klpe_fscache_put_cache, "fscache" },
	{ "fscache_volume_hash", (void *)&klpe_fscache_volume_hash,
	  "fscache" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int bsc1229275_fs_fscache_volume_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void bsc1229275_fs_fscache_volume_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}
