/*
 * bsc1229275_fs_cachefiles_volume
 *
 * Fix for CVE-2024-41057, bsc#1229275
 *
 *  Copyright (c) 2024 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from fs/cachefiles/volume.c */
#include <linux/fs.h>

/* klp-ccp: from include/linux/fscache.h */
struct fscache_volume;

/* klp-ccp: from fs/cachefiles/internal.h */
#include <linux/cred.h>

#include <linux/xarray.h>

struct cachefiles_volume {
	struct cachefiles_cache		*cache;
	struct list_head		cache_link;	/* Link in cache->volumes */
	struct fscache_volume		*vcookie;	/* The netfs's representation */
	struct dentry			*dentry;	/* The volume dentry */
	struct dentry			*fanout[256];	/* Fanout subdirs */
};

static bool (*klpe_cachefiles_set_volume_xattr)(struct cachefiles_volume *volume);

/* klp-ccp: from fs/cachefiles/volume.c */
static void (*klpe___cachefiles_free_volume)(struct cachefiles_volume *volume);

void klpp_cachefiles_withdraw_volume(struct cachefiles_volume *volume)
{
	(*klpe_cachefiles_set_volume_xattr)(volume);
	(*klpe___cachefiles_free_volume)(volume);
}


#include "livepatch_bsc1229275.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "cachefiles"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "__cachefiles_free_volume", (void *)&klpe___cachefiles_free_volume,
	  "cachefiles" },
	{ "cachefiles_set_volume_xattr",
	  (void *)&klpe_cachefiles_set_volume_xattr, "cachefiles" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int bsc1229275_fs_cachefiles_volume_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void bsc1229275_fs_cachefiles_volume_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}
