/*
 * bsc1225429_net_ethtool_stats
 *
 * Fix for CVE-2021-47517, bsc#1225429
 *
 *  Copyright (c) 2024 SUSE
 *  Author: Fernando Gonzalez <fernando.gonzalez@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/ethtool/netlink.h */
#include <linux/ethtool_netlink.h>
#include <linux/netdevice.h>

/* klp-ccp: from include/net/genetlink.h */
struct genl_info;

/* klp-ccp: from net/ethtool/netlink.h */
struct ethnl_req_info {
	struct net_device	*dev;
	u32			flags;
};

struct ethnl_reply_data {
	struct net_device		*dev;
};

void ethnl_ops_complete(struct net_device *dev);
int ethnl_ops_begin(struct net_device *dev);

/* klp-ccp: from net/ethtool/common.h */
#include <linux/ethtool.h>

/* klp-ccp: from net/ethtool/stats.c */
struct stats_req_info {
	struct ethnl_req_info		base;
	DECLARE_BITMAP(stat_mask, __ETHTOOL_STATS_CNT);
};

#define STATS_REQINFO(__req_base) \
	container_of(__req_base, struct stats_req_info, base)

struct stats_reply_data {
	struct ethnl_reply_data		base;
	struct ethtool_eth_phy_stats	phy_stats;
	struct ethtool_eth_mac_stats	mac_stats;
	struct ethtool_eth_ctrl_stats	ctrl_stats;
	struct ethtool_rmon_stats	rmon_stats;
	const struct ethtool_rmon_hist_range	*rmon_ranges;
};

#define STATS_REPDATA(__reply_base) \
	container_of(__reply_base, struct stats_reply_data, base)

int klpp_stats_prepare_data(const struct ethnl_req_info *req_base,
			      struct ethnl_reply_data *reply_base,
			      struct genl_info *info)
{
	const struct stats_req_info *req_info = STATS_REQINFO(req_base);
	struct stats_reply_data *data = STATS_REPDATA(reply_base);
	struct net_device *dev = reply_base->dev;
	int ret;

	ret = ethnl_ops_begin(dev);
	if (ret < 0)
		return ret;

	/* Mark all stats as unset (see ETHTOOL_STAT_NOT_SET) to prevent them
	 * from being reported to user space in case driver did not set them.
	 */
	memset(&data->phy_stats, 0xff, sizeof(data->phy_stats));
	memset(&data->mac_stats, 0xff, sizeof(data->mac_stats));
	memset(&data->ctrl_stats, 0xff, sizeof(data->ctrl_stats));
	memset(&data->rmon_stats, 0xff, sizeof(data->rmon_stats));

	if (test_bit(ETHTOOL_STATS_ETH_PHY, req_info->stat_mask) &&
	    dev->ethtool_ops->get_eth_phy_stats)
		dev->ethtool_ops->get_eth_phy_stats(dev, &data->phy_stats);
	if (test_bit(ETHTOOL_STATS_ETH_MAC, req_info->stat_mask) &&
	    dev->ethtool_ops->get_eth_mac_stats)
		dev->ethtool_ops->get_eth_mac_stats(dev, &data->mac_stats);
	if (test_bit(ETHTOOL_STATS_ETH_CTRL, req_info->stat_mask) &&
	    dev->ethtool_ops->get_eth_ctrl_stats)
		dev->ethtool_ops->get_eth_ctrl_stats(dev, &data->ctrl_stats);
	if (test_bit(ETHTOOL_STATS_RMON, req_info->stat_mask) &&
	    dev->ethtool_ops->get_rmon_stats)
		dev->ethtool_ops->get_rmon_stats(dev, &data->rmon_stats,
						 &data->rmon_ranges);

	ethnl_ops_complete(dev);
	return 0;
}
